# -*- coding: utf-8 -*-

import json

from django.http import HttpResponseNotAllowed
from django.core.exceptions import ValidationError
from django.core.urlresolvers import reverse
from django.utils.translation import ugettext
from shapely.geometry import Point, MultiPoint, LineString

from common.models.geo import CityMajority, Country, Region, Station, Settlement
from common.models.transport import TransportType
from common.utils.geo import center_span_zoom, clip
from common.utils.httpresponses import jsonp_response


PURPLE_T_TYPES = (
    TransportType.TRAIN_ID,
    TransportType.PLANE_ID,
    TransportType.BUS_ID,
    TransportType.SUBURBAN_ID,
) + tuple(
    TransportType.WATER_TTYPE_IDS
)


def json_www_station(s):
    t_type = TransportType.objects.get(id=s.t_type_id)

    return {
        'id': s.id,
        'title': s.title,
        'lon': s.longitude,
        'lat': s.latitude,
        'settlement': s.settlement_id and {'id': s.settlement_id},
        'geoadmin_link': reverse('admin:www_station_change', args=(s.id,)),
        't_type': t_type.code,
        't_type_id': s.t_type_id,
        'majority_id': s.majority_id,
        'station_type_id': s.station_type_id,
        'hidden': s.hidden
    }


def polygon_coords(points):
    if not points:
        return []

    ln = len(points)

    if ln == 1:
        polygon = Point(points[0])
    elif ln == 2:
        polygon = LineString(points)
    else:
        polygon = MultiPoint(points).convex_hull

    return list(polygon.buffer(0.01).exterior.coords)


def settlement_json(settlement):
    return {
        'id': settlement.id,
        'title': settlement.title,
        'title_ru': settlement.title_ru,
        'title_uk': settlement.title_uk,
        'title_tr': settlement.title_tr,
        'title_en': settlement.title_en,
        'country': {
            'id': settlement.country.id,
            'title': settlement.country.title,
        } if settlement.country else None,
        'region': {
            'id': settlement.region.id,
            'title': settlement.region.title,
        } if settlement.region else None,
        'majority_id': settlement.majority_id,
        'lat': settlement.latitude,
        'lon': settlement.longitude,
        'exterior': polygon_coords(getattr(settlement, 'points', [])),
        'stations': [json_www_station(s) for s in getattr(settlement, 'stations', [])],
    }


@jsonp_response
def settlement_polygons(request):
    center, span, zoom = center_span_zoom(request)

    if center is None or span is None or zoom is None:
        return None

    if zoom < 9:
        return {
            'error': ugettext(u'Слишком маленький масштаб')
        }

    error = None

    # Сначала ищем все города которые попадают во viewport
    stations = Station.objects.filter(
        t_type__in=PURPLE_T_TYPES,
        settlement__isnull=False
    )

    stations = clip(
        stations,
        center, span,
        zoom_radius=1.1,
        lon_field='longitude',
        lat_field='latitude'
    )

    settlement_ids = list(stations.values_list('settlement_id', flat=True).distinct())

    # Выкидываем Москву и Санкт-Петербург, в них слишком много станций
    if Settlement.MOSCOW_ID in settlement_ids:
        settlement_ids.remove(Settlement.MOSCOW_ID)
    if Settlement.SPB_ID in settlement_ids:
        settlement_ids.remove(Settlement.SPB_ID)

    # Города вытаскиваем отдельными запросами, т.к. они лежат в памяти
    settlements = Settlement.objects.in_bulk(settlement_ids)

    # Геометрии и браузеру нужны точки не только из вьюпорта, а все видимых городов
    stations = Station.objects.filter(
        settlement_id__in=settlement_ids,
        t_type__in=PURPLE_T_TYPES,
        longitude__isnull=False,
        latitude__isnull=False
    )

    # Браузер треснет столько рисовать
    if stations.count() > 1000:
        error = ugettext(u'Слишком много станций')
        settlements = {}
        stations = []

    for station in stations:
        settlement = settlements[station.settlement_id]

        if not hasattr(settlement, 'points'):
            settlement.points = []

        settlement.points.append((station.longitude, station.latitude))

        if not hasattr(settlement, 'stations'):
            settlement.stations = []

        settlement.stations.append(station)

    return {
        'settlements': [settlement_json(s) for s in settlements.values()],
        'error': error,
    }


@jsonp_response
def stations_no_city(request):
    center, span, zoom = center_span_zoom(request)

    if center is None or span is None or zoom is None:
        return None

    if zoom < 9:
        return {
            'error': ugettext(u'Слишком маленький масштаб')
        }

    error = None

    # Станции без города во viewport
    stations = Station.objects.filter(
        t_type__in=PURPLE_T_TYPES,
        settlement__isnull=True
    )

    stations = clip(
        stations,
        center, span,
        zoom_radius=1.1,
        lon_field='longitude',
        lat_field='latitude'
    )

    # Браузер треснет столько рисовать
    if stations.count() > 1000:
        error = ugettext(u'Слишком много станций без города')
        stations = []

    return {
        'stations': [json_www_station(s) for s in stations],
        'error': error,
    }


@jsonp_response
def regions(request, id):
    try:
        country = Country.objects.get(id=int(id))
    except:
        return []

    return [{
        'id': r.id,
        'title': r.title
    } for r in Region.objects.filter(country=country)]


@jsonp_response
def create_settlement(request):
    if request.method != 'POST':
        return HttpResponseNotAllowed(('POST',))

    data = json.loads(request.body)

    settlement = Settlement(**{param: data.get(param) for param in (
        'title', 'title_ru', 'title_uk', 'title_tr', 'title_en', 'time_zone'
    )})

    if data.get('country_id') is not None:
        settlement.country = Country.objects.get(id=data['country_id'])

    if data.get('region_id') is not None:
        settlement.region = Region.objects.get(id=data['region_id'])

    if data.get('majority_id') is not None:
        settlement.majority = CityMajority.objects.get(id=data['majority_id'])

    stations = Station.objects.filter(id__in=data['station_ids'])

    if len(stations) == 1:
        settlement.longitude = stations[0].longitude
        settlement.latitude = stations[0].latitude
    elif stations:
        settlement_point = MultiPoint([
            (station.longitude, station.latitude)
            for station in stations
        ]).centroid
        settlement.longitude = settlement_point.x
        settlement.latitude = settlement_point.y

    try:
        settlement.full_clean()
    except ValidationError as e:
        return {
            'error': u'Ошибки валидации города ({})'.format('; '.join(
                u'{}: {}'.format(field, ', '.join(messages))
                for field, messages in e.message_dict.items()
            ))
        }

    settlement.save()
    stations.update(settlement=settlement)


@jsonp_response
def settlements(request, id):
    try:
        region = Region.objects.get(id=int(id))
    except:
        return []

    return [{
        'id': s.id,
        'title': s.title,
        'lat': s.latitude,
        'lon': s.longitude,
    } for s in Settlement.objects.filter(region=region)]


@jsonp_response
def station_details(request, id):
    station = Station.objects.get(id=id)

    if request.method == 'PUT':
        data = json.loads(request.body)

        update_params = (
            'title', 'title_ru', 'title_uk', 'title_tr', 'title_en',
            'hidden', 'country', 'region', 'settlement',
            'majority_id', 'station_type_id'
        )

        for param in update_params:
            if param not in data:
                continue

            val = data.get(param)

            if param == 'country' and val:
                val = val.get('id')
                val = Country.objects.get(id=val)

            if param == 'region' and val:
                val = val.get('id')
                val = Region.objects.get(id=val)

            if param == 'settlement' and val:
                val = val.get('id')
                val = Settlement.objects.get(id=val)

            setattr(station, param, val)

        station.save()

        station = Station.objects.get(id=id)

    return {
        'id': station.id,
        'title_ru': station.title_ru,
        'title_uk': station.title_uk,
        'title_tr': station.title_tr,
        'title_en': station.title_en,
        'country': {
            'id': station.country.id,
            'title': station.country.title
        } if station.country else None,
        'region': {
            'id': station.region.id,
            'title': station.region.title
        } if station.region else None,
        'settlement': {
            'id': station.settlement.id,
            'title': station.settlement.title
        } if station.settlement else None
    }


@jsonp_response
def settlement_actions(request, id):
    settlement = Settlement.objects.get(id=id)

    if request.method == 'PUT':
        data = json.loads(request.body)

        update_params = ('title', 'title_ru', 'title_uk', 'title_tr', 'title_en',
                         'country', 'region', 'majority_id')

        for param in update_params:
            val = data.get(param)

            if param == 'country' and val:
                val = val.get('id')
                val = Country.objects.get(id=val)

            if param == 'region' and val:
                val = val.get('id')
                val = Region.objects.get(id=val)

            setattr(settlement, param, val)

        settlement.save()

        settlement = Settlement.objects.get(id=id)

    return {
        'id': settlement.id,
        'title': settlement.title,
        'title_ru': settlement.title_ru,
        'title_uk': settlement.title_uk,
        'title_tr': settlement.title_tr,
        'title_en': settlement.title_en,
        'country': {
            'id': settlement.country.id,
            'title': settlement.country.title,
        } if settlement.country else None,
        'region': {
            'id': settlement.region.id,
            'title': settlement.region.title,
        } if settlement.region else None,
        'majority_id': settlement.majority_id,
        'lat': settlement.latitude,
        'lon': settlement.longitude,
    }


@jsonp_response
def settlement_suggest(request):
    query = request.GET.get('part')

    if query:
        query = query.strip()

    if not query or len(query) < 3:
        return []

    settlements = Settlement.objects.filter(
        title__istartswith=query
    ).order_by('majority', 'threads_amount', 'title')[:10]

    return [
        query,
        [['settlement', unicode(i), {'js': {'id': i.id}}] for i in settlements]
    ]
