# -*- coding: utf-8 -*-

from __future__ import absolute_import

import json

from django.core.exceptions import ValidationError
from django.db import transaction
from django.utils.translation import ugettext_lazy as _, ugettext

from common.models.geo import Station
from common.utils.httpresponses import jsonp_response

from travel.rasp.admin.admin.red.models import Package, MetaRouteStation, MetaRoute
from travel.rasp.admin.admin.red.metaimport import PackageAction, MetaRouteAction
from travel.rasp.admin.admin.red.convertors import package_to_json, metaroute_to_json, station_to_json
from travel.rasp.admin.admin.red.views import redadmin_permission_required_or_403
from travel.rasp.admin.admin.red.cartesian import distance


# Views
METAROUTE_KEYS = ['package_id', 'number', 'title', 'apply_base_stations', 'scheme',
                  'comment', 't_type_id', 'supplier_id', 't_subtype_id']


@redadmin_permission_required_or_403
@jsonp_response
def packages(request):
    packages = Package.objects.select_related('country').all()
    return [package_to_json(p) for p in packages]


@redadmin_permission_required_or_403
@jsonp_response
def package_import(request, id):
    try:
        package = Package.objects.get(pk=id)
    except MetaRoute.DoesNotExist:
        return {'status': 'error', 'message': ugettext(u'Нет такого пакета')}

    action = PackageAction(package, 'import_package')

    if action.do():
        return {'status': 'ok', 'url': action.log_view_url}
    else:
        return {'status': 'error', 'message': ugettext(u'Не смогли импортировать %s') % (action.error or u'')}


@redadmin_permission_required_or_403
@jsonp_response
def threads(request):
    if request.method == 'POST':
        data = json.loads(request.body)
        data['t_subtype_id'] = data['t_subtype_id'] or None

        metaroute = MetaRoute()
        for key in METAROUTE_KEYS:
            setattr(metaroute, key, data.get(key))
        metaroute.save()
        return metaroute_to_json(metaroute)

    package_id = request.GET.get('package_id')
    metaroutes = MetaRoute.objects.filter(package_id=package_id).prefetch_related('route_set')

    return [metaroute_to_json(p) for p in metaroutes]


def metaroute_atomic_clone(thread, reverse):
    # TODO: в django 1.8 нужно пометить atomic
    scheme = not reverse and thread.scheme or ''
    comment = not reverse and thread.comment or ''

    clone = MetaRoute.objects.create(
        package=thread.package,
        number=thread.number,
        title=thread.title,
        apply_base_stations=thread.apply_base_stations,
        scheme=scheme,
        comment=comment,
        t_type=thread.t_type,
        supplier=thread.supplier,
    )

    mrstations = list(thread.path)

    if mrstations:
        final = mrstations[-1].arrival

        if reverse:
            mrstations.reverse()

        for i, mrs in enumerate(mrstations):
            arrival = mrs.arrival
            departure = mrs.departure

            if reverse:
                arrival = final - mrs.departure if mrs.departure is not None else None
                departure = final - mrs.arrival if mrs.arrival is not None else None

            clone.metaroutestation_set.create(station=mrs.station,
                                              arrival=arrival, departure=departure,
                                              is_fuzzy=mrs.is_fuzzy,
                                              order=i)

    return metaroute_to_json(clone)


@redadmin_permission_required_or_403
@jsonp_response
@transaction.atomic
def thread(request, id):
    try:
        metaroute = MetaRoute.objects.get(pk=id)
    except MetaRoute.DoesNotExist:
        return {'status': 'error', 'message': ugettext(u'Нет такого мета-рейса')}

    validation_status = None

    if request.method == 'POST':
        action = request.POST.get('action')
        if action == 'clone':
            reverse = request.POST.get('reverse', False)
            if reverse and reverse == 'false':
                reverse = False
            else:
                reverse = True

            try:
                return metaroute_atomic_clone(metaroute, reverse)
            except Exception as e:
                return {'status': 'error', 'message': u'Ошибка %s' % e}

        else:
            return {'status': 'error', 'message': u'Некорректное действие'}

    elif request.method == 'PUT':
        data = json.loads(request.body)
        data['t_subtype_id'] = data['t_subtype_id'] or None
        for key in METAROUTE_KEYS:
            setattr(metaroute, key, data.get(key))

        error_messages = []
        try:
            metaroute.full_clean()
        except ValidationError as e:
            if hasattr(e, 'message_dict'):
                for field, messages in e.message_dict.items():
                    for message in messages:
                        error_messages.append(u'{}: {}'.format(field, message))
            else:
                error_messages.extend(e.messages)
        else:
            metaroute.save()

        error_messages.extend(metaroute.check_mrstations())

        # Проверка валидности времен у станций нитки
        validation_status = u'<br />'.join(error_messages)

    elif request.method == 'DELETE':
        metaroute.delete()
        return {'status': 'ok'}

    json_response = metaroute_to_json(metaroute)
    # подмешиваем в объект нитки сообщение о валидации
    if validation_status:
        json_response['validationStatus'] = validation_status

    return json_response


@redadmin_permission_required_or_403
@jsonp_response
def thread_import(request, id):
    try:
        metaroute = MetaRoute.objects.get(pk=id)
    except MetaRoute.DoesNotExist:
        return {'status': 'error', 'message': ugettext(u'Нет такого мета-рейса')}

    # На всякий случай поставим верные времена для первой и последней станции
    count = metaroute.path.count()
    if count > 1:
        first_station = metaroute.path[0]
        last_station = metaroute.path[count - 1]
    elif count == 1:
        first_station = last_station = metaroute.path[0]
    else:
        return {'status': 'error', 'message': ugettext(u'Нельзя импортировать рейс без станций')}

    first_station.arrival = None
    first_station.departure = 0
    last_station.departure = None
    first_station.save()
    last_station.save()

    action = MetaRouteAction(metaroute, 'import_metaroute')

    if action.do():
        return {'status': 'ok', 'url': action.log_view_url}
    else:
        return {'status': 'error',
                'message': ugettext(u'Не смогли импортировать %s') % (action.error or u'')}


@redadmin_permission_required_or_403
@jsonp_response
def stations_reorder(request):
    try:
        metaroute = MetaRoute.objects.get(pk=request.POST.get('thread_id'))
    except (MetaRoute.DoesNotExist, ValueError, TypeError):
        return {
            'status': 'error',
            'message': _(u'Метарейс не найден')
        }

    order_list = request.POST.getlist('order[]')

    stations = list(metaroute.path)

    if len(order_list) != len(stations):
        return {
            'status': 'error',
            'message': _(u'Количество станций не совпадает')
        }

    stations_dict = {s.id: s for s in stations}

    start_order = max(stations, key=lambda x: x.order).order + 1

    for i, station_id in enumerate(order_list, start=start_order):
        station = stations_dict.get(int(station_id))
        station.order = i
        station.save()

    stations = stations_dict.values()
    stations.sort(key=lambda x: x.order)

    return [station_to_json(s) for s in stations]


@redadmin_permission_required_or_403
@jsonp_response
def stations(request):
    if request.method == 'POST':
        data = json.loads(request.body)
        try:
            www_station_id = data.get('www_station').get('id')
            metaroute_id = data.get('metaroute_id')
            metaroute = MetaRoute.objects.get(pk=metaroute_id)
            www_station = Station.objects.exclude(t_type__code='pseudo-gortrans').\
                get(pk=www_station_id)
            station = MetaRouteStation(metaroute=metaroute, station=www_station)

            before = data.get('before')
            after = data.get('after')
            if before is not None:
                obj = MetaRouteStation.objects.get(pk=int(before))
                station.insert('before', obj)
            elif after is not None:
                obj = MetaRouteStation.objects.get(pk=int(after))
                station.insert('after', obj)
            else:
                station.save()

            return station_to_json(station)
        except:
            return {'status': 'error', 'message': u'Не удалось привязать станцию'}

    thread_id = int(request.GET.get('thread_id'))
    stations = MetaRouteStation.objects.filter(metaroute_id=thread_id).select_related('station')

    return [station_to_json(s) for s in stations]


@redadmin_permission_required_or_403
@jsonp_response
def station(request, id):
    try:
        station = MetaRouteStation.objects.get(pk=id)
    except MetaRoute.DoesNotExist:
        return {'status': 'error', 'message': ugettext(u'Нет такой станции')}

    if request.method == 'PUT':
        fuzzy_flags = ['is_fuzzy', 'is_searchable_from', 'is_searchable_to',
                       'in_station_schedule']
        fuzzy_mapping = {'yes': True, 'no': False, 'inherit': None}

        data = json.loads(request.body)
        try:
            station.arrival = int(data['arrival'])
        except:
            station.arrival = None
        try:
            station.departure = int(data['departure'])
        except:
            station.departure = None
        for key in fuzzy_flags:
            setattr(station, key, fuzzy_mapping[data.get(key, 'inherit')])
        station.save()

    elif request.method == 'DELETE':
        station.delete()
        return {'status': 'ok'}

    return station_to_json(station)


@redadmin_permission_required_or_403
@jsonp_response
def stations_distances(request):
    try:
        metaroute = MetaRoute.objects.get(pk=request.GET.get('thread_id'))
    except (MetaRoute.DoesNotExist, ValueError, TypeError):
        return {
            'status': 'error',
            'message': _(u'Метарейс не найден')
        }

    stations = list(metaroute.path)
    distances = []

    i = 0
    while i < len(stations) - 1:
        s = stations[i]
        next_s = stations[i + 1]

        a = (s.station.latitude, s.station.longitude)
        b = (next_s.station.latitude, next_s.station.longitude)

        distances.append({
            'from': s.station.id,
            'to': next_s.station.id,
            'value': distance(a, b),
        })

        i += 1

    return distances
