# -*- coding: utf-8 -*-

import os
import logging
import codecs
from collections import defaultdict

from django.db import transaction, connection
from django.http import HttpResponseRedirect
from django.contrib.contenttypes.models import ContentType
from django.conf import settings

from common.models.schedule import RThread, RTStation, Route, RThreadType
from travel.rasp.library.python.common23.date import environment
from cysix.filters.apply_base_stations import apply_base_stations
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin
from travel.rasp.admin.lib.logs import get_rasp_log_path, ContextRaspFormatter
from travel.rasp.admin.lib.maintenance.flags import flags, job
from travel.rasp.admin.lib.processes import child_finalizer
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError
from travel.rasp.admin.scripts.support_methods import full_create_route
from travel.rasp.admin.www.utils.mysql import fast_delete_routes

from travel.rasp.admin.admin.red.models import Package
from travel.rasp.admin.admin.red import scheme


log = logging.getLogger(__name__)


class PackageAdmin(RaspExportModelAdmin):
    raw_id_fields = ('country', 'region')
    list_display = ('title', 'content_manager', 'last_import_date', 'last_mask_date', 'autoimport')
    readonly_fields = ('last_import_date', 'last_mask_date')

    fieldsets = (
        ('Profile field', {
            'fields': (
                'content_manager',
                'title',
                'region',
                'country',
                't_type',
                't_subtype',
                'last_import_date',
                'last_mask_date',
                'autoimport',
            )
        }),
    )

    def change_view(self, request, object_id, **kwargs):
        package = Package.objects.get(pk=object_id)
        actions = self.get_change_view_actions(package)
        extra_context = kwargs.pop('extra_context', {})
        extra_context['actions'] = actions
        action = self.get_action_by_name(request.POST.get('__action'), actions)

        if request.method == 'POST' and action:
            request.method = 'GET'  # admin stub

            if action.do():
                return HttpResponseRedirect(request.path + ('?_show_log=%s' % action.name))

        return super(PackageAdmin, self).change_view(request, object_id,
                                                     extra_context=extra_context, **kwargs)

    def get_change_view_actions(self, package):
        return (
            PackageAction(package, 'test_import_package', u'Тестовый импорт', u'Лог тестового импорта'),
            PackageAction(package, 'import_package', u'Импорт', u'Лог импорта'),
            PackageAction(package, 'clean_package', u'Удалить все созданные рейсы', u'Лог очистки'),
        )

    def get_action_by_name(self, name, actions):
        for action in actions:
            if action.name == name:
                return action


class Action(object):
    def __init__(self, name, button_title='', log_name=''):
        self.name = name
        self.button_title = button_title
        self.log_name = log_name
        self.error = None
        self.log_view_url = None
        self.log = ''

    def __eq__(self, other):
        self.name = other.name

    def finalize(self):
        pass

    def do(self):

        if flags['maintenance']:
            self.error = u'Нельзя работать с пакетом пока поднят флаг'
            return False

        self.related_log.add_handler()

        try:
            old_flag = flags['maintenance']
            flags['maintenance'] = job.RED_META_IMPORT.flag_value
            log.info(u'Установили флаг')

        except:
            log.exception(u'Не можем установить флаг')

            return

        pid = os.fork()

        if pid:
            log.info(u'Импорт запущен')

            return True

        else:
            with child_finalizer():
                connection.close()

                log.info(u'Продолжается импорт в фоновом процессе')
                try:
                    self.action_function()

                except:
                    log.exception(u'Ошибка при выполнении команды')

                try:
                    flags['maintenance'] = old_flag

                except:
                    log.exception(u'Не можем восстановить флаг')

                self.finalize()


class PackageAction(Action):
    def __init__(self, package, *args, **kwargs):
        super(PackageAction, self).__init__(*args, **kwargs)
        self.package = package
        self.related_log = RelatedLog('admin.red', self.package, self.name)
        self.log_view_url = self.related_log.log_view_url
        self.log = self.related_log.read()

    def action_function(self):
        action_function = {
            'test_import_package': fake_import_package,
            'import_package': import_package,
            'clean_package': clean_package,
        }.get(self.name, lambda *dummy_args, **dummy_kwargs: None)

        importer = RedPackageImporter(self.package)

        return action_function(importer)

    def finalize(self):
        self.related_log.remove_handler()


@transaction.atomic
def fake_import_package(importer):
    try:
        importer.import_package()

    except Exception, e:
        log.exception(u'Ошибка при тестовом импорте пакета %s' % e.message)

    finally:
        log.info(u'Откатываем изменения тестового импорта')
        transaction.set_rollback(True)


@transaction.atomic
def import_package(importer):
    try:
        importer.import_package()

    except Exception, e:
        log.exception(u'Ошибка при импорте пакета %s' % e.message)
        transaction.set_rollback(True)


@transaction.atomic
def clean_package(importer):
    try:
        importer.clean_package()

    except Exception, e:
        log.exception(u'Ошибка при очитске пакета %s' % e.message)
        transaction.set_rollback(True)


class MetaRouteAction(Action):
    def __init__(self, metaroute, *args, **kwargs):
        super(MetaRouteAction, self).__init__(*args, **kwargs)
        self.metaroute = metaroute
        self.related_log = RelatedLog('admin.red', self.metaroute, self.name)
        self.log_view_url = self.related_log.log_view_url
        self.log = self.related_log.read()

    def action_function(self):
        action_function = {
            'import_metaroute': import_metaroute,
            'clean_metaroute': clean_metaroute,
        }.get(self.name, lambda *dummy_args, **dummy_kwargs: None)

        importer = RedMetaRouteImporter(self.metaroute)

        return action_function(importer)

    def finalize(self):
        self.related_log.remove_handler()


@transaction.atomic
def import_metaroute(importer):
    try:
        importer.import_metaroute()

    except RaspImportError as e:
        log.error(u"Ошибка при импорте метарейса %s: %s", importer.metaroute, unicode(e))
        transaction.set_rollback(True)

    except Exception, e:
        log.exception(u'Ошибка при импорте метарейса %s' % e.message)
        transaction.set_rollback(True)


@transaction.atomic
def clean_metaroute(importer):
    try:
        importer.clean_metaroute()

    except Exception, e:
        log.exception(u'Ошибка при очитске метарейса %s' % e.message)
        transaction.set_rollback(True)


class RedPackageImporter(object):
    def __init__(self, package):
        self.package = package

    def import_package(self):
        log.info(u'Импортируем пакет %s' % self.package)

        for metaroute in self.package.metaroute_set.all():
            log.info(u'Импорт метарейса %s' % metaroute)

            try:
                RedMetaRouteImporter(metaroute).import_metaroute()

            except RaspImportError as e:
                log.error(u"Ошибка при импорте метарейса %s: %s", metaroute, unicode(e))

            except Exception as e:
                log.exception(u'Ошибка при импорте метарейса %s' % metaroute)

        self.package.last_import_date = environment.today()
        self.package.last_import_datetime = environment.now()
        self.package.update_last_mask_date()

        self.package.save()

        log.info(u'Импорт пакета %s завершен' % self.package)

    def clean_package(self):
        for metaroute in self.package.metaroute_set.all():
            log.info(u'Очистка метарейса %s' % metaroute)

            try:
                RedMetaRouteImporter(metaroute).clean_metaroute()

            except Exception, e:
                log.error(u'Ошибка при очистке метарейса %s %s' % (metaroute, e))

        log.info(u'Очистка пакета %s завершена' % self.package)


class TimeTableParserError(RaspImportError):
    pass


class RedMetaRouteImporter(object):
    def __init__(self, metaroute):
        self.metaroute = metaroute
        self.scheme = []
        self.route = None

    def get_log_view_url(self):
        return ''

    def import_metaroute(self):
        check_errors = self.metaroute.check_mrstations()

        if check_errors:
            raise RaspImportError(u'\n'.join(check_errors))

        try:
            self.parse_scheme()

        except scheme.ParseError, e:
            log.error(u'Ошибка разбора схемы метарейса %s. %s' % (self.metaroute, unicode(e)))

            return

        if len(self.scheme) == 0:
            log.warning(u'Схема пуста. Пропускаем рейс')

            return

        self.clean_metaroute()

        self.create_routes()

    def parse_scheme(self):
        self.scheme = self.metaroute.get_scheme()

    def clean_metaroute(self):
        log.info(u'Удаляем маршруты метарейса %s' % self.metaroute)

        fast_delete_routes(Route.objects.filter(script_protected=False, red_metaroute=self.metaroute))

    def create_routes(self):
        today = environment.today()
        threads = self.create_threads()

        routes = {}
        for thread in threads:
            route_uid = thread.gen_route_uid(use_company=True, use_stations=True, red_metaroute_id=self.metaroute.id)
            route = routes.get(route_uid)
            if not route:
                route = Route(
                    t_type=self.metaroute.t_type,
                    supplier=self.metaroute.supplier,
                    route_uid=route_uid,
                    red_metaroute=self.metaroute
                )
                route.threads_by_import_uids = {}
                routes[route_uid] = route

            thread.route = route
            thread.gen_import_uid()

            if thread.import_uid in route.threads_by_import_uids:
                base_thread = route.threads_by_import_uids[thread.import_uid]
                base_thread.year_days = str(base_thread.get_mask(today) | thread.get_mask(today))
            else:
                route.threads_by_import_uids[thread.import_uid] = thread

        for route in routes.values():
            route.threads = route.threads_by_import_uids.values()
            full_create_route(route, log)

    def create_threads(self):
        threads = []

        self.mrstations = self.metaroute.path
        self.stations = [mrs.station for mrs in self.mrstations]
        self.start_station = self.stations and self.stations[0] or None

        if self.start_station is None:
            raise RaspImportError(u'Не нашли начальной станции')

        self.time_zone = self.start_station.time_zone
        if not self.time_zone:
            raise RaspImportError(u'У первой станции маршрута не указана временная зона')

        comment = self.metaroute.comment

        # Интервальная нитка если комментарий не пуст и содержит символ '|'
        is_interval = comment and '|' in comment
        if is_interval:
            density, comment = ([''] + comment.split('|')[:2])[-2:]

            interval_groups = defaultdict(list)
            for local_start_time, full_local_mask, can_extrapolate in self.scheme:
                interval_groups[full_local_mask, can_extrapolate].append(local_start_time)

            for (full_local_mask, can_extrapolate), interval_times in interval_groups.items():
                interval_times.sort()
                begin_time = interval_times[0]
                end_time = interval_times[-1]

                # Во время отправления нитки записываем конец интервала
                local_start_time = end_time

                thread = self.add_thread(local_start_time, full_local_mask, begin_time, end_time,
                                         comment, density, is_interval=True, can_extrapolate=can_extrapolate)
                if thread:
                    threads.append(thread)

        else:
            for local_start_time, full_local_mask, can_extrapolate in self.scheme:
                thread = self.add_thread(local_start_time, full_local_mask, comment=comment,
                                         can_extrapolate=can_extrapolate)
                if thread:
                    threads.append(thread)

        return threads

    def add_thread(self, local_start_time, local_mask, begin_time=None, end_time=None,
                   comment='', density='', is_interval=False, can_extrapolate=True):

        if not local_mask:
            log.error(u'Пустая маска у нитки, пропускаем нитку')

            return

        thread = RThread(
            type_id=RThreadType.INTERVAL_ID if is_interval else RThreadType.BASIC_ID,
            t_type=self.metaroute.t_type,
            t_subtype=self.metaroute.t_subtype or self.metaroute.package.t_subtype,
            supplier=self.metaroute.supplier,
            number=self.metaroute.number,
            time_zone=self.time_zone,
            tz_start_time=local_start_time,
            begin_time=begin_time,
            end_time=end_time,
            density=density,
            comment=comment,
            changed=True,
            has_extrapolatable_mask=can_extrapolate
        )

        try:
            self.fill_thread(thread, local_mask)

        except RaspImportError, e:
            log.error(u'%s, пропускаем нитку', unicode(e))

            return

        thread.gen_title(strategy='mta')

        return thread

    def fill_thread(self, thread, local_mask):
        rtstations = []

        thread.mask = local_mask
        thread.year_days = str(thread.mask)

        prev_rts = None
        for index, mrstation in enumerate(self.mrstations):
            is_first = (index == 0)

            rts = RTStation()
            rts.time_zone = self.time_zone

            rts.station = mrstation.station

            rts.is_searchable_to = mrstation.actual_is_searchable_to
            rts.is_searchable_from = mrstation.actual_is_searchable_from
            rts.in_station_schedule = mrstation.actual_in_station_schedule

            if is_first:
                rts.is_fuzzy = False
                rts.tz_arrival = None
                rts.tz_departure = 0
            else:
                rts.is_fuzzy = mrstation.actual_is_fuzzy
                rts.tz_departure = mrstation.departure
                rts.tz_arrival = mrstation.arrival

                if rts.tz_arrival < prev_rts.tz_departure:
                    raise RaspImportError(u"Время отправления с {} меньше, чем прибытие на {}".format(
                        prev_rts.station.title, rts.station.title
                    ))

            rtstations.append(rts)

            prev_rts = rts

        if len(rtstations) < 2:
            raise TimeTableParserError(u'В нитке меньше двух станций')

        rtstations = self.apply_base_stations(rtstations)
        thread.rtstations = rtstations

    def apply_base_stations(self, rtstations):
        if not self.metaroute.apply_base_stations:
            return rtstations

        return apply_base_stations(rtstations)


class RelatedLog:
    """
    Не доработан, и оформлять его как лог не нужно.
    В идеале нужна класс база с логами с историей.
    TODO: переделать
    """
    def __init__(self, log_or_logname, obj, name):
        self.logger = logging.getLogger(log_or_logname)
        self.obj = obj
        self.name = name
        content_type = ContentType.objects.get_for_model(obj)
        self.log_name = '%s_%s_%s_%s' % (content_type.app_label, content_type.model, self.name, self.obj.id)
        self.log_view_url = '/admin/maintenance/show_log/%s.log' % self.log_name
        self.baseFilename = get_rasp_log_path('special.admin_run.%s' % self.log_name)

    def add_handler(self, format_=settings.LOG_FORMAT):
        for handler in self.logger.handlers:
            if isinstance(handler, logging.FileHandler) and handler.baseFilename == self.baseFilename:
                self.handler = handler

        try:
            self.handler = logging.FileHandler(self.baseFilename, 'w')
            self.handler.setLevel(logging.INFO)
            self.handler.setFormatter(ContextRaspFormatter(format_))
            self.logger.addHandler(self.handler)
        except:
            log.exception(u'Ошибка создания обработчика лога')

    def remove_handler(self):
        if self.handler:
            self.logger.removeHandler(self.handler)

    def read(self):
        if os.path.exists(self.baseFilename):
            return codecs.open(self.baseFilename, 'r', 'utf-8').read()
        else:
            return ''
