# -*- coding: utf-8 -*-

from django.utils.translation import ugettext_lazy as _
from django.db import models
from django.core.exceptions import ValidationError

from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.fields import TrimmedCharField, TrimmedTextField
from common.utils.iterrecipes import pairwise
from travel.rasp.admin.importinfo.utils.thread import get_threads_last_mask_date


class Package(models.Model):
    title = TrimmedCharField(_(u'Название пакета'), max_length=255, unique=True)
    region = models.ForeignKey('www.Region', verbose_name=_(u'Регион'), blank=True, null=True)
    country = models.ForeignKey('www.Country', verbose_name=_(u'Cтрана'), default=225)
    autoimport = models.BooleanField(_(u"Автоимпорт"), default=True,
                                     help_text=_(u"Переимпортировать при большом импорте"))
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'), blank=False, null=False,
                               default=TransportType.BUS_ID)
    t_subtype = models.ForeignKey('www.TransportSubtype', verbose_name=_(u'Подтип транспорта'), blank=True,
                                  null=True, default=None)
    last_import_date = models.DateField(_(u'Дата импорта'), null=True, blank=True)
    last_import_datetime = models.DateTimeField(_(u'Дата-время импорта'), null=True)
    last_mask_date = models.DateField(_(u'Дата, после которой все календари ниток будут пустыми'),
                                      null=True, blank=True)
    content_manager = models.ForeignKey(
        'importinfo.ContentManager',
        verbose_name=_(u'Ответственный контент-менеджер'),
        null=True,
        blank=False
    )

    class Meta:
        app_label = 'red'
        verbose_name = _(u'Пакет КГА')
        verbose_name_plural = _(u'Пакеты КГА')

    def __unicode__(self):
        return self.title

    def __json__(self):
        return {
            'id': str(self.id),
            'title': self.title,
            'metaroute_id': self.metaroute_set.count() and str(self.metaroute_set.all()[0].id) or '',
        }

    def update_last_mask_date(self):
        from common.models.schedule import RThread

        thread_queryset = RThread.objects.filter(route__red_metaroute__package=self)

        self.last_mask_date = get_threads_last_mask_date(thread_queryset)

    def clean(self):
        if self.t_subtype_id and self.t_subtype.t_type_id != self.t_type_id:
            raise ValidationError(
                _(u'Подтипу транспорта {} соответствует тип транспорта {}')
                .format(self.t_subtype, self.t_subtype.t_type)
            )


class MetaRoute(models.Model):
    package = models.ForeignKey(Package, verbose_name=_(u'Пакет КГА'))
    number = TrimmedCharField(_(u'Номер рейса'), max_length=44, default='', blank=True)
    title = TrimmedCharField(_(u'Название рейса'), max_length=255, default='', blank=False)
    scheme = TrimmedTextField(_(u'Схема курсирования'), default='', null=False, blank=True)
    comment = TrimmedTextField(_(u'Комментарий'), default='', null=False, blank=True)
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'тип транспорта'))
    t_subtype = models.ForeignKey('www.TransportSubtype', verbose_name=_(u'Подтип транспорта'),
                                  blank=True, null=True, default=None)
    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u'поставщик'))
    allow_back_pseudo_routes = models.BooleanField(_(u'Разрешить построение обратных псевдорейсов'),
                                                   null=False, default=False)
    apply_base_stations = models.BooleanField(_(u"Учитывать «базовость» автовокзалов"), default=True)

    @property
    def full_name(self):
        return u'%s %s' % (self.number, self.title)

    @property
    def redadmin_link(self):
        return '/admin/redmap/%s/#route_id=%s' % (self.package_id, self.id)

    @property
    def path(self):
        """Упорядоченный query_set metaroutestations"""

        return self.metaroutestation_set.order_by('order')

    class Meta:
        app_label = 'red'
        verbose_name = _(u'метарейс')
        verbose_name_plural = _(u'метарейсы')

    def __unicode__(self):
        return u'%s %s' % (self.number, self.title)

    def get_scheme(self, today=None):
        from travel.rasp.admin.admin.red.scheme import SchemeParser
        if not today:
            today = environment.today()

        return SchemeParser(today).parse(self.scheme)

    def delete(self):
        from travel.rasp.admin.admin.red.metaimport import clean_metaroute, RedMetaRouteImporter
        clean_metaroute(RedMetaRouteImporter(self))
        super(MetaRoute, self).delete()

    def check_mrstations(self):
        return check_mrstations(list(self.path))

    def clean(self):
        if self.t_subtype_id and self.t_subtype.t_type_id != self.t_type_id:
            raise ValidationError(
                _(u'Подтипу транспорта {} соответствует тип транспорта {}')
                .format(self.t_subtype, self.t_subtype.t_type)
            )


class MetaRouteStation(models.Model):
    FLAG_CHOISES = ((None, _(u'Наследовать')), (True, _(u'Да')), (False, _(u'Нет')))

    metaroute = models.ForeignKey(MetaRoute, verbose_name=_(u'метарейс'))
    station = models.ForeignKey('www.Station', verbose_name=_(u'станция расписаний'))

    arrival = models.IntegerField(null=True, blank=True,
                                  help_text=_(u"Время от начала маршрута до прибытия (в минутах). "
                                              u"Прибытие на первую станцию заполнять нельзя(оставить пустым)"))

    departure = models.IntegerField(null=True, blank=True,
                                    help_text=_(u"Время от начала маршрута до отправления (в минутах). "
                                                u"Отправление с последней станции заполнять нельзя(оставить пустым)"))

    # TODO не использовать, удалить в 2.29
    time_inpath = models.IntegerField(_(u'Время в пути'), default=0)
    order = models.IntegerField(_(u'Порядок'), default=0, blank=True, editable=False)

    is_fuzzy = models.NullBooleanField(
        _(u"Нечеткие данные"),
        help_text=_(u"точное время не известно"),
        default=None, choices=FLAG_CHOISES)

    is_searchable_from = models.NullBooleanField(
        _(u"искать от станции"), default=None, choices=FLAG_CHOISES)

    is_searchable_to = models.NullBooleanField(
        _(u"искать до станции"),
        default=None, choices=FLAG_CHOISES)

    in_station_schedule = models.NullBooleanField(
        _(u"в расписании по станции"), default=None, choices=FLAG_CHOISES)

    in_thread = models.NullBooleanField(
        _(u"показывать на странице нитки"), default=None, choices=FLAG_CHOISES)

    @property
    def actual_is_fuzzy(self):
        if self.is_fuzzy is not None:
            return self.is_fuzzy
        else:
            return self.station.is_fuzzy

    @property
    def actual_is_searchable_to(self):
        if self.is_searchable_to is not None:
            return self.is_searchable_to
        else:
            return self.station.is_searchable_to

    @property
    def actual_is_searchable_from(self):
        if self.is_searchable_from is not None:
            return self.is_searchable_from
        else:
            return self.station.is_searchable_to

    @property
    def actual_in_station_schedule(self):
        if self.in_station_schedule is not None:
            return self.in_station_schedule
        else:
            return self.station.in_station_schedule

    @property
    def actual_in_thread(self):
        if self.in_thread is not None:
            return self.in_thread

        else:
            return self.station.in_thread

    class Meta:
        app_label = 'red'
        verbose_name = _(u'станция метарейса')
        verbose_name_plural = _(u'станции метарейсов')
        ordering = ['metaroute', 'order', 'id']
        unique_together = ("metaroute", "order")

    def save(self, *args, **kwargs):
        if not self.id and not self.order:
            try:
                self.order = self.metaroute.metaroutestation_set.all().order_by('-order')[0].order + 1
            except IndexError:
                self.order = 0

        return super(MetaRouteStation, self).save(*args, **kwargs)

    def insert(self, position, obj):
        if position == 'after':
            self.order = obj.order + 1
            increment_from = obj.order

        elif position == 'before':
            self.order = obj.order
            increment_from = obj.order - 1

        tail = list(self.metaroute.metaroutestation_set.filter(order__gt=increment_from))

        for station in tail:
            station.order += 1

        tail.reverse()

        for station in tail:
            station.save()

        new_obj = self.save()

        return new_obj

    def __unicode__(self):
        return u'%s - %s' % (self.station.L_title(), self.metaroute)

    def __json__(self):
        def null_bool(val):
            return {None: 'inherit', True: 'yes', False: 'no'}[val]

        return {
            'id': str(self.id),
            'www_station': {'id': self.station.id,
                            'title': self.station.title,
                            'majority_id': self.station.majority.id},
            'arrival': self.arrival,
            'departure': self.departure,
            'lon': self.station.longitude,
            'lat': self.station.latitude,
            'is_fuzzy': null_bool(self.is_fuzzy),
            'is_searchable_to': null_bool(self.is_searchable_to),
            'is_searchable_from': null_bool(self.is_searchable_from),
            'in_station_schedule': null_bool(self.in_station_schedule),
            'order': self.order,
        }


def check_mrstations(mrstations):
    check_errors = []

    _check_mrstations_fill(mrstations, check_errors)
    _check_mrstations_stop_time(mrstations, check_errors)
    _check_mrstations_order(mrstations, check_errors)

    return check_errors


def _check_mrstations_fill(mrstations, check_errors):
    if len(mrstations) < 2:
        return

    _check_fill_first_mrs(mrstations[0], check_errors)
    _check_fill_last_mrs(mrstations[-1], check_errors)

    for mrs in mrstations[1:-1]:
        _check_fill_middle_rts(mrs, check_errors)


def _check_fill_first_mrs(mrs, check_errors):
    if mrs.arrival is not None:
        check_errors.append(_(u'Время прибытия на первую станцию "{station}" должно быть пустым')
                            .format(station=mrs.station.title))

    if mrs.departure != 0:
        check_errors.append(_(u'Время отправления с первой станции "{station}" должно быть равным нулю')
                            .format(station=mrs.station.title))


def _check_fill_last_mrs(mrs, check_errors):
    if mrs.departure is not None:
        check_errors.append(_(u'Время отправления с последней станции "{station}" должно быть пустым')
                            .format(station=mrs.station.title))

    if mrs.arrival is None:
        check_errors.append(_(u'Должно быть указано время прибытия на последнюю станцию "{station}"')
                            .format(station=mrs.station.title))


def _check_fill_middle_rts(mrs, check_errors):
    if mrs.arrival is None:
        check_errors.append(_(u'Должно быть указано время прибытия на станцию "{station}"')
                            .format(station=mrs.station.title))

    if mrs.departure is None:
        check_errors.append(_(u'Должно быть указано время отправления со станции "{station}"')
                            .format(station=mrs.station.title))


def _check_mrstations_stop_time(mrstations, check_errors):
    for mrs in mrstations:
        if mrs.departure is None or mrs.arrival is None:
            continue

        if mrs.arrival > mrs.departure:
            check_errors.append(_(u'Время прибытия больше времени отправления для станциям "{station}"')
                                .format(station=mrs.station.title))

        if mrs.arrival == mrs.departure:
            check_errors.append(_(u'Время стоянки не может быть менее 1 минуты (станция "{station}")')
                                .format(station=mrs.station.title))


def _check_mrstations_order(mrstations, check_errors):
    for mrs, mrs_next in pairwise(mrstations):
        if mrs.departure is not None and mrs_next.arrival is not None:
            if mrs.departure > mrs_next.arrival:
                check_errors.append(_(u'Время для станции "{station}" больше, '
                                      u'чем для станции "{next_station}"')
                                    .format(station=mrs.station.title,
                                            next_station=mrs_next.station.title))
