# -*- coding: utf-8 -*-

import re
from datetime import datetime, date
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.importinfo.triangle_import.mask_parser import TriangleMaskTemplateParser
from travel.rasp.admin.scripts.schedule.utils.mask_builders import MaskBuilder
from travel.rasp.admin.lib.import_bounds import import_bounds


class ParseError(SimpleUnicodeException):
    pass


class SchemeParser(object):

    def __init__(self, today, log=None):
        self.today = today
        self.log = log

    def parse(self, src_scheme):
        e_time_separators = ur'[\.\:\-]'
        e_time = ur'\d{1,2}' + e_time_separators + ur'\d{1,2}'
        re_scheme = re.compile(ur'((' + e_time + ur')\s*(?:\((.*?)\))?)', flags=re.U | re.M | re.I)
        re_scheme_split = re.compile(ur'' + e_time + ur'\s*(?:\(.*?\))?', flags=re.U | re.M | re.I)
        days_tokens = re_scheme_split.split(src_scheme)
        for day_token in days_tokens:
            day_token = day_token and day_token.strip() or ''
            if day_token and day_token != ',':
                raise ParseError(u'Ошибка в %s' % day_token)

        found = re_scheme.findall(src_scheme)

        def parse_time(src):
            src = re.sub(e_time_separators, '.', src)
            try:
                return datetime.strptime(src, '%H.%M').time()
            except ValueError:
                raise ParseError(u'Ошибка в значении времени %s' % src)

        result = [(parse_time(t), self.parse_days(d)) for _, t, d in found]

        return [(time, mask, can_extrapolate) for time, (mask, can_extrapolate) in result if mask]

    def parse_days(self, template_text):
        start, end = import_bounds(self.today, forward=90)
        return RedMaskTemplateParser(start, end, self.today, self.log).parse_template(template_text)


class RedMaskTemplateParser(object):
    default_can_extrapolate = True

    def __init__(self, start, end, today, log):
        self.today = today
        self.start = start
        self.end = end
        self.log = log
        self.builder = MaskBuilder(self.start, self.end, self.today, log=self.log)

    def parse_template(self, template_text):
        """ метод возвращает маску и можно ли ее экстраполировать. """
        self.original_template_text = template_text
        template_text = template_text.strip().lower()

        if not template_text:
            return self.builder.daily_mask(), self.default_can_extrapolate

        for parse_function in [self.parse_disabled, self.parse_triangle, self.parse_alexandr, self.parse_bus_specific]:
            try:
                return parse_function(template_text)
            except Exception:
                pass

        raise ParseError(u'Не смогли разобрать {}'.format(template_text))

    def parse_triangle(self, template_text):
        triangle_mask_parser = TriangleMaskTemplateParser(self.start, self.end, self.today)
        mask = triangle_mask_parser.parse_template(template_text)
        can_extrapolate = not getattr(triangle_mask_parser, 'has_modifier_end_date', False)
        return mask, can_extrapolate

    def parse_disabled(self, template_text):
        if template_text in [u'он', u'отм', u'отменен']:
            return None, self.default_can_extrapolate
        raise ParseError(u'Не является схемой отмененного рейса {}'.format(template_text))

    def parse_alexandr(self, template_text):
        standard = {
            u'пв': '567',        # по пятницам и выходным
            u'кпв': '1234',        # кроме пятниц и выходных
            u'кп': '123467',        # кроме пятниц
            u'кс': '123457',        # кроме суббот
            u'квск': '123456',        # кроме воскресений
            u'кпвск': '12346',        # кроме пятниц и воскресений
            u'пвск': '57',        # по пятницам и воскресеньям
            u'пс': '56',        # по пятницам и субботам
            u'пнпв': '1567',        # по понедельникам, пятницам и выходным дням
            u'п': '5',        # по пятницам
            u'с': '6',        # по субботам
        }

        if template_text in standard:
            return self.builder.mask_from_days_of_week(standard[template_text]), self.default_can_extrapolate

        def parse_one_mask(mask_text):
            allowed_tokens = {
                u'пн': '1',        # понедельник
                u'вт': '2',        # вторник
                u'ср': '3',        # среда
                u'чт': '4',        # четверг
                u'пт': '5',        # пятница
                u'сб': '6',        # субботу
                u'вс': '7',        # воскресенье
                u'вск': '7',        # воскресенье
                u'в': lambda: self.builder.ycal_weekends_and_holidays_mask(),        # выходные дни
                u'р': lambda: self.builder.ycal_workdays_mask(),        # рабочие дни
                u'неч': lambda: self.builder.odd_mask(),        # нечетные числа
                u'четн': lambda: self.builder.even_mask(),        # четные числа
            }

            if mask_text in allowed_tokens:
                element = allowed_tokens[mask_text]

                if hasattr(element, '__call__'):
                    return element()
                else:
                    return self.builder.mask_from_days_of_week(element)
            else:
                try:
                    dt = datetime.strptime(re.sub('[:-]', '.', mask_text), '%d.%m')
                except ValueError:
                    raise ParseError()

                def condition_for_day(day):
                    return lambda checke_day: checke_day.month == day.month and checke_day.day == day.day

                return self.builder.mask_from_day_condition(condition_for_day(dt))

            raise ParseError(u'Не можем разобрать [%s]' % mask_text)

        elems = set(re.sub('[,;]|(?<!\d)[:\.](?!\d)', ' ', template_text).split())
        reverted = u'кр' in elems
        elems.discard(u'кр')
        mask = reduce(lambda q, p: q | p, map(parse_one_mask, elems))
        if reverted:
            mask = self.builder.daily_mask() ^ mask

        return mask, self.default_can_extrapolate

    def parse_bus_specific(self, template_text):
        # "через день с 20 июля"
        months = [
            u'янв\w*',
            u'фев\w*',
            u'март\w*',
            u'апр\w*',
            u'май\w*',
            u'июн\w*',
            u'июл\w*',
            u'авг\w*',
            u'сен\w*',
            u'окт\w*',
            u'ноя\w*',
            u'дек\w*',
        ]
        e_month = ur'|'.join(e for e in months)
        re_through_the_day = re.compile(ur'^\s*через\s+день\s+с\s+(\d{1,2})\s+(' + e_month + ur')\s*$', re.U)
        through_the_day = re_through_the_day.match(template_text)
        if through_the_day:
            day, month = through_the_day.groups()
            for month_num, e in enumerate(months):
                month_num += 1
                if re.compile(e, re.U).match(month):
                    break

            from_date = date(self.start.year, month_num, int(day))
            if from_date < self.start:
                from_date = date(self.start.year + 1, month_num, int(day))

            return self.builder.through_the_day_mask(from_date), self.default_can_extrapolate

        # "2 и 4 вторник месяца"
        weekdays = [
            u'поне\w*',
            u'вторн\w*',
            u'сред\w*',
            u'четверг\w*',
            u'пятниц\w*',
            u'суббот\w*',
            u'воскресень\w*',
        ]
        match = re.compile(ur'^((?:(?:\d+)\s*[ ,и]?\s*)*) ' + ur'(' + ur'|'.join(weekdays) + ur').*$',
                           re.U).match(template_text)

        if match:
            nums, day = match.groups()
            nums = set(map(int, re.compile(r'\d+').findall(nums)))
            for week_day, text in enumerate(weekdays):
                if re.compile(text, re.U).match(day):
                    mask = self.builder.mask_from_day_condition(
                        lambda tested_date:
                            tested_date.weekday() == week_day and ((tested_date.day - 1) / 7 + 1) in nums
                    )
                    return mask, self.default_can_extrapolate

        raise ParseError(u'Не можем разобрать [%s]' % template_text)
