# -*- coding: utf-8 -*-

from __future__ import absolute_import

import logging
import re
from functools import wraps
from math import fabs

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.core.urlresolvers import reverse
from django.db import transaction
from django.db.models import Min, Max
from django.http import HttpResponseForbidden, HttpResponseRedirect
from django.shortcuts import redirect
from django.template.context_processors import csrf
from django.utils import translation
from django.utils.translation import ugettext_lazy as _, ugettext
from django.views.decorators.csrf import ensure_csrf_cookie

from common.models.geo import Station, StationMajority, Settlement, Country, Region
from common.models.schedule import Supplier
from common.models.transport import TransportType, TransportSubtype
from common.utils.httpresponses import jsonp_response
from travel.rasp.admin.lib.jinja import render_to_response

from travel.rasp.admin.admin.red.forms import PackageForm, MetaRouteInfoForm
from travel.rasp.admin.admin.red.models import Package, MetaRouteStation, MetaRoute
from travel.rasp.admin.admin.red.metaimport import PackageAction, MetaRouteAction
from travel.rasp.admin.admin.www_stations.convertors import json_www_station
from travel.rasp.admin.admin.views import geoadmin


log = logging.getLogger(__name__)


def redadmin_permission_required_or_403(func):
    @wraps(func)
    def wrapper(request, *args, **kwargs):
        if not request.user.has_perm('red.change_metaroute'):
            return HttpResponseForbidden()

        return func(request, *args, **kwargs)

    return wrapper


@redadmin_permission_required_or_403
@ensure_csrf_cookie
def admin(request, package_id=None):
    context = {
        'csrf_token': csrf(request)['csrf_token'],
        'MAIN_INSTANCE': 'false' if settings.SERVICE_INSTANCE else 'true'
    }

    packages = Package.objects.all()
    package = None

    if package_id:
        try:
            package = Package.objects.get(pk=package_id)

        except Package.DoesNotExist:
            pass

    if not package and len(packages):
        return redirect('admin_red_old', package_id=packages[0].id)

    if package:
        package.routes = list(package.metaroute_set.all())
        package.form = PackageForm(instance=package)

        package.routes = list(package.metaroute_set.all())

    context.update({'STATIC_URL': settings.STATIC_URL,
                    'package': package,
                    'package_json': package and package.__json__() or {},
                    'packages': packages,
                    'route_info_form': MetaRouteInfoForm(),
                    't_types': dict([(t.id, {u'id': t.id,
                                             u'code': t.code,
                                             u'title': t.L_title()})
                                     for t in TransportType.objects.all()]),
                    'station_majorities': [{'id': m.id,
                                            'title': m.title}
                                           for m in StationMajority.objects.all()]
                    })

    country_code = package and package.country.code or ''
    context['map_lang'] = settings.ADMIN_COUNTRY_YMAPLANGUAGE_MAP.get(country_code, 'ru-RU')

    return render_to_response('red/admin.jhtml', context, request=request)


@redadmin_permission_required_or_403
@jsonp_response
@transaction.atomic
def www_goroda(request):
    settlement_id = request.POST.get('settlement_id', None)
    settlement = Settlement.objects.get(pk=settlement_id)

    sids = set(request.POST.getlist('sids[]'))

    stations = Station.objects.exclude(t_type__code='pseudo-gortrans').\
        filter(id__in=sids)

    if stations.count() < len(sids):
        return {'status': 'bad',
                'message': 'can\'t find all requested stations'}

    stations.update(settlement=settlement)

    return {'status': 'ok'}


@login_required
@jsonp_response
def settlement(request, settlement_id):
    settlement = Settlement.objects.get(pk=settlement_id)

    return {
        'id': settlement.id,
        'title': settlement.title,
    }


def get_bounds(metaroute):

    bounds = metaroute.metaroutestation_set.aggregate(
        min_lon=Min('station__longitude'), min_lat=Min('station__latitude'),
        max_lon=Max('station__longitude'), max_lat=Max('station__latitude'))

    # На крайний случай, откроем карту на Москве
    if None in bounds.values():
        bounds = {'min_lon': 37.6622, 'min_lat': 55.757601,
                  'max_lon': 37.6622, 'max_lat': 55.757601}

    w_lon = fabs(bounds['max_lon'] - bounds['min_lon'])
    w_lat = fabs(bounds['max_lat'] - bounds['min_lat'])

    radius = 0.05

    if w_lon < radius:
        c_lon = ((bounds['min_lon'] + bounds['min_lon']) % 360) / 2
        bounds['min_lon'] = c_lon - radius
        bounds['max_lon'] = c_lon + radius

    if w_lat < radius:
        c_lat = ((bounds['min_lat'] + bounds['min_lat']) % 360) / 2
        bounds['min_lat'] = (c_lat - radius + 180) % 360 - 180
        bounds['max_lat'] = (c_lat + radius + 180) % 360 - 180

    return [[bounds['min_lon'] - 0.1, bounds['min_lat'] - 0.1],
            [bounds['max_lon'] + 0.1, bounds['max_lat'] + 0.1]]


def json_metaroute(r):
    return {
        'id': r.id,
        'number': r.number,
        'title': r.title,
        'scheme': r.scheme,
        'comment': r.comment,

        't_type': r.t_type.id,
        'supplier': r.supplier.id,
        'geoadmin_link': '/admin/red/metaroute/%s/' % r.id,
        'bounds': get_bounds(r),
        'stations': list(r.path),
        'routes': [{'id': route.id, 'title': route.title}
                   for route in r.route_set.all()],
    }


@redadmin_permission_required_or_403
@jsonp_response
def metaroute(request, metaroute_id):
    try:
        route = MetaRoute.objects.get(pk=metaroute_id)

    except MetaRoute.DoesNotExist:
        from django.http import Http404
        raise Http404

    return json_metaroute(route)


@redadmin_permission_required_or_403
@jsonp_response
def mrstation_delete(request):
    station_id = request.POST.get('station_id', None)

    try:
        mrstation = MetaRouteStation.objects.get(pk=station_id)

    except MetaRouteStation.DoesNotExist:
        return {}

    metaroute = mrstation.metaroute
    mrstation.delete()

    new_order = dict(metaroute.metaroutestation_set.values_list('pk', 'order'))

    return {
        'status': 'ok',
        'order': new_order,
        'route_check_error': u'<br />'.join(metaroute.check_mrstations()),
    }


@redadmin_permission_required_or_403
@jsonp_response
def mrstation_add(request):
    metaroute_id = request.POST.get('metaroute_id', None)
    station_id = request.POST.get('station_id', None)
    order = request.POST.get('order', None)
    before = request.POST.get('before', None)
    after = request.POST.get('after', None)

    if before and after:
        return {'status': 'wrong position'}

    metaroute = MetaRoute.objects.get(pk=metaroute_id)
    www_station = Station.objects.exclude(t_type__code='pseudo-gortrans').\
        get(pk=station_id)

    mrstation = MetaRouteStation(metaroute=metaroute, station=www_station)

    if before or after:
        if before:
            position = 'before'
            obj = MetaRouteStation.objects.get(pk=int(before))

        elif after:
            position = 'after'
            obj = MetaRouteStation.objects.get(pk=int(after))

        mrstation = MetaRouteStation(metaroute=metaroute,
                                     station=www_station,
                                     order=order)
        mrstation.insert(position, obj)

    else:
        mrstation.save()

    return {
        'mrstation': mrstation,
        'route_check_error': u'<br />'.join(metaroute.check_mrstations()),
    }


@redadmin_permission_required_or_403
@jsonp_response
@transaction.atomic
def mrstation_create(request):
    country_id = request.POST['country_id'] or None
    region_id = request.POST['region_id'] or None
    settlement_id = request.POST['settlement_id'] or None
    title = request.POST['title']
    longitude = request.POST['longitude']
    latitude = request.POST['latitude']
    metaroute_id = request.POST['metaroute_id']
    majority_id = request.POST.get('majority_id', 4)
    before = request.POST.get('before', None)
    after = request.POST.get('after', None)

    if before and after:
        return {'status': 'wrong position'}

    if not title.strip():
        return {'status': 'emty title'}

    metaroute = MetaRoute.objects.get(pk=metaroute_id)

    try:
        country = Country.objects.get(pk=country_id)

    except Country.DoesNotExist:
        country = None

    try:
        region = Region.objects.get(pk=region_id)

    except Region.DoesNotExist:
        region = None

    try:
        settlement = Settlement.objects.get(pk=settlement_id)

    except Settlement.DoesNotExist:
        settlement = None

    region = settlement and settlement.region or region
    country = (settlement and settlement.country) or \
              (region and region.country) or country

    www_station = Station.objects.create(
        country=country,
        region=region,
        settlement=settlement,
        t_type=metaroute.t_type,
        title=title,
        longitude=longitude,
        latitude=latitude,
        majority_id=majority_id,
    )

    if before or after:
        if before:
            position = 'before'
            obj = MetaRouteStation.objects.get(pk=int(before))

        elif after:
            position = 'after'
            obj = MetaRouteStation.objects.get(pk=int(after))

        mrstation = MetaRouteStation(metaroute=metaroute,
                                     station=www_station)
        mrstation.insert(position, obj)

    else:
        mrstation = MetaRouteStation.objects.create(metaroute=metaroute,
                                                    station=www_station)

    return {
        'www_station': json_www_station(www_station),
        'red_station': mrstation,
        'route_check_error': u'<br />'.join(metaroute.check_mrstations()),
    }


@redadmin_permission_required_or_403
@jsonp_response
def package_save(request):
    package_id = request.POST.get('package_id')
    title = request.POST.get('title', '')

    if not title.strip():
        return {}

    package = Package.objects.get(pk=package_id)
    package.title = title
    package.save()

    return {'status': 'ok'}


@redadmin_permission_required_or_403
@jsonp_response
@transaction.atomic
def route_save(request):
    kwargs = dict([(k, request.POST.get(k))
                   for k in ('metaroute_id', 'number', 'title', 'scheme',
                             'comment', 't_type_id', 'supplier_id',)])
    metaroute = MetaRoute.objects.get(pk=kwargs.pop('metaroute_id'))

    def item_parse(item):
        match = re.match(ur'(\d+)\|(\d*)\|(\d*)\|([\w\|]+)', item, re.U)

        id_, arrival, departure, fuzzy_flags = match.groups()

        fuzzy_mapping = {'yes': True, 'no': False, 'inherit': None}
        fuzzy_flags = [fuzzy_mapping[val] for val in fuzzy_flags.split('|')]

        fuzzy_flags_names = ['is_fuzzy', 'is_searchable_to',
                             'is_searchable_from', 'in_station_schedule']
        fuzzy_flags = dict(zip(fuzzy_flags_names, fuzzy_flags))

        arrival = int(arrival) if arrival else None
        departure = int(departure) if departure else None

        return int(id_), (arrival, departure, fuzzy_flags)

    for k, v in kwargs.items():
        setattr(metaroute, k, v)

    schedule_list = request.POST.getlist('schedule[]')
    schedule = dict(map(item_parse, schedule_list))
    mrstations = list(metaroute.path)

    for index, mrs in enumerate(mrstations):
        mrs.arrival, mrs.departure, fuzzy_flags = schedule[mrs.id]

        for key, val in fuzzy_flags.items():
            log.debug('%s %s %s' % (mrs.id, key, val))
            setattr(mrs, key, val)

    metaroute.save()

    for mrs in mrstations:
        mrs.save()

    return {
        'status': 'ok',
        'route_check_error': u'<br />'.join(metaroute.check_mrstations()),
    }


@redadmin_permission_required_or_403
@jsonp_response
def route_create(request):
    package_id = request.POST.get('package_id')
    package = Package.objects.get(pk=package_id)

    try:
        t_type_id, supplier_id = package.metaroute_set.\
            order_by('-id').values_list('t_type_id', 'supplier_id')[0]

    except Exception:
        t_type_id, supplier_id = 3, 1

    metaroute = package.metaroute_set.create(t_type_id=t_type_id,
                                             supplier_id=supplier_id)
    return json_metaroute(metaroute)


@redadmin_permission_required_or_403
@jsonp_response
@transaction.atomic
def route_clone(request):
    try:
        metaroute = MetaRoute.objects.get(pk=request.POST.get('metaroute_id'))

    except (MetaRoute.DoesNotExist, ValueError, TypeError):
        return {'message': _(u'Не нашли вашего метарейса')}

    reversed_ = bool(request.POST.get('reversed', False))
    scheme = not reversed_ and metaroute.scheme or ''
    comment = not reversed_ and metaroute.comment or ''

    clone = MetaRoute.objects.create(
        package=metaroute.package,
        number=metaroute.number,
        title=metaroute.title,
        scheme=scheme,
        comment=comment,
        t_type=metaroute.t_type,
        supplier=metaroute.supplier,
    )

    mrstations = list(metaroute.path)

    if mrstations:
        final = mrstations[-1].arrival

        if reversed_:
            mrstations.reverse()

        for i, mrs in enumerate(mrstations):
            arrival = mrs.arrival
            departure = mrs.departure

            if reversed_:
                arrival = final - mrs.departure if mrs.departure is not None else None
                departure = final - mrs.arrival if mrs.arrival is not None else None

            clone.metaroutestation_set.create(station=mrs.station,
                                              arrival=arrival, departure=departure,
                                              is_fuzzy=mrs.is_fuzzy,
                                              order=i)

    return json_metaroute(clone)


@redadmin_permission_required_or_403
@jsonp_response
def route_delete(request):
    MetaRoute.objects.get(pk=request.POST.get('metaroute_id')).\
        delete()

    return {'status': 'ok'}


@redadmin_permission_required_or_403
@jsonp_response
def route_reorder(request):
    try:
        metaroute = MetaRoute.objects.get(pk=request.POST.get('metaroute_id'))

    except (MetaRoute.DoesNotExist, ValueError, TypeError):
        return {
            'status': 'error',
            'message': _(u'Метарейс не найден')
        }

    order_list = request.POST.getlist('order[]')

    stations = list(metaroute.path)

    if len(order_list) != len(stations):
        return {
            'status': 'error',
            'message': _(u'Количество станций не совпадает')
        }

    stations_dict = {s.id: s for s in stations}

    start_order = max(stations, key=lambda x: x.order).order + 1

    for i, station_id in enumerate(order_list, start=start_order):
        station = stations_dict.get(int(station_id))
        station.order = i
        station.save()

    return {
        'status': 'ok',
        'order': dict((k, v.order) for k, v in stations_dict.iteritems()),
        'route_check_error': u'<br />'.join(metaroute.check_mrstations()),
    }


@redadmin_permission_required_or_403
def import_metaroute(request):
    try:
        metaroute_id = request.POST.get('route_id')
        metaroute = MetaRoute.objects.get(pk=metaroute_id)

    except (MetaRoute.DoesNotExist, ValueError, TypeError):
        return render_to_response('red/import_error_message.jhtml',
                                  {'message': _(u'Нет такого мета-рейса')},
                                  request=request)

    action = MetaRouteAction(metaroute, 'import_metaroute')

    if action.do():
        return HttpResponseRedirect(action.log_view_url)

    else:
        return render_to_response('red/import_error_message.jhtml',
                                  {'message': _(u'Не смогли импортировать %s') % (action.error or u'')},
                                  request=request)


@redadmin_permission_required_or_403
def import_package(request):
    try:
        package_id = request.POST.get('package_id')
        package = Package.objects.get(pk=package_id)

    except (MetaRoute.DoesNotExist, ValueError, TypeError):
        return render_to_response('red/import_error_message.jhtml',
                                  {'message': _(u'Нет такого пакета')},
                                  request=request)

    action = PackageAction(package, 'import_package')

    if action.do():
        return HttpResponseRedirect(action.log_view_url)

    else:
        return render_to_response('red/import_error_message.jhtml',
                                  {'message': u'Не смогли импортировать %s' %
                                              (action.error or u'')},
                                  request=request)


@redadmin_permission_required_or_403
@ensure_csrf_cookie
def new(request):
    lang = translation.get_language()
    options = {
        'title': ugettext(u'Красный геоадмин'),
        'adminType': 'red',
        'staticUrl': settings.STATIC_URL + 'rasp/geoadmins/',
        'maintenanceStatusUrl': reverse('maintenance_status'),
        'mainInstance': False if settings.SERVICE_INSTANCE else True,
        'lang': lang,
        't_types': dict([(t.id, {u'id': t.id, u'code': t.code, u'title': t.L_title()})
                         for t in TransportType.objects.all()]),
        't_subtypes':
            [{u'id': '', u'title': u'--------'}] +
            [{u'id': t.id, u'title': t.L_title()} for t in TransportSubtype.objects.all()],
        'station_majorities': [{'id': m.id, 'title': m.title}
                               for m in StationMajority.objects.all()],
        'suppliers': [{'id': s.id, 'title': s.title} for s in Supplier.objects.all()],
    }

    return geoadmin.render(options, lang)
