# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import json
from urlparse import urlparse, urlunparse

import re
import shutil
import time
import os
import os.path
import zipfile

import six
from codecs import open
from collections import defaultdict, OrderedDict
from datetime import datetime, date
from dateutil import parser
from email.header import Header
from itertools import islice
from io import StringIO
from traceback import format_exc

import pytils.dt as russian_dt
from chardet.universaldetector import UniversalDetector
from django import forms
from django.apps import apps
from django.conf import settings
from django.contrib import messages
from django.contrib.admin import site
from django.contrib.admin.widgets import AdminDateWidget, ForeignKeyRawIdWidget
from django.contrib.auth.decorators import login_required, permission_required, user_passes_test
from django.core.exceptions import PermissionDenied
from django.core.files import File
from django.core.urlresolvers import reverse
from django.db import transaction
from django.db.models import Count, Sum
from django.forms.widgets import CheckboxInput
from django.http import HttpResponse, HttpResponseRedirect, Http404, JsonResponse
from django.shortcuts import get_object_or_404, render as django_render
from django.template.response import TemplateResponse
from django.utils.html import escape
from django.utils.safestring import mark_safe, mark_for_escaping
from django.utils.encoding import smart_text
from django.utils.text import Truncator

from common.db.mds.clients import mds_s3_common_client
from common.models.geo import (
    District, Country, Settlement, Region, Station, StationPhone, ExternalDirection, Direction
)
from common.models.schedule import RThread, Supplier
from common.models.staticpages import StaticPage
from common.models.transport import TrainPseudoStationMap
from common.db.maintenance import read_conf
from common.db.switcher import switcher
from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.utils.httpresponses import jsonp_response
from common.utils.jsonutils import serialize
from common.utils.unicode_csv import UnicodeDictReader
from travel.rasp.admin.admin.modules.zone_reimport import (
    remove_rtstaion_references, get_direction_xml_tree, reimport_direction, import_direction, import_suburbanzones
)
from travel.rasp.admin.importinfo import admin as importinfo_admin
from travel.rasp.admin.importinfo.models import RelatedLog
from travel.rasp.admin.importinfo.models.af import AFScheduleFile, AFSubwayFile
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.jinja import render_to_string
from travel.rasp.admin.lib.logs import add_stream_handler, remove_stream_handler, get_collector, remove_collector
from travel.rasp.admin.lib.maintenance.flags import JOBS, flags as maintenance_flags
from travel.rasp.admin.lib.maintenance.scripts import (run_script, has_script_permission, get_allowed_scripts,
                                     get_script_params, run_admin_task)
from travel.rasp.admin.lib.tmpfiles import clean_temp, get_tmp_filepath
from travel.rasp.admin.lib.unzip import unpack_zip_file
from travel.rasp.admin.mds_files_viewer.views import files_listing
from order.models import Partner, StatisticsEntry
from travel.rasp.admin.scripts.support_methods import esr_leading_zeros
from travel.rasp.admin.scripts.utils.file_wrapper.uploaders import get_schedule_upload_wrapper
from travel.rasp.admin.scripts.utils.file_wrapper.mds_utils import get_log_key, get_instance_mds_prefix, MDS_CRON_RUNS_LOG, MDS_ADMIN_RUNS_LOG
from travel.rasp.admin.www.utils.common import first
from travel.rasp.admin.www.utils.data import items_tds
from travel.rasp.admin.www.utils.district_import import import_district


log = logging.getLogger(__name__)


DUMP_SANDBOX_TEMPLATE = 'https://sandbox.yandex-team.ru/resources?type=RASP_MYSQL_DBDUMP&limit=20&attrs={{"environment":"{}"}}'
define_setting(
    'DUMP_SANDBOX_URL',
    {Configuration.PRODUCTION: DUMP_SANDBOX_TEMPLATE.format(Configuration.PRODUCTION)},
    default=DUMP_SANDBOX_TEMPLATE.format(Configuration.TESTING)
)


# quick tree zone
@login_required
def quick_tree_planet(request, id=None):
    """ Страница планеты быстрого спуска по гео-дереву """
    context = {
        'sub_classes': [{
            'objects_rows': items_tds(Country.objects.all(), 4),
            'object_type': 'country',
            'not_lowest': 1,
            'add_object_params': '',
            'title': u'Страны'
        }]
    }

    return django_render(request, 'admin/quick_tree.html', context)


@login_required
def quick_tree_country(request, id):
    """ Страница страны быстрого спуска по гео-дереву """
    try:
        country = Country.objects.get(id=id)
    except Country.DoesNotExist:
        id = 225
        country = Country.objects.get(id=id)

    regions = Region.objects.filter(country=country)
    settlements = Settlement.objects.filter(country=country, region__isnull=True)
    stations = Station.objects.filter(hidden=False, country=country, region__isnull=True, settlement__isnull=True)

    context = {
        'current_node': country,
        'sub_classes': [
            {'objects_rows': items_tds(regions),
             'object_type': 'region',
             'not_lowest': 1,
             'add_object_params': 'country_id=' + str(id),
             'title': u'Области'},
            {'objects_rows': items_tds(settlements),
             'object_type': 'settlement',
             'not_lowest': 1,
             'add_object_params': 'country_id=' + str(id),
             'title': u'Города'},
            {'objects_rows': items_tds(stations),
             'object_type': 'station',
             'not_lowest': 1,
             'add_object_params': 'country_id=' + str(id),
             'title': u'Станции'}
        ]
    }

    return django_render(request, 'admin/quick_tree.html', context)


@login_required
def quick_tree_region(request, id):
    """ Страница области быстрого спуска по гео-дереву """
    context = {}

    try:
        region = Region.objects.get(id=id)
        settlements = Settlement.objects.filter(region=region).order_by('majority__id', 'title')
        settlements = map(lambda o: {
            'id': o.id,
            'title': o.title,
            'style': _get_settlement_style(o)
        }, settlements)
    except Region.DoesNotExist:
        region = None
        settlements = []
    country_id = None
    if region and region.country:
        country_id = region.country.id

    sub_classes = [{
        'objects_rows': items_tds(settlements),
        'object_type': 'settlement',
        'not_lowest': 1,
        'add_object_params': 'region_id=' + str(id) + (country_id and ('&country_id=' + str(country_id)) or ''),
        'title': u'Города'
    }]

    if region:
        stations = (Station.objects.filter(hidden=False, region=region, settlement__isnull=True)
                                   .order_by('majority__id', 'title'))
        stations = map(lambda o: {
            'id': o.id,
            'title': o.title,
            'code': o.t_type.code,
            'style': _get_station_style(o)
        }, stations)
        sub_classes.append({
            'objects_rows': items_tds(stations),
            'object_type': 'station',
            'not_lowest': 1,
            'add_object_params': 'region_id=' + str(region.id),
            'title': u'Станции'
        })
    context['current_node'] = region
    context['sub_classes'] = sub_classes

    return django_render(request, 'admin/quick_tree.html', context)


def _get_settlement_style(sett):
    if sett.majority_id in [1, 2]:
        return 'font-weight: bold;'
    if sett.majority_id in [5]:
        return 'font-size: 9px; font-style: Italic;'
    return ''


def _get_station_style(st):
    if st.majority_id in [1]:
        return 'font-weight: bold;'
    if st.majority_id in [3]:
        return 'font-size: 9px; font-style: Italic;'
    return ''


@login_required
def quick_tree_settlement(request, id):
    """ Страница города быстрого спуска по гео-дереву """
    context = {}
    try:
        settlement = Settlement.objects.get(id=id)
        stations = Station.objects.filter(settlement=settlement).order_by('majority__id', 'title')
        stations = map(lambda o: {
            'id': o.id,
            'title': o.title,
            'icon': '%s_small.gif' % o.t_type.code,
            'style': _get_station_style(o)
        }, stations)
    except Settlement.DoesNotExist:
        settlement = None
        stations = []

    sub_classes = [{
        'objects_rows': items_tds(stations),
        'object_type': 'station',
        'not_lowest': 1,
        'add_object_params': 'settlement_id=' + str(id),
        'title': u'Станции'
    }]
    context['current_node'] = settlement
    context['sub_classes'] = sub_classes

    return django_render(request, 'admin/quick_tree.html', context)


@login_required
def quick_tree_station(request, id):
    """ Страница области быстрого спуска по гео-дереву """
    context = {}
    try:
        station = Station.objects.get(hidden=False, id=id)
        station_phones = StationPhone.objects.filter(station=station)
        station_phones = map(lambda o: {
            'id': o.id,
            'title': o.phone + (o.note and u' (%s)' % o.note)
        }, station_phones)
    except Station.DoesNotExist:
        station = None
        station_phones = []
    sub_classes = [{
        'objects_rows': items_tds(station_phones),
        'object_type': 'stationphone',
        'not_lowest': 0,
        'add_object_params': 'station_id=' + str(id),
        'title': u'Телефоны станции'
    }]
    context['current_node'] = station
    context['sub_classes'] = sub_classes

    return django_render(request, 'admin/quick_tree.html', context)


@login_required
def show_file(request, app_label, model, field_name, id):
    model = apps.get_model(app_label, model)
    if model and request.user.has_perm(u"%s.change_%s" % (app_label,
                                                          model.__name__.lower())):
        instance = model.objects.get(id=id)
        content = getattr(instance, field_name)
        encoding = content.encoding
        filename = getattr(instance, '%s_name' % field_name, None)
        response = HttpResponse(content.encode(encoding), content_type="text/plain; charset=%s" % encoding)

        if filename:
            response['Content-Disposition'] = 'attachment; filename=%s' % str(Header(filename, encoding))

        return response
    else:
        HttpResponseRedirect("/admin/")


@login_required
def show_log(request, log_file_key):
    content = mds_s3_common_client.get_data(key=log_file_key).read().decode('utf8')
    return HttpResponse(content, content_type="text/plain; charset=utf8")


def show_run_log_list(request, prefix, retrieve_dt, log_title, add_preview=False, count=None):
    files = files_listing(prefix=prefix, mds_client=mds_s3_common_client)
    log_names = [f['key'].split('/')[-1] for f in files if f['key'].endswith('.log')]
    logs = []

    for log_name in log_names:
        dt = retrieve_dt(log_name)
        try:
            dt = parser.parse(dt, yearfirst=True)
        except ValueError:
            continue

        logs.append({
            'dt': dt,
            'log_name': log_name,
            'key': get_log_key(prefix, log_name),
            'first_lines': ''
        })

    logs.sort(reverse=True, key=lambda x: x['dt'])

    if count:
        logs = logs[:count]

    if add_preview:
        for log in logs:
            first_lines = list(islice(mds_s3_common_client.get_data(key=log['key']).iter_lines(), 10))
            first_lines_text = '\n'.join([line.decode('utf8') for line in first_lines])
            log['first_lines'] = first_lines_text

    return django_render(request, 'admin/log_list.html', {'logs': logs, 'log_title': log_title})


@login_required
def show_admin_run_log_list(request):
    def retrieve_dt(dt_str):
        return (' '.join(dt_str.split('_')[:2])).replace('.', ':')

    prefix = get_instance_mds_prefix(MDS_ADMIN_RUNS_LOG)
    log_title = 'Список логов от вручную запущенных скриптов'
    return show_run_log_list(request, prefix=prefix, retrieve_dt=retrieve_dt, log_title=log_title, add_preview=True, count=20)


@login_required
def show_cron_run_log_list(request):
    def retrieve_dt(dt_str):
        return ' '.join(dt_str.split('.log')[0].split('_')[-2:])

    prefix = get_instance_mds_prefix(MDS_CRON_RUNS_LOG)
    log_title = 'Список логов скриптов по крону'
    return show_run_log_list(request, prefix=prefix, retrieve_dt=retrieve_dt, log_title=log_title)


@login_required
def maintenance(request):
    """ Админская страничка для управления майнтенансом """
    allowed_scripts_by_flag = defaultdict(list, **{
        JOBS.NO_JOB.flag_value: get_allowed_scripts(),
        JOBS.DEPLOY.flag_value: ['finalize_deploy'],
    })

    show_flags = settings.SERVICE_INSTANCE
    version = u"%s" % settings.PKG_VERSION

    conf = read_conf()

    context = {
        'work_db': switcher.get_db_alias(settings.WORK_DB),
        'service_db': switcher.get_db_alias(settings.SERVICE_DB),
        'cache_tag': conf.get('cache_tag', u'нет'),
        'partial_preparation': maintenance_flags['partial_preparation'],
        'switch_on': maintenance_flags['switch'],
        'show_flags': show_flags,
        'version': version,
        'has_permission': True,
        'dump_sandbox_url': settings.DUMP_SANDBOX_URL
    }

    allowed_scripts = allowed_scripts_by_flag[maintenance_flags['maintenance']]
    context['scripts'] = [(script, get_script_params(script)) for script in allowed_scripts
                          if has_script_permission(request.user, script)]
    for row in context['scripts']:
        if 'select' in row[1]:
            if callable(row[1]['select']):
                row[1]['select_options'] = row[1]['select']()
            elif type(row[1]['select']) in [tuple, list]:
                row[1]['select_options'] = row[1]['select']

    if show_flags:
        if request.POST.get('enable') and not context['switch_on']:
            def enable():
                maintenance_flags['switch'] = True
                log.info(u'Включили переключение')

            run_admin_task({
                'code': 'enable_switching',
                'callable': enable
            }, request, request.user.username)

            return HttpResponseRedirect('/admin/maintenance/')

        elif request.POST.get('disable') and context['switch_on']:
            def disable():
                maintenance_flags['switch'] = False
                log.info(u'Отключили переключение')

            run_admin_task({
                'code': 'disable_switching',
                'callable': disable
            }, request, request.user.username)

            return HttpResponseRedirect('/admin/maintenance/')

        elif request.POST.get('partial_preparation') and not context['partial_preparation']:
            def enable():
                maintenance_flags['partial_preparation'] = True
                log.info(u'Включили частичный пересчет')

            run_admin_task({
                'code': 'enable_partial_preparation',
                'callable': enable
            }, request, request.user.username)

            return HttpResponseRedirect('/admin/maintenance/')

        elif request.POST.get('full_preparation') and context['partial_preparation']:
            def disable():
                maintenance_flags['partial_preparation'] = False
                log.info(u'Отключили частичный пересчет')

            run_admin_task({
                'code': 'disable_partial_preparation',
                'callable': disable
            }, request, request.user.username)

            return HttpResponseRedirect('/admin/maintenance/')

    if request.method == 'POST':
        for key, value in request.POST.items():
            if value == "run":
                to_run = key

                if to_run in allowed_scripts_by_flag[maintenance_flags['maintenance']]:
                    pass
                elif maintenance_flags['maintenance']:
                    messages.add_message(request, messages.ERROR,
                                         u'Нельзя запустить скрипт, какой-то скрипт уже в работе')
                    return HttpResponseRedirect('/admin/maintenance/')

                if has_script_permission(request.user, to_run):
                    if request.META.get('gunicorn.socket'):
                        fobj_to_close = [request.META['gunicorn.socket']]
                    else:
                        fobj_to_close = []

                    message = run_script(to_run, request, username=request.user.username,
                                         fobj_to_close=fobj_to_close)

                    messages.add_message(request, messages.SUCCESS, message)

                    return HttpResponseRedirect('/admin/maintenance/')

                else:
                    messages.add_message(request, messages.ERROR, u"Не хватает прав запускать скрипт %s" % to_run)
                    break

    return django_render(request, 'admin/maintenance.html', context)


@login_required
@permission_required("www.add_rthread")
def copy_thread(request):
    thread_id = request.POST.get('thread_id')
    try:
        thread = RThread.objects.get(id=thread_id)
    except RThread.DoesNotExist:
        return HttpResponseRedirect('/admin/www/rthread/%s' % thread_id)

    thread = thread.copy()
    messages.add_message(request, messages.SUCCESS, u"Нитка скопирована")
    return HttpResponseRedirect('/admin/www/rthread/%d' % thread.id)


class UploadZipForm(forms.Form):
    file_encoding = 'cp1251'

    zip_file = forms.FileField(required=True)

    def clean_zip_file(self):
        if 'zip_file' not in self.cleaned_data:
            return self.cleaned_data

        try:
            zip_file = zipfile.ZipFile(self.cleaned_data['zip_file'])

            names = zip_file.namelist()

            detector = UniversalDetector()

            not_unicode_file_count = 0
            for name in names:
                if not isinstance(name, unicode):
                    detector.feed(name)
                    not_unicode_file_count += 1

            detector.close()

            if not_unicode_file_count and detector.result['confidence'] < 0.5:
                raise forms.ValidationError(u"Непонятная кодировка имен файлов")

            name_encoding = detector.result['encoding']

            files = {}

            for name in names:
                unicode_name = name if isinstance(name, unicode) else name.decode(name_encoding)

                if self.file_encoding != 'bin':
                    try:
                        files[unicode_name] = zip_file.read(name).decode(self.file_encoding)
                    except UnicodeDecodeError:
                        raise forms.ValidationError(u'Не смогли декодировать файл %s из %s' %
                                                    (unicode_name, self.file_encoding))

            self.cleaned_data['files'] = files

            if self.file_encoding != 'bin' and not files:
                raise forms.ValidationError(u"Архив не содержит файлов")

            return self.cleaned_data['zip_file']

        except zipfile.error:
            raise forms.ValidationError(u"Неправильный zip файл")


class LoadAFScheduleZipFileForm(UploadZipForm):
    region = importinfo_admin.admin.site._registry[AFScheduleFile]\
        .formfield_for_dbfield(AFScheduleFile._meta.get_field('region'), None)


@login_required
@permission_required("www.add_afschedulefile")
def load_zip_file(request):
    if request.method == "POST":
        form = LoadAFScheduleZipFileForm(request.POST, request.FILES)
        if form.is_valid():
            region = form.cleaned_data['region']
            for fname, fdata in form.cleaned_data['files'].items():
                AFScheduleFile(region=region, schedule_file_name=fname,
                               schedule_file=fdata).save(force_insert=True)
            messages.add_message(request, messages.SUCCESS, u"Архив с расписаниями загружен")
            return HttpResponseRedirect("/admin/importinfo/afschedulefile/")
    else:
        form = LoadAFScheduleZipFileForm()
    context = {'form': form,
               'title': u"Добавить zip файл с расписаниями",
               'opts': AFScheduleFile._meta}
    return django_render(request, 'admin/www/load_zip_file_form.html', context)


class LoadAFSubwayZipFileForm(UploadZipForm):
    settlement = site._registry[AFSubwayFile].formfield_for_dbfield(AFSubwayFile._meta.get_field('settlement'), None)
    add_mode = AFSubwayFile._meta.get_field('add_mode').formfield(widget=CheckboxInput())

    def clean_zip_file(self):
        UploadZipForm.clean_zip_file(self)

        zip_contents = self.cleaned_data['files']

        if len(zip_contents) != 3:
            raise forms.ValidationError(u'В архиве должно быть три файла')

        for f in ['stations.csv', 'transfers.csv', 'threads.xml']:
            if f not in zip_contents:
                raise forms.ValidationError(u'В архиве не содержится файла %s' % f)


@login_required
@permission_required("www.add_afsubwayfile")
def load_subway_zip_file(request):
    if request.method == 'POST':
        form = LoadAFSubwayZipFileForm(request.POST, request.FILES)

        if form.is_valid():
            settlement = form.cleaned_data['settlement']

            stations = form.cleaned_data['files']['stations.csv']
            passages = form.cleaned_data['files']['transfers.csv']
            threads = form.cleaned_data['files']['threads.xml']

            af = AFSubwayFile(subway_file_name=request.FILES['zip_file'].name,
                              stations_file=stations,
                              stations_file_name='stations.csv',
                              passages_file=passages,
                              passages_file_name='transfers.csv',
                              threads_file=threads,
                              threads_file_name='threads.xml',
                              settlement=settlement,
                              add_mode=form.cleaned_data['add_mode'])
            af.save()
            return HttpResponseRedirect("/admin/importinfo/afsubwayfile/%d/" % af.pk)
    else:
        form = LoadAFSubwayZipFileForm()

    context = {'form': form,
               'title': u"Добавить zip файл с расписаниями",
               'opts': AFSubwayFile._meta}

    return django_render(request, 'admin/www/load_zip_file_form.html', context)


@login_required
@permission_required("www.change_externaldirection")
def load_schemas_zip_file(request):
    if request.method == "POST":
        form = UploadZipForm(request.POST, request.FILES)

        if form.is_valid():
            missing = []

            code_re = re.compile(ur'([\w-]+)\.', re.U)

            for name, data in form.cleaned_data['files'].items():
                m = code_re.search(name)

                if not m:
                    continue

                code = m.group(1)

                try:
                    direction = ExternalDirection.objects.get(code=code)
                    direction.schema_file = data
                    direction.save()
                except ExternalDirection.DoesNotExist:
                    missing.append(name)

            message = u"Файл со схемами направлений загружен"

            if missing:
                message += u'. Не найдены направления для файлов: %s' % ", ".join(missing)

            messages.add_message(request, messages.SUCCESS, mark_safe(message))

            return HttpResponseRedirect("/admin/www/externaldirection/load_zip_file")
    else:
        form = UploadZipForm()
    context = {'form': form,
               'title': u"Загрузить zip файл со схемами направлений",
               'opts': ExternalDirection._meta}
    return django_render(request, 'admin/www/load_zip_file_form.html', context)


class BinUploadZipForm(UploadZipForm):
    file_encoding = 'bin'


@login_required
@permission_required("www.change_station")
def load_station_schema_images_zip_file(request):
    logged = ''

    if request.method == "POST":
        form = BinUploadZipForm(request.POST, request.FILES)

        if form.is_valid():
            collector = get_collector('')
            collector.setLevel(settings.LOG_LEVEL)

            try:
                zip_file_obj = form.cleaned_data['zip_file']
                zip_file_obj.seek(0)
                files = unpack_zip_file(zip_file_obj)

                esr_code_re = re.compile(r'^\s*(\d+)\s+.*')

                for name, filepath in files.items():
                    log.info(u"Обрабатываем %s", name)
                    name = os.path.basename(name)
                    m = esr_code_re.match(name)

                    if not name.endswith(('.png', '.jpg')):
                        log.error(u"Пропускаем не jpg png файл %s", name)
                        continue

                    if not m:
                        log.info(u"Не нашли esr_code в файле %s", name)
                        continue

                    esr_code = esr_leading_zeros(m.group(1))

                    try:
                        station = Station.get_by_code('esr', esr_code)

                        with File(open(filepath), name=name) as f:
                            station.schema_image = f
                            station.save()
                            log.info(u"Поставили cхему станции %s %s", station.id, station.title)

                    except Station.DoesNotExist:
                        log.error(u"Не нашли станции по коду %s", esr_code)

            except Exception:
                log.exception(u"Ошибка импортирования схем")
            finally:
                log.info(u"Закончили импорт")

                logged = collector.get_collected()

                remove_collector('', collector)

            message = u"Файл со схемами загружен"

            messages.add_message(request, messages.SUCCESS, mark_safe(message))

    else:
        form = BinUploadZipForm()

    context = {'form': form,
               'title': u"Загрузить zip файл с изображениями схем",
               'opts': Station._meta,
               'app_label': Station._meta.app_label,
               'has_change_permission': True,
               'original': u"Загрузить zip файл с изображениями схем",
               'log': logged}

    return django_render(request, 'admin/www/load_zip_file_form.html', context)


@login_required
def recalc_static_pages(request):
    StaticPage.recalc_materialized_paths()

    return HttpResponseRedirect(request.META['HTTP_REFERER'])


@login_required
@permission_required("www.change_externaldirection")
def externaldirection_edit_schema(request, pk):
    direction = get_object_or_404(ExternalDirection.objects, pk=pk)

    if request.method == 'POST':
        direction.parse_schema_json(request.POST['schema'])

        direction.save()

        if '_save' in request.POST:
            return HttpResponseRedirect('../')

    context = {
        'title': u'Просмотр / редактирование схемы направления',
        'original': direction,
        'opts': direction._meta,
        'schema': render_to_string('scheme/scheme.html', {
            'edit': True,
            'schema': direction.schema(edit=True),
        }, request=request),
    }

    return django_render(request, 'admin/www/externaldirection/edit_schema.html', context)


@permission_required("www.change_externaldirection")
@jsonp_response
def externaldirection_stations(request, pk):
    direction = ExternalDirection.objects.get(pk=pk)

    rv = [(s.title, s.get_schedule_url(type_='suburban', direction=direction.code)) for s in direction.stations]

    return rv


class RawIdWidget(forms.TextInput):
    """
    A Widget for displaying ForeignKeys in the "raw_id" interface rather than
    in a <select> box.
    """
    def __init__(self, model, attrs=None):
        self.model = model
        super(RawIdWidget, self).__init__(attrs)

    def render(self, name, value, attrs=None, renderer=None):
        if attrs is None:
            attrs = {}
        related_url = '/admin/{}/{}/'.format(
            self.model._meta.app_label, self.model._meta.object_name.lower()
        )
        params = self.url_parameters()
        if params:
            url = '?' + '&amp;'.join(['%s=%s' % (k, v) for k, v in params.items()])
        else:
            url = ''
        attrs.setdefault('class', 'vForeignKeyRawIdAdminField')  # The JavaScript looks for this hook.
        output = [super(RawIdWidget, self).render(name, value, attrs=attrs, renderer=renderer)]
        # TODO: "id_" is hard-coded here. This should instead use the correct
        # API to determine the ID dynamically.
        output.append('<a href="{}{}" class="related-lookup" id="lookup_id_{}"'
                      ' onclick="return showRelatedObjectLookupPopup(this);"> '
                      .format(related_url, url, name))
        output.append('<img src="{}img/admin/search.svg" width="16" height="16" alt="{}" /></a>'
                      .format(settings.ADMIN_MEDIA_PREFIX, 'Lookup'))
        if value:
            output.append(self.label_for_value(value))
        return mark_safe(u''.join(output))

    def base_url_parameters(self):
        params = {}
        return params

    def url_parameters(self):
        from django.contrib.admin.views.main import TO_FIELD_VAR
        params = self.base_url_parameters()
        params.update({TO_FIELD_VAR: "pk"})
        return params

    def label_for_value(self, value):
        key = "pk"
        obj = self.model._default_manager.get(**{key: value})
        return '&nbsp;<strong>%s</strong>' % escape(Truncator(obj).words(14))


class ScheduleLeafsetForm(forms.Form):
    external_direction = forms.ModelChoiceField(queryset=ExternalDirection.objects.all(),
                                                required=False)
    station1 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station2 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station3 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station4 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station5 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    station6 = forms.ModelChoiceField(queryset=Station.objects.all(),
                                      required=False, widget=RawIdWidget(Station))
    date1 = forms.DateField(required=False, widget=AdminDateWidget)
    date2 = forms.DateField(required=False, widget=AdminDateWidget)
    date3 = forms.DateField(required=False, widget=AdminDateWidget)
    date4 = forms.DateField(required=False, widget=AdminDateWidget)
    date5 = forms.DateField(required=False, widget=AdminDateWidget)
    date6 = forms.DateField(required=False, widget=AdminDateWidget)

    def clean(self):
        stations = []
        dates = []
        for n in range(1, 7):
            if self.cleaned_data.get('station%s' % n):
                stations.append(self.cleaned_data.get('station%s' % n))
        for n in range(1, 7):
            if self.cleaned_data.get('date%s' % n):
                dates.append(self.cleaned_data.get('date%s' % n))
        if not dates:
            raise forms.ValidationError(u"Нужно выбрать хотя бы одну дату")
        if not self.cleaned_data['external_direction'] and not stations:
            raise forms.ValidationError(u"Нужно выбрать хотя бы одну станцию или внешнее направление")
        elif self.cleaned_data['external_direction'] and stations:
            raise forms.ValidationError(u"Нужно выбрирать либо станцию, либо направления")

        self.cleaned_data['dates'] = dates
        self.cleaned_data['stations'] = stations
        return self.cleaned_data


@login_required
def upload_schedule_file(request):
    # Поставщики. Код для формы, название, имя файла
    suppliers = filter(lambda s: s.filename, Supplier.objects.filter(filename__isnull=False))
    full_path = os.path.normpath(settings.SCHEDULE_UPLOAD)

    context = {
        'full_path': full_path,
        'suppliers': suppliers,
        'files': get_files_info(full_path)
    }

    if os.path.exists(full_path):
        if request.FILES:
            supplier = first(lambda s: s.code == request.POST['supplier'], suppliers)
            f = request.FILES['file']
            file_path = os.path.join(full_path, supplier.filename)
            destination = open(file_path, 'wb+')
            for chunk in f.chunks():
                destination.write(chunk)
            destination.close()

            file_wrapper = get_schedule_upload_wrapper(file_path)
            file_wrapper.upload()

            messages.add_message(
                request, messages.SUCCESS,
                mark_safe(u"Загрузили файл расписаний {} поставщика {} {}".format(
                    supplier.filename, supplier.title, supplier.code))
            )
            return HttpResponseRedirect(request.META['PATH_INFO'])
    else:
        context['error'] = u"Каталог {} не существует".format(full_path)

    return django_render(request, 'admin/upload_schedule_file.html', context)


@login_required
@clean_temp
def upload_railway_file(request):
    full_path = os.path.normpath(os.path.join(settings.DATA_PATH, 'upload', 'railway'))

    if not os.path.exists(full_path):
        os.makedirs(full_path)

    context = {
        'full_path': full_path,
        'files': get_files_info(full_path)
    }

    if not request.FILES:
        return django_render(request, 'admin/upload_railway_file.html', context)

    try:
        fp = request.FILES['file']
        filepath = save_file_in_temp(fp)
        unzip_and_rewrite_railway_data(filepath, full_path)
        messages.add_message(request, messages.SUCCESS, mark_safe(u'Загрузили новый слой железных дорог'))
    except Exception, e:
        messages.add_message(request, messages.ERROR,
                             mark_for_escaping(u'При загрузке файла произошла ошибка. {}'.format(unicode(e))))

    return HttpResponseRedirect(request.META['PATH_INFO'])


def get_files_info(full_path):
    if not os.path.exists(full_path):
        return []

    files_info = []
    for filename in os.listdir(full_path):
        if filename.startswith('.'):
            continue

        filepath = os.path.join(full_path, filename)
        if os.path.isdir(filepath):
            continue

        stats = os.stat(filepath)
        files_info.append({
            'name': filename,
            'size': u'{}'.format(stats.st_size),
            'dt': datetime.fromtimestamp(stats.st_mtime).strftime('%Y-%m-%d %H:%M')
        })

    return files_info


def save_file_in_temp(fp):
    filepath = get_tmp_filepath('railway.zip')
    with open(filepath, 'wb+') as destination:
        for chunk in fp.chunks():
            destination.write(chunk)

    return filepath


def unzip_and_rewrite_railway_data(filepath, full_path):
    with open(filepath, 'rb') as fp:
        file_map = unpack_zip_file(fp)

    mif, mid = get_mif_and_mid_files(file_map)

    # remove old files (только если нашлись нужные файлы)
    for root, __, files in os.walk(full_path):
        for name in files:
            os.remove(os.path.join(root, name))

    shutil.move(mif, os.path.join(full_path, 'rw_full.mif'))
    shutil.move(mid, os.path.join(full_path, 'rw_full.mid'))


class FindMifAndMidError(SimpleUnicodeException):
    pass


def get_mif_and_mid_files(file_map):
    mif_candidates = [filename for filename in file_map if filename.endswith('.mif')]
    mid_candidates = [filename for filename in file_map if filename.endswith('.mid')]

    if len(mif_candidates) != 1 or len(mid_candidates) != 1:
        raise FindMifAndMidError(u'В загружаемом архиве должен быть один mif файл и один mid файл.')

    return file_map[mif_candidates[0]], file_map[mid_candidates[0]]


@login_required
@permission_required("www.delete_direction")
@transaction.atomic
def direction_delete_safely(request, pk):
    direction = get_object_or_404(Direction, pk=pk)

    remove_rtstaion_references(direction)

    return HttpResponseRedirect("../delete/")


@login_required
def export_direction(request, modelname, pk):
    direction_model = apps.get_model('www', modelname)
    direction = direction_model.objects.get(pk=pk)
    encoding = str(request.GET.get('encoding', 'utf-8'))

    tree = get_direction_xml_tree(direction, use_esr=request.GET.get('use_esr', False))

    response = HttpResponse()
    response['Content-Type'] = 'text/xml; encoding=%s' % encoding
    response['Content-Disposition'] = 'attachment; filename=%s' % str(Header(direction.code + u'.xml', "utf8"))

    tree.write(response, encoding=encoding, xml_declaration=True, pretty_print=True)

    return response


@login_required
@permission_required("www.change_direction")
def reimport_direction_view(request, modelname, pk):
    try:
        direction_model = apps.get_model('www', modelname)
        direction = direction_model.objects.get(pk=pk)

        direction_file = request.FILES.get('direction_file')
        if direction_file is not None:
            reimport_direction(direction, direction_file)

            message = u"Перезагрузили маркеры направления %s" % direction.title
            messages.add_message(request, messages.SUCCESS, mark_safe(message))

        return HttpResponseRedirect('../')
    except Exception:
        if settings.DEBUG:
            raise
        else:
            return HttpResponse(content=format_exc(), status=200, content_type="text/plain")


@login_required
@permission_required("www.add_direction")
def import_direction_view(request, modelname):
    try:
        direction_model = apps.get_model('www', modelname)

        direction_file = request.FILES.get('direction_file')
        if direction_file is not None:
            direction = import_direction(direction_model, direction_file)

            message = u"Загрузили направление %s" % direction.title
            messages.add_message(request, messages.SUCCESS, mark_safe(message))

        return HttpResponseRedirect('../')
    except Exception:
        if settings.DEBUG:
            raise
        else:
            return HttpResponse(content=format_exc(), status=200, content_type="text/plain")


@login_required
@permission_required("www.add_suburbanzone")
def import_suburbanzone_view(request):
    stream = StringIO()
    all_log = logging.getLogger()
    add_stream_handler(all_log, stream)
    try:
        zones_file = request.FILES.get('zones_file')
        if zones_file is not None:
            log.info("Грузим файл с пригородными зонами")
            zones = import_suburbanzones(zones_file, 'test' in request.POST)

            log.info("Загрузили пригородные зоны %s" % u", ".join(z.title for z in zones))

    except Exception:
        if settings.DEBUG:
            raise
        log.exception("Ошибка при заргузке пригородных зон")

    finally:
        content = stream.getvalue()
        remove_stream_handler(all_log, stream)

    return HttpResponse(content=content, status=200, content_type="text/plain; charset=utf-8")


@login_required
def sindbad_stats(request):
    today = date.today()

    context = {'month': today.month,
               'year': today.year}

    if 'month' in request.GET:
        context['month'] = request.GET['month']

    context['month_int'] = int(context['month'])
    context['year_int'] = int(context['year'])

    if 'year' in request.GET:
        context['year'] = request.GET['year']

    context['years'] = range(2011, today.year + 1)
    context['months'] = [[i + 1, m[1]] for i, m in enumerate(russian_dt.MONTH_NAMES)]
    context['month_name'] = russian_dt.MONTH_NAMES[int(context['month']) - 1][0]

    partner = Partner.objects.get(code='sindbad')

    entries = (
        StatisticsEntry.for_month(partner, int(context['year']), int(context['month']))
        .values('day').annotate(orders=Count('id'), sum_price=Sum('price'))
    )

    context['entries'] = list(entries)
    context['total_orders'] = sum([e['orders'] for e in entries])
    context['total_sum'] = sum([e['sum_price'] for e in entries])

    return django_render(request, 'admin/sindbad_stats.html', context)


@login_required
def download_db_file(request, app_label, model_name, pk, field_name):
    model = apps.get_model(app_label, model_name)
    if model and request.user.has_perm(u"%s.change_%s" % (app_label,
                                                          model.__name__.lower())):
        instance = model.objects.get(id=pk)
        memory_file = getattr(instance, field_name)
        response = HttpResponse(memory_file, content_type=memory_file.content_type)

        response['Content-Disposition'] = 'attachment; filename=%s' % str(Header(memory_file.name, 'utf-8'))

        return response
    else:
        HttpResponseRedirect("/admin/")


@login_required
def delete_db_file(request, app_label, model_name, pk, field_name):
    permission = (u"%s.change_%s" % (app_label, model_name)).lower()
    if not request.user.has_perm(permission):
        raise PermissionDenied()

    model = apps.get_model(app_label, model_name)

    instance = model.objects.get(id=pk)
    setattr(instance, field_name, None)
    instance.save()

    return HttpResponseRedirect(
        reverse("admin:%s_%s_change" % (app_label, model_name.lower()),
                args=(instance.id,))
    )


@login_required
@permission_required("www.change_district")
def import_district_view(request, district_id):
    f = request.FILES['district_file']
    district = District.objects.get(id=int(district_id))
    success = import_district(district, f)

    if success:
        messages.info(request, u"Успешно привязались к району %s" % district)
    else:
        messages.error(request, u"Ошибка привязки к району %s" % district)

    return HttpResponseRedirect('../')


@login_required
@permission_required('www.change_trainpseudostationmap')
def load_trainpseudostationmap(request):

    db_log = RelatedLog.get_by_model(TrainPseudoStationMap, 'admin_load')
    error = u""

    if request.method == 'POST':
        if 'file' not in request.FILES:
            error = u"Не указан файл"
        else:
            collector = get_collector('')
            collector.setLevel(settings.LOG_LEVEL)
            try:
                load_trainpseudostationmap__file(request.FILES['file'])
            except Exception:
                log.exception(u"Ошибка импортирования замен")
            finally:
                logged = collector.get_collected()

                db_log.log = logged
                db_log.save()

                remove_collector('', collector)

            return HttpResponseRedirect("")

    context = {'load_log': db_log.log, 'error': error}
    return django_render(request, 'admin/www/trainpseudostationmap/load_map.html', context)


def load_trainpseudostationmap__file(fileobj):
    log.info(u"Загружаем файл %s", fileobj.name)

    reader = UnicodeDictReader(fileobj, ('number', 'pseudo_station_id', 'station_id'),
                               encoding='cp1251', delimiter=';', restkey='rest',
                               strip_values=True)
    for row in reader:
        if not row['number']:
            continue

        log.info(u"Загружаем замену для псевдостанций %s %s %s, %s",
                 row['number'], row['pseudo_station_id'], row['station_id'],
                 u" # ".join(map(unicode, row['rest'] or [])))

        if not TrainPseudoStationMap.objects.filter(number=row['number'], station=row['station_id']):
            try:
                tps_map = TrainPseudoStationMap()
                tps_map.number = row['number']
                tps_map.pseudo_station_id = row['pseudo_station_id']
                tps_map.station_id = row['station_id']
                tps_map.save()
                log.info(u"Замена загружена успешно %s %s %s", tps_map.number, tps_map.pseudo_station.title,
                         tps_map.station.title)
            except Exception:
                log.exception(u"Ошибка создания замены")
        else:
            log.warning(u"Такая замена уже есть в базе")


@jsonp_response
def get_maintenance_status(request):
    now = environment.now()
    admin_type = request.GET.get('admin_type')

    maintenance_state = JOBS.get_current_state()
    warn_state = JOBS.get_next_warn(now)

    status = {'state': False, 'warn_state': False}

    if maintenance_state:
        state_dict = maintenance_state.as_dict()
        state_dict['block'] = admin_type in maintenance_state.block_admins
        status['state'] = state_dict

    if warn_state:
        warn_state_dict = warn_state.as_dict()
        status['warn_state'] = warn_state_dict

        warn_state_dict['remaining_seconds'] = warn_state.get_remaining_seconds(now)

    return status


@login_required
def get_configuration(request):
    database_settings = settings.DATABASES['default']

    data = {
        'caches': settings.CACHES.keys(),
        'replica_info': {
            'master': database_settings.get('HOST'),
            'replicas': database_settings.get('CLUSTER')
        },
    }

    json_data = json.dumps(
        data,
        default=serialize,
        ensure_ascii=False,
        separators=(',', ':'),
        indent=4
    )

    return HttpResponse(json_data, content_type="application/json; charset=utf-8")


def hide_secrets(data):
    if isinstance(data, dict):
        new_data = OrderedDict()
        keys = data.keys() if isinstance(data, OrderedDict) else sorted(data.keys())
        for k in keys:
            if _is_secret_key(k):
                new_data[k] = '****'
            else:
                new_data[k] = hide_secrets(data[k])
        return new_data
    elif isinstance(data, list):
        return [hide_secrets(el) for el in data]
    elif isinstance(data, six.string_types):
        return _hide_url_parts(data)
    else:
        return data


def _is_secret_key(key):
    if isinstance(key, six.string_types):
        return bool(re.match(r'.*(?:token|oauth|password|secret).*', key, re.IGNORECASE))
    else:
        return False


def _hide_url_parts(maybe_url):
    result = urlparse(maybe_url)
    if result.scheme and result.path and result.password:
        netloc = '{}:{}@{}'.format(result.username, '****', result.hostname)
        if result.port:
            netloc += ':{}'.format(result.port)
        result = result._replace(netloc=netloc)
        return urlunparse(result)
    else:
        return maybe_url


@login_required
@user_passes_test(lambda u: u.is_active and u.is_staff and u.is_superuser)
def get_settings(request):
    data = OrderedDict((k, getattr(settings, k)) for k in sorted(dir(settings)) if not k.startswith('_'))

    for k in data:
        try:
            json.dumps(data[k], default=serialize, ensure_ascii=False)
        except Exception:
            data[k] = smart_text(data[k])

    data = hide_secrets(data)

    json_data = json.dumps(
        data,
        default=serialize,
        ensure_ascii=False,
        separators=(',', ':'),
        indent=4,
    )

    return HttpResponse(json_data, content_type="application/json; charset=utf-8")


@login_required
def gtfs_form(request):
    from travel.rasp.admin.lib.gtfs import convert_file

    if request.method == "POST":
        country = Country.objects.get(pk=request.POST.get('country').strip())
        file = request.FILES['file']
        xml = convert_file(file, country.code)
        response = HttpResponse(content_type='application/zip')
        response['Content-Disposition'] = 'filename=%s.zip' % datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
        response.write(xml)
        return response
    else:
        widget = ForeignKeyRawIdWidget(Region._meta.get_field("country").rel, site)
        params = {'country_raw_widget': widget.render('country', None, attrs={'id': 'id_country'})}
        return django_render(request, 'admin/gtfsconverter/form.html', params)


@login_required
def gen_teaser_uid(request):
    from common.apps.info_center.models import Info

    return JsonResponse({'value': str(Info.generate_uuid())})


@login_required
def find_packages_to_reload_to_use_is_base(request):
    from travel.rasp.admin.scripts.find_packages_to_reload_to_use_is_base import run, run_red
    return TemplateResponse(request,
                            'admin/base_station/package_station_list.html',
                            {'packet_base_stations': run(), 'red_packet_base_stations': run_red()})

