# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import operator

import json
from django import forms
from django.contrib.auth.decorators import login_required
from django.core.exceptions import ValidationError
from django.core.serializers.json import DjangoJSONEncoder
from django.http import JsonResponse
from django.template.response import TemplateResponse, HttpResponse
from django.template.context_processors import csrf
from django.views.decorators.http import require_http_methods

from common.dynamic_settings.core import NOTHING
from common.dynamic_settings.default import conf

# нужно импортануть все места регистрации настроек, чтобы их было видно
import common.apps.info_center  # noqa
import common.apps.suburban_events  # noqa
import common.apps.train_order  # noqa
import common.data_api.billing.trust_client  # noqa
import common.data_api.platforms.client  # noqa
import common.data_api.sandbox.generate_geometry_resources  # noqa
import common.data_api.sup.client # noqa
import common.data_api.yandex_bus.api  # noqa
import travel.rasp.admin.scripts.sitemap.sitemaps  # noqa


log = logging.getLogger(__name__)


class JsonField(forms.CharField):
    def validate(self, value):
        pass

    def clean(self, value):
        if value in self.empty_values:
            return None
        if isinstance(value, dict):
            return value
        if isinstance(value, list):
            return value
        try:
            return json.loads(value)
        except ValueError as err:
            raise ValidationError(err.message, code='invalid')

    def prepare_value(self, value):
        if isinstance(value, basestring):
            return value
        if isinstance(value, dict) or isinstance(value, list):
            try:
                return json.dumps(value, ensure_ascii=False, cls=DjangoJSONEncoder)
            except Exception:
                pass
        return value


class SettingsForm(forms.Form):
    FIELD_CLASSES = {
        int: forms.IntegerField,
        float: forms.FloatField,
        bool: forms.BooleanField,
        str: forms.CharField,
        unicode: forms.CharField,
        list: JsonField,
        dict: JsonField,
    }

    def __init__(self, settings, *args, **kwargs):
        super(SettingsForm, self).__init__(*args, **kwargs)
        for setting in settings:
            field_class = self.FIELD_CLASSES.get(setting.converter)
            if field_class:
                field = field_class(
                    label=setting.name,
                    initial=setting.get_from_storage(),
                    required=field_class is not forms.BooleanField and setting.required
                )
                self.fields[setting.name] = field
                self.fields[setting.name].setting = setting
                self.fields[setting.name].default_value = field.prepare_value(setting.default_value)
            else:
                log.warning("Can't display field '{}': unknown type '{}'".format(setting.name, setting.converter))


@login_required
def list_view(request):
    settings = sorted(conf.get_settings().values(), key=operator.attrgetter('name'))
    context = {'form': SettingsForm(settings)}
    context.update(csrf(request))
    return TemplateResponse(request, 'admin/dynamic_settings/list.html', context)


@login_required
@require_http_methods(['POST'])
def save_setting(request):
    changed_field = request.POST['changedField']
    new_field_value = request.POST['newFieldValue']
    form = SettingsForm([conf.get_settings()[changed_field]], {changed_field: new_field_value})
    if form.is_valid():
        value = form.cleaned_data[changed_field]
        with conf.saving_context(user=request.user.username):
            setattr(conf, changed_field, value)

        return HttpResponse('{} saved with value {}'.format(changed_field, value))
    else:
        return HttpResponse(u'<br>'.join(form.errors[changed_field]), status=400)


@login_required
def raw_setting_data(request, setting_name):
    info = conf.get_setting_info(setting_name)
    info['cache_time'] = info.pop('cache_time').total_seconds()

    if info['cached_value'] is NOTHING:
        info.pop('cached_value')

    if info['stored_info'] and info['stored_info'].get('_id'):
        info['stored_info']['_id'] = str(info['stored_info'].get('_id'))

    return JsonResponse(info, json_dumps_params={'ensure_ascii': False})
