# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.http import JsonResponse
from marshmallow import Schema, fields
from rasp_vault.api import get_secret

from travel.rasp.admin.lib.maintenance.flags import set_flag, get_flag, job

log = logging.getLogger(__name__)

FLAG_NAME = 'maintenance'


def get_maintenance_flag_value(role=None):
    try:
        return get_flag(FLAG_NAME, role)
    except Exception as ex:
        message = 'get_flag failed: {}'.format(repr(ex))
        log.error(message)
        return message


def set_maintenance_flag_value(value, role=None):
    value_before = get_maintenance_flag_value(role)

    if value_before == job.NO_JOB.flag_value or value == job.NO_JOB.flag_value:
        try:
            set_flag(FLAG_NAME, value, role)
            status = 'ok'
        except Exception as ex:
            message = 'set_flag failed: {}'.format(repr(ex))
            log.error(message)
            status = message
    else:
        status = 'maintenance in process'

    value_after = get_maintenance_flag_value(role)

    return {
        'status': status,
        'old_value': value_before,
        'value': value_after
    }


class FlagsSchema(Schema):
    value = fields.Int(required=True)
    role = fields.String(missing=None)


def local_get_maintenance_flag(request):
    query, errors = FlagsSchema().load(request.GET)
    return JsonResponse({'value': get_maintenance_flag_value(query['role'])})


def local_set_maintenance_flag(request):
    query, errors = FlagsSchema().load(request.GET)
    return JsonResponse(set_maintenance_flag_value(query['value'], query['role']))


def local_unset_maintenance_flag(request):
    query, errors = FlagsSchema().load(request.GET)
    return JsonResponse(set_maintenance_flag_value(job.NO_JOB.flag_value, query['role']))


get_maintenance_flag = login_required(local_get_maintenance_flag)
set_maintenance_flag = login_required(local_set_maintenance_flag)
unset_maintenance_flag = login_required(local_unset_maintenance_flag)


def set_maintenance_flag_deploy(request):
    request_secret = request.META.get('HTTP_ADMIN_DEPLOY_SECRET')
    applied_config = getattr(settings, 'APPLIED_CONFIG', None)
    yav_secret = get_secret('rasp-admin-{}.admin_deploy_secret'.format(applied_config))

    if request_secret and request_secret == yav_secret:
        query, errors = FlagsSchema().load(request.GET)
        return JsonResponse(set_maintenance_flag_value(query['value'], query['role']))

    return JsonResponse({'error': 'bad request'}, status=400)
