# -*- coding: utf-8 -*-

from __future__ import absolute_import

import logging
import json

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.db import transaction
from django.utils.translation import gettext

from travel.rasp.admin.admin.www_stations.convertors import json_www_station
from common.utils.geo import center_span_zoom, clip
from common.utils.httpresponses import jsonp_response
from common.models.geo import Station, StationMajority, StationCode, Settlement, Country, Region
from common.models.schedule import RThread, RTStation
from common.models_utils import fetch_related


log = logging.getLogger(__name__)


@login_required
@jsonp_response
def www_stations(request):
    if request.method == 'POST':
        data = json.loads(request.body)
        try:
            station = _www_station_atomic_create(data)
            return json_www_station(station)
        except Exception, e:
            return {'status': 'error', 'message': u'Ошибка %s' % e}

    center, span, zoom = center_span_zoom(request)

    if center is None or span is None or zoom is None:
        return None

    stations = request.GET.getlist('stations')

    t_types = request.GET.getlist('t_types')

    filtered = False

    for t_type in list(t_types):
        filtered = _filter_t_types_by_zoom(request, t_types, zoom, t_type) or filtered

    if filtered and not t_types:
        return {
            'settlements': {},
            'stations': [],
            'gort_stations': None
        }

    # if zoom < 10 or stations:
    if stations:
        stations = Station.objects.exclude(t_type__code='pseudo-gortrans').filter(id__in=stations)
    else:
        stations = Station.objects.exclude(t_type__code='pseudo-gortrans')

        if t_types:
            stations = stations.filter(t_type__code__in=t_types)

            # RASP-13310 Для автобусов разделить масштабы отображения на два уровня, опираясь на типы
            # станций. Автовокзалы и автостанции в одном слое показывать начиная например с масштаба 10
            # (более "обзорного"), а автобусные остановки показывать начиная с масштаба 12
            if 'bus' in t_types and zoom < (_get_t_type_zoom(request, 'bus') + 2):
                stations = stations.exclude(station_type__id=11)

        if filtered and not t_types:
            stations = []

        else:
            stations = clip(
                stations,
                center, span,
                zoom_radius=1.1,
                lon_field='longitude',
                lat_field='latitude'
            )

    if len(stations) >= settings.TOO_MUCH_STATIONS:
        return {
            'settlements': {}
        }

    fetch_related(stations, 'settlement', model=Station)

    result = {
        'settlements': dict([(c.id, {'id': c.id, 'title': c.title})
                             for c in set([s.settlement for s in stations
                                           if s.settlement])]),
        'stations': [json_www_station(s) for s in stations]
    }

    if 'pseudo-gortrans' in t_types and \
            zoom >= _get_t_type_zoom(request, 'pseudo-gortrans'):
        gort_stations = clip(
            Station.objects.filter(t_type__code='pseudo-gortrans').all(),
            center, span,
            zoom_radius=1.1,
            lon_field='longitude',
            lat_field='latitude'
        )

        result['gort_stations'] = [json_www_station(s) for s in gort_stations]

    return result


@login_required
@jsonp_response
def www_stations_one(request, id):
    station = Station.objects.exclude(t_type__code='pseudo-gortrans').get(pk=id)
    data = json.loads(request.body)
    majority_id = data.get('majority_id')
    if majority_id is not None:
        station.majority = StationMajority.objects.get(id=majority_id)

    station_type_id = data.get('station_type_id')
    if station_type_id is not None:
        station.station_type_id = station_type_id

    t_type_id = data.get('t_type_id')
    if t_type_id is not None:
        station.t_type_id = t_type_id

    lat = data.get('lat')
    if lat is not None:
        station.latitude = lat

    lon = data.get('lon')
    if lon is not None:
        station.longitude = lon

    station.save()

    return json_www_station(station)


@login_required
@jsonp_response
def www_station_move(request):
    try:
        station = Station.objects.exclude(t_type__code='pseudo-gortrans').get(pk=request.POST.get('station_id'))
        station.longitude = request.POST.get('lon')
        station.latitude = request.POST.get('lat')

        if station.longitude and station.latitude:
            station.save()

        else:
            return {'status': 'blank coords'}

    except Station.DoesNotExist:
        return {'status': "can't move"}

    return {'status': 'ok'}


@login_required
@jsonp_response
@transaction.atomic
def www_station_join(request):
    try:
        src = Station.objects.exclude(t_type__code='pseudo-gortrans').get(pk=request.POST.get('src_id'))
        dst = Station.objects.exclude(t_type__code='pseudo-gortrans').get(pk=request.POST.get('dst_id'))
    except (Station.DoesNotExist, ValueError, TypeError):
        return {'status': 'bad request'}

    if src.t_type_id != dst.t_type_id:
        return {'status': 'bad request'}

    logmessages = []

    logmessages.append([u'%s\t%s\t%s -> %s', src.pk, dst.pk, src.title, dst.title])

    # все нитки в нашей базе, проходящие через станцию А (независимо от поставщика),
    # пометить на пересчёт
    RThread.objects.filter(rtstation__station=src).all().update(changed=True)

    # все станции ниток (www_rtstation), привязанные к станции А,
    # перепривязать к станции Б
    RTStation.objects.filter(station=src).all().update(station=dst)

    # для красного админа
    from travel.rasp.admin.admin.red.models import MetaRouteStation
    # все соответствия и коды в разных системах, привязанные к станции А,
    # перепривязать к станции Б
    MetaRouteStation.objects.filter(station=src).update(station=dst)

    # для синего админа
    from travel.rasp.admin.importinfo.models.mappings import StationMapping
    for m in StationMapping.objects.filter(station=src).all():
        m.station = dst
        m.save()

    for sc in StationCode.objects.filter(station=src).all():
        # Для каждой станции в одной системе код уникален
        try:
            sc.station = dst
            sc.save()
        except:
            logmessages.append([u'lost code\t%s\t%s\t%s\t%s',
                                sc.system.code, sc.code, src.pk, src.title])

    # станцию А удалить из нашей базы
    src.delete()

    for message in logmessages:
        log.info(*message)

    return {'status': 'ok'}


@login_required
@jsonp_response
def www_station_change_majority(request):
    try:
        station_id = request.POST.get('station_id')
        majority_id = request.POST.get('majority_id')
        station = Station.objects.exclude(t_type__code='pseudo-gortrans').get(pk=station_id)
        station.majority = StationMajority.objects.get(id=majority_id)
        station.save()

    except Exception, e:
        log.error(u'www_station_change_majority', e)
        return {'status': 'bad'}

    return {'status': 'ok'}


def _www_station_atomic_create(data):
    must_keys = ['title', 'lon', 'lat', 't_type_id', 'majority_id']

    values = {}
    for key in must_keys:
        value = data.get(key)
        if isinstance(value, basestring):
            value = value.strip()

        if not value:
            raise Exception(gettext(u'отсутствует значение %s') % key)

        values[key] = value

    try:
        country = Country.objects.get(pk=data.get('country_id'))
    except Country.DoesNotExist:
        country = None

    try:
        region = Region.objects.get(pk=data.get('region_id'))
    except Region.DoesNotExist:
        region = None

    try:
        settlement = Settlement.objects.get(pk=data.get('settlement_id'))
    except Settlement.DoesNotExist:
        settlement = None

    region = settlement and settlement.region or region
    country = (settlement and settlement.country) or \
              (region and region.country) or country

    values['longitude'] = values['lon']
    values['latitude'] = values['lat']
    del values['lon']
    del values['lat']

    www_station = Station.objects.create(
        country=country,
        region=region,
        settlement=settlement,
        **values
    )

    return www_station


def _filter_t_types_by_zoom(request, t_types, zoom, t_type):
    """
    Убирает из типов транспорта, станции которых необходимо получить, типы не отображаемые
    на текущем масштабе карты
    """
    if t_types and zoom < _get_t_type_zoom(request, t_type) and t_type in t_types:
        t_types.remove(t_type)

        return True

    return False


def _get_t_type_zoom(request, t_type):
    """
    Возвращает масштаб карты с которого необходимо отображать остановки t_type транспорта
    """
    return int(float(request.COOKIES.get("map_%s_zoom" % t_type, 10)))
