# coding: utf-8

from __future__ import unicode_literals

from django.contrib import admin
from django.utils.html import format_html
from django.utils.translation import ugettext_lazy as _

from common.apps.train.models import (
    CoachFacility, PlacePriceRules, ServiceClass, TariffInfo, UFSErrorDescription, UFSNewOrderBlackList,
    CoachType, Facility)
from common.models_utils.i18n import L_field
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin


@admin.register(PlacePriceRules)
class PlacePriceRulesAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station_from', 'station_to')
    list_filter = ('rule_name', 'coach_type')
    search_fields = ('station_from__title', 'station_to__title')
    list_display = ('rule_name', 'coach_type', 'place_name', 'place_numbers', 'priority', 'price_percent', 'owner',
                    'train_number', 'station_from', 'station_to', 'departure_period_begin', 'departure_period_end',
                    'sale_period_begin', 'sale_period_end', 'is_deluxe', 'coach_numbers', 'is_dinamic_tariff')
    readonly_fields = ('m2m_schemas',)

    def m2m_schemas(self, obj):
        schemas = list(obj.schemas.all())
        if not schemas:
            return ''

        return ', '.join('<{}>'.format(schema) for schema in schemas)
    m2m_schemas.short_description = 'Список схем'


@admin.register(ServiceClass)
class ServiceClassAdmin(RaspExportModelAdmin):
    list_display = ('code', 'name', 'title_ru', 'description_ru')
    search_fields = ('code', 'name', 'title_ru', 'description_ru')
    filter_horizontal = ('facilities',)


@admin.register(CoachFacility)
class CoachFacilityAdmin(RaspExportModelAdmin):
    list_display = ('code', 'name', 'order', 'title_ru')
    search_fields = ('code', 'name', 'title_ru')


@admin.register(UFSErrorDescription)
class UFSErrorDescriptionAdmin(RaspExportModelAdmin):
    list_display = ('ufs_code', 'name', 'show')
    search_fields = ('name', 'ufs_code', 'ufs_description', 'description_ru')


@admin.register(UFSNewOrderBlackList)
class UFSNewOrderBlackListAdmin(RaspExportModelAdmin):
    list_display = ('number', 'pre_comment')
    search_fields = ('number', 'comment')

    def pre_comment(self, obj):
        return format_html('<pre>{}</pre>', obj.comment)

    pre_comment.short_description = _('Комментарий')


@admin.register(TariffInfo)
class TariffInfoAdmin(RaspExportModelAdmin):
    list_display = ('code', 'title_ru')

    fieldsets = (
        (None, {
            'fields': ('code',) + L_field.admin_fields(TariffInfo, ('title',))
        }),
        (_('Ограничения тарифа'), {
            'fields': ('need_document', 'without_place',
                       'min_age', 'min_age_includes_birthday',
                       'max_age', 'max_age_includes_birthday')
        }),
        (_('Коды УФС'), {
            'fields': ('ufs_request_code', 'ufs_response_codes')
        }),
        (_('Коды ИМ'), {
            'fields': ('im_request_code', 'im_response_codes')
        }),
    )


@admin.register(CoachType)
class CoachTypeAdmin(RaspExportModelAdmin):
    list_display = ('code', 'im_code', 'name', 'order', 'title_ru')
    search_fields = ('code', 'im_code', 'name', 'title_ru')


@admin.register(Facility)
class FacilityAdmin(RaspExportModelAdmin):
    list_display = ('code', 'im_code', 'for_place', 'name', 'order', 'title_ru')
    search_fields = ('code', 'im_code', 'name', 'title_ru')
