# coding: utf-8

import logging

from django.utils.translation import ugettext as _, gettext_noop as N_
from lxml import etree

from common.models.geo import Country
from common.models.schedule import Company
from common.models.transport import TransportType, TransportModel, TransportSubtype
from common.utils.caching import cache_method_result
from travel.rasp.admin.importinfo.models import CompanySynonym
from travel.rasp.admin.importinfo.models.two_stage_import import TSISetting
from travel.rasp.admin.importinfo.models.mappings import CompanyMapping, TransportModelMapping
from travel.rasp.admin.lib.exceptions import FormattingException
from travel.rasp.admin.scripts.schedule.utils import SupplierTransportModel, TransportModelFinder
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError


log = logging.getLogger(__name__)


class CysixConvertError(RaspImportError):
    pass


class CriticalImportError(FormattingException):
    u"""Полностью прерывает импорт, должна перехватываться только на верхнем уровне"""
    pass


class CysixImportSettings(object):
    set_hidden_number = True
    start_date = None
    end_date = None
    use_thread_in_station_code = False

    standard_stop_time = 1

    __company_country = None
    # Чтобы при copy.copy не отваливался debugger
    # debugger отваливается т.к. __getattr__ делает getattr(self.tsi_settings, attr)
    # а при воссоздании объекта tsi_settings не заполнен т.к. copy не вызывает __init__ и где-то внутрях возникает
    # бесконечный цикл.
    tsi_settings = None

    def __init__(self, tsi_setting=None):
        self.tsi_settings = tsi_setting or TSISetting()

    def _get_company_country(self):
        if self.__company_country is None:
            self.__company_country = Country.objects.get(code="RU")

        return self.__company_country

    def _set_company_country(self, country):
        self.__company_country = country

    company_country = property(_get_company_country, _set_company_country)

    def update_company_country(self, value):
        self.country = Country.objects.get(code=value)

    def __getattr__(self, attr):
        return getattr(self.tsi_settings, attr)


class CysixContext(object):
    station_code_system = None
    carrier_code_system = None
    vehicle_code_system = None
    t_type = None
    subtype = None
    timezone = None

    params_and_defauls = {
        'station_code_system': "vendor",
        'carrier_code_system': "vendor",
        'vehicle_code_system': "vendor",
        't_type': None,
        'subtype': None,
        'timezone': 'local',
        # country_code_system, region_code_system, settlement_code_system, language not used

        'schedule_comment': u'',
        'schedule_density': u'',
    }

    def __init__(self):
        for k in self.params_and_defauls:
            setattr(self, k, self.params_and_defauls[k])

    def update_from(self, el):
        for attr, value in el.attrib.items():
            value = value.strip()

            if attr in self.params_and_defauls:
                self.update_context_value(attr, value)

            context_attr = el.tag + '_' + attr
            if context_attr in self.params_and_defauls:
                self.update_context_value(context_attr, value)

    def update_context_value(self, attr, value):
        if hasattr(self, 'update_%s' % attr):
            getattr(self, 'update_%s' % attr)(value)
        else:
            setattr(self, attr, value)

    def update_t_type(self, value):
        if value in ('sea', 'river'):
            self.update_subtype(value)
            value = 'water'
        self.t_type = TransportType.objects.get(code__iexact=value)

    def update_subtype(self, value):
        if not value:
            return

        try:
            self.subtype = TransportSubtype.objects.get(code__iexact=value)
        except TransportSubtype.DoesNotExist:
            log.error(N_(u'Не существует подтипа транспорта "%s"'), value)
            self.subtype = None

    def update_station_code_system(self, value):
        value = value.lower()
        value = 'local' if value == 'temporary_vendor' else value

        self.station_code_system = value

    def update_carrier_code_system(self, value):
        value = value.lower()
        value = 'local' if value == 'temporary_vendor' else value

        self.carrier_code_system = value

    def update_vehicle_code_system(self, value):
        value = value.lower()
        value = 'local' if value == 'temporary_vendor' else value

        self.vehicle_code_system = value


class CysixCompany(object):
    t_type = None
    strange = False
    country = None
    phone = u''
    address = u''
    contact_info = u''
    email = u''
    title = u''
    code = u''
    url = u''
    carrier_code = None
    carrier_code_system = None

    SUPPORTER_ATTRS = ('email', 'contact_info', 'address', 'phone')

    def get_key(self):
        return (self.title, self.code)

    def __hash__(self):
        return hash(self.get_key())

    def __eq__(self, other):
        return other.get_key() == self.get_key()

    def __ne__(self, other):
        return other.get_key() == self.get_key()

    @classmethod
    def create_from_thread_el(cls, thread_el, group_el, factory, context, settings):
        if not thread_el.get('carrier_code', u'').strip():
            return

        group_code = group_el.get('code', u'').strip()

        carrier_code = thread_el.get('carrier_code', u'').strip()

        carrier_code_system = context.carrier_code_system

        ref_code = u'_'.join([group_code, carrier_code_system, carrier_code])

        if carrier_code_system == 'local':
            code = u'_'.join([group_code, carrier_code_system])
        else:
            code = ref_code

        carrier_el = factory.get_data_provider().get_carrier_el_by_ref_code(ref_code)

        if carrier_el is None and carrier_code_system not in CysixCompanyFinder.SUPPORTED_CODE_SYSTEMS:
            raise RaspImportError(_(u'Не нашли в справочнике перевозчика с кодом {} система {}')
                                  .format(carrier_code, carrier_code_system))

        title = carrier_el.get('title', u'').strip() if carrier_el is not None else u''

        if not title and carrier_code_system not in CysixCompanyFinder.SUPPORTED_CODE_SYSTEMS:
            raise RaspImportError(_(u'Не указано название у перевозчика с кодом {} система {}')
                                  .format(carrier_code, carrier_code_system))

        country_code = carrier_el.get('country_code', u'').strip() if carrier_el is not None else u''

        try:
            country = Country.objects.get(code=country_code) if country_code else None

        except (Country.DoesNotExist, Country.MultipleObjectsReturned):
            country = None

        cysix_company = cls()

        cysix_company.title = title
        cysix_company.code = code
        cysix_company.carrier_code = carrier_code
        cysix_company.carrier_code_system = carrier_code_system
        cysix_company.t_type = context.t_type
        cysix_company.url = carrier_el.get('url', u'') if carrier_el is not None else u''
        cysix_company.country = country

        if carrier_el is not None:
            for attr in cls.SUPPORTER_ATTRS:
                setattr(cysix_company, attr, carrier_el.get(attr, u"").strip())

        return cysix_company


class CysixCompanyFinder(object):
    SUPPORTED_CODE_SYSTEMS = ('iata', 'sirena', 'icao')

    def __init__(self, supplier, settings):
        self.supplier = supplier
        self.settings = settings

    COPY_ATTRIBS = ('title', 't_type', 'phone', 'country', 'email', 'contact_info', 'address', 'url')
    DEFAULTS = {
        'title': u'',
        'email': u'',
        'contact_info': u'',
        'address': u'',
        'phone': u'',
        'url': u'',
    }

    @cache_method_result
    def find(self, supplier_company):
        try:
            return self.find_in_mappings(supplier_company)
        except Company.DoesNotExist:
            pass

        try:
            return self.find_by_supported_codes(supplier_company)
        except Company.DoesNotExist:
            pass

        try:
            return self.find_by_title(supplier_company)
        except Company.DoesNotExist:
            pass

        if supplier_company.title:
            company = Company()

            for attr in self.COPY_ATTRIBS:
                setattr(company, attr, getattr(supplier_company, attr, None) or self.DEFAULTS.get(attr))

            if self.settings.mark_new_company_as_strange:
                company.strange = True

            company.save()

            log.info(N_(u'Создали компанию %s %s'), company.id, company.title)

            CompanyMapping.objects.create(supplier=self.supplier, title=supplier_company.title,
                                          code=supplier_company.code, company=company)

            return company
        else:
            log.error(N_(u"Не нашли и не смогли создать компанию без названия code='%s' code_system='%s'"),
                      supplier_company.carrier_code, supplier_company.carrier_code_system)

    def find_in_mappings(self, supplier_company):
        try:
            return CompanyMapping.objects.filter(title=supplier_company.title, code=supplier_company.code,
                                                 supplier=self.supplier)[0].company
        except IndexError:
            raise Company.DoesNotExist

    def find_by_supported_codes(self, supplier_company):
        if supplier_company.carrier_code_system in self.SUPPORTED_CODE_SYSTEMS:
            try:
                company = getattr(self, 'find_by_%s' % supplier_company.carrier_code_system)(supplier_company)

            except Company.DoesNotExist:
                log.error(N_(u"Не нашли компании по %s %s"),
                          supplier_company.carrier_code_system, supplier_company.carrier_code)
                raise Company.DoesNotExist

            except Company.MultipleObjectsReturned:
                log.error(N_(u"Несколько компании с кодом %s %s"),
                          supplier_company.carrier_code_system, supplier_company.carrier_code)
                raise Company.DoesNotExist

            CompanyMapping.objects.create(supplier=self.supplier, title=supplier_company.title,
                                          code=supplier_company.code, company=company)

            return company

        raise Company.DoesNotExist

    def find_by_title(self, supplier_company):
        company = self.get_company_by_title(supplier_company)

        if company:
            CompanyMapping.objects.create(supplier=self.supplier, title=supplier_company.title,
                                          code=supplier_company.code, company=company)

            return company

        raise Company.DoesNotExist

    def find_by_iata(self, supplier_company):
        return Company.objects.get(iata=supplier_company.carrier_code)

    def find_by_sirena(self, supplier_company):
        return Company.objects.get(sirena_id=supplier_company.carrier_code)

    def find_by_icao(self, supplier_company):
        return Company.objects.get(icao=supplier_company.carrier_code)

    def get_company_by_title(self, supplier_company):
        try:
            return CompanySynonym.objects.get(title=supplier_company.title, t_type=supplier_company.t_type).company
        except CompanySynonym.DoesNotExist:
            pass

        try:
            return Company.objects.filter(title=supplier_company.title, t_type=supplier_company.t_type)[0]
        except IndexError:
            pass


class CysixTransportModel(SupplierTransportModel):
    @classmethod
    def create_from_thread_el(cls, thread_el, group_el, factory, context, settings):
        if not thread_el.get('vehicle_code', u"").strip():
            return

        group_code = group_el.get('code', u"").strip()

        vehicle_code = thread_el.get('vehicle_code', u"").strip()

        code_system = context.vehicle_code_system

        ref_code = u"_".join([group_code, code_system, vehicle_code])

        if code_system == 'local':
            code = u"_".join([group_code, code_system])
        else:
            code = ref_code

        vehicle_el = factory.get_data_provider().get_vehicle_el_by_ref_code(ref_code)

        if vehicle_el is None and code_system not in CysixTransportModelFinder.SUPPORTED_CODE_SYSTEMS:
            raise RaspImportError(_(u"Не нашли в справочнике модели транспорта с кодом {} система {}")
                                  .format(vehicle_code, code_system))

        title = vehicle_el.get('title', u"").strip() if vehicle_el is not None else u""

        if not title and code_system not in CysixTransportModelFinder.SUPPORTED_CODE_SYSTEMS:
            raise RaspImportError(_(u"Не указано название у модели транспорта с кодом {} система {}")
                                  .format(vehicle_code, code_system))

        cysix_transport_model = cls()

        cysix_transport_model.title = title
        cysix_transport_model.t_type = context.t_type
        cysix_transport_model.code = code
        cysix_transport_model.vehicle_code = vehicle_code
        cysix_transport_model.vehicle_code_system = code_system

        real_title = vehicle_el.get('recommended_title', u"").strip() if vehicle_el is not None else u""
        if real_title:
            cysix_transport_model.real_title = real_title

        return cysix_transport_model


class CysixTransportModelFinder(TransportModelFinder):
    SUPPORTED_CODE_SYSTEMS = ('sirena', 'oag')

    def extra_find(self, supplier_t_model):
        return self.find_by_supported_codes(supplier_t_model)

    def log_not_found(self, supplier_t_model):
        log.error(N_(u"Не нашли и не смогли создать модель транспорта без названия code='%s' code_system='%s'"),
                  supplier_t_model.vehicle_code, supplier_t_model.vehicle_code_system)

    def find_by_supported_codes(self, supplier_t_model):
        if supplier_t_model.vehicle_code_system not in self.SUPPORTED_CODE_SYSTEMS:
            raise TransportModel.DoesNotExist

        try:
            t_model = getattr(self, 'find_by_%s' % supplier_t_model.vehicle_code_system)(supplier_t_model)

        except TransportModel.DoesNotExist:
            log.error(N_(u'Не нашли модели транспорта по %s %s'),
                      supplier_t_model.vehicle_code_system, supplier_t_model.vehicle_code)
            raise TransportModel.DoesNotExist

        except TransportModel.MultipleObjectsReturned:
            log.error(N_(u'Несколько моделей транспорта с кодом %s %s'),
                      supplier_t_model.vehicle_code_system, supplier_t_model.vehiclde_code)
            raise TransportModel.DoesNotExist

        TransportModelMapping.objects.create(supplier=self.supplier, title=supplier_t_model.title,
                                             code=supplier_t_model.code, t_model=t_model)

        return t_model

    def find_by_oag(self, supplier_t_model):
        return TransportModel.objects.get(code_en=supplier_t_model.vehicle_code)

    def find_by_sirena(self, supplier_t_model):
        return TransportModel.objects.get(code=supplier_t_model.vehicle_code)


def get_xml_safe_parser():
    return etree.XMLParser(remove_blank_text=True, resolve_entities=False, load_dtd=False,
                           no_network=True)


def safe_parse_xml(filepath):
    return etree.parse(filepath, parser=get_xml_safe_parser())
