# -*- coding: utf-8 -*-

import logging
import json
from copy import deepcopy

from django.core.exceptions import ValidationError


log = logging.getLogger(__name__)


class FilterParameters(object):
    NEEDED_KEYS_IN_PARAM = (u"title", u"code", u"type", u"value")

    def __init__(self, string):
        self.__parameters = []

        self.__string = string

        try:
            self.__parameters = json.loads(string)

        except ValueError as e:
            self.__parameters = []
            self._report_error(e.message)

        self.check_parameters()

    def __unicode__(self):
        return self.get_string()

    @classmethod
    def _value_to_type(cls, value, type_):
        if type_ == u'int':
            return int(value)

        elif type_ == u'float':
            return float(value)

        elif type_ == u'string':
            return unicode(value)

        elif type_ == u'bool':
            return bool(value)

        elif type_ == u'select':
            v = dict(value)['selected']
            return unicode(v)

        elif type_ in [u'begin_block', u'end_block']:
            return None

        raise NotImplementedError()

    def _report_error(self, err_msg=''):
        raise ValueError('Incorrect parameters string. ' + err_msg)

    def check_parameters(self):
        if not isinstance(self.__parameters, list):
            self.__parameters = []
            self._report_error()

        for p in self.__parameters:
            # Все нужные ключи и только они
            if not isinstance(p, dict) or set(p.keys()) != set(self.NEEDED_KEYS_IN_PARAM):
                self.__parameters = []
                self._report_error()

                return

            # Все значения, кроме value - непустые
            keys = [key for key in self.NEEDED_KEYS_IN_PARAM if key != u'value']
            for key in keys:
                if not p[key]:
                    self.__parameters = []
                    self._report_error()

                    return

            # тип поддерживается
            try:
                FilterParameters._value_to_type(p[u'value'], p[u'type'])

            except ValueError:
                # значения проверяются отдельно validate_values()
                pass

            except NotImplemented:
                self.__parameters = []
                self._report_error()

                return

    def validate_values(self):
        """
        Валидация значений сделана отдельно для того,
        чтобы можно было собирать значения из всех полей формы,
        и только потом отдельно делать валидацию.
        Иначе не будет работать валидация формы.
        """

        for parameter in self.__parameters:
            # значения должны быть нужного типа
            try:
                if parameter[u'type'] == u'select':
                    value = dict(parameter[u'value'])

                    if set(value.keys()) != set([u'all', u'selected']):
                        raise ValueError

                    possible_values = list(value[u'all'])
                    if isinstance(possible_values[0], (list, tuple)):
                        possible_values = [v for v, d in possible_values]

                    if not unicode(value[u'selected']) in possible_values:
                        raise ValueError

                else:
                    parameter[u'value'] = FilterParameters._value_to_type(
                        parameter[u'value'], parameter[u'type']
                    )

            except ValueError:
                self.__parameters = []
                msg = u'Неправильное значение параметера "%s". Должен быть тип "%s".' % (
                    parameter[u'title'], parameter[u'type']
                )

                raise ValidationError(msg)

    def set_parameters(self, parameters):
        self.__parameters = parameters
        self.check_parameters()

    def get_parameters(self):
        return self.__parameters

    def get_parameters_as_dict(self):
        params = dict()

        for p in self.__parameters:
            key = p[u'code']
            value = FilterParameters._value_to_type(p[u'value'], p[u'type'])
            params[key] = value

        return params

    def get_string(self):
        return unicode(json.dumps(self.__parameters, ensure_ascii=False, encoding='utf8', indent=4))

    def find_index_by_code(self, code):
        for index, p in enumerate(self.__parameters):
            if p['code'] == code:
                return index

        return None

    def insert_parameter(self, index, parameter):
        self.__parameters.insert(index, parameter)
        self.check_parameters()
        self.validate_values()

    def insert_parameter_before(self, code, parameter):
        index = self.find_index_by_code(code)

        if index is None:
            index = 0

        self.insert_parameter(index, parameter)

    def insert_parameter_after(self, code, parameter):
        index = self.find_index_by_code(code)

        if index is None:
            index = len(self.__parameters)

        self.insert_parameter(index + 1, parameter)

    def add_parameter(self, parameter):
        self.insert_parameter(len(self.__parameters), parameter)

    def update_parameter_value(self, code, value):
        parameter = self.get_parameter_by_code(code)

        if parameter['type'] == 'select':
            parameter['value']['selected'] = value

        else:
            parameter['value'] = value

        self.update_parameter(parameter)

    def update_parameter(self, parameter):
        index = self.find_index_by_code(parameter['code'])

        self.__parameters[index] = parameter

        self.check_parameters()
        self.validate_values()

    def delete_parameter(self, code):
        index = self.find_index_by_code(code)

        del self.__parameters[index]

        self.check_parameters()
        self.validate_values()

    def get_parameter_by_code(self, code):
        for p in self.__parameters:
            if p['code'] == code:
                return p

        return None

    def move_parameter_before(self, code, code_to_move):
        parameter_to_move = self.get_parameter_by_code(code_to_move)

        if parameter_to_move is None:
            return

        parameter_to_move = deepcopy(parameter_to_move)

        self.delete_parameter(code_to_move)
        self.insert_parameter_before(code, parameter_to_move)

    def move_parameter_after(self, code, code_to_move):
        parameter_to_move = self.get_parameter_by_code(code_to_move)

        if parameter_to_move is None:
            return

        parameter_to_move = deepcopy(parameter_to_move)

        self.delete_parameter(code_to_move)
        self.insert_parameter_after(code, parameter_to_move)

    def delete_parameters_with_types(self, types):
        new_parameters = []

        for p in self.__parameters:
            if p['type'] not in types:
                new_parameters.append(p)

        self.__parameters = new_parameters

        self.check_parameters()
        self.validate_values()


def create_select_parameter(code, title, variants, selected):
    return {
        'code': code,
        'type': 'select',
        'value': create_select_value(variants, selected),
        'title': title
    }


def create_select_value(variants, selected):
    return {
        'all': variants,
        'selected': selected
    }
