# coding: utf-8

import logging

from django.utils.translation import gettext_noop as N_

from travel.rasp.admin.timecorrection.models import PermanentPathData
from travel.rasp.admin.timecorrection.path_checkers import GeometryCheck


log = logging.getLogger(__name__)


class Filter(object):
    """
    Проверка геометрии пути.
    Если геометрия невалидна то xml_thread устанавливается флаг is_thread_hidden

    Проверяется наличие предустановленного значения в базе для данной комбинации координат маршрута.
    Если такого значения нет, то оно создается.
    """

    def __init__(self, params):
        self.params = params

    def apply(self, xml_thread):
        if xml_thread.is_thread_hidden:
            log.info(N_(u'Пропускаем проверку геометрии. Флаг is_thread_hidden уже установлен.'))

        elif self.params:
            return self.check_path_geometry(xml_thread)

        return xml_thread

    def check_path_geometry(self, xml_thread):
        log.info(N_(u'Проверяем геометрию пути {}'.format(xml_thread)))

        stations = [rts.station for rts in xml_thread.supplier_rtstations]
        is_path_correct = GeometryCheck.is_path_correct(stations)

        permanent_path_data = PermanentPathData.get_or_create_from_rtstations(
            rtstations=xml_thread.supplier_rtstations,
            defaults={
                'path_correct': is_path_correct
            }, update_data=self.params['update_data'])

        if is_path_correct != permanent_path_data.path_correct:
            log.info(N_(u'Предустановленно значение отличающееся от расчетного. '
                        u'В БД path_correct=={db_value}'.format(db_value=permanent_path_data.path_correct)))

        if not permanent_path_data.path_correct:
            log.info(N_(u'Геометрия пути невалидна. Устанавливаем is_thread_hidden = True'))
            xml_thread.is_thread_hidden = True

        return xml_thread
