# -*- coding: utf-8 -*-

import logging

from django.utils.translation import ugettext as _, gettext_noop as N_

from common.models.geo import StationMajority
from cysix.filters.core.base import SkipOnRaspImportErrorFilter
from travel.rasp.admin.scripts.schedule.utils import get_supplier_station_path_key, RaspImportError


log = logging.getLogger(__name__)


class SortSupplierRTStationsByDistance(SkipOnRaspImportErrorFilter):
    def process_item(self, xml_thread):
        filter_ = xml_thread.factory.get_filter('sort_stoppoints_by_supplier_distance')
        xml_thread.supplier_rtstations = filter_.apply(xml_thread.supplier_rtstations)

        yield xml_thread


class FillCysixXmlThreadPathKey(SkipOnRaspImportErrorFilter):
    def process_item(self, xml_thread):
        xml_thread.path_key = get_supplier_station_path_key(
            rtss.supplier_station
            for rtss in xml_thread.supplier_rtstations
        )

        yield xml_thread


class SkipStationWithoutTimesConditional(SkipOnRaspImportErrorFilter):
    def process_item(self, xml_thread):
        filter_ = xml_thread.factory.get_filter('cysix_xml_thread.skip_station_without_times')
        if filter_ and filter_.params is not None:
            xml_thread = filter_.apply(xml_thread)

        yield xml_thread


class FilterRTStations(SkipOnRaspImportErrorFilter):
    def __init__(self):
        self.exclude_from_path_majority_id = StationMajority.objects.get(code='exclude_from_path').id

    def process_item(self, xml_thread):
        if len(xml_thread.supplier_rtstations) < 2:
            raise RaspImportError(_(u'Не хватает станций для маршрута'))

        self.skip_thread_if_not_mapped_first_or_last_stoppoint(xml_thread.supplier_rtstations)

        new_supplier_rtstations = []
        for supplier_rtstation in xml_thread.supplier_rtstations:
            if self.need_to_skip_not_mapped_station(supplier_rtstation):
                continue

            if self.need_to_skip_station_by_majority(supplier_rtstation):
                continue

            new_supplier_rtstations.append(supplier_rtstation)

        xml_thread.supplier_rtstations = new_supplier_rtstations

        yield xml_thread

    def skip_thread_if_not_mapped_first_or_last_stoppoint(self, supplier_rtstations):
        if not supplier_rtstations[0].station:
            raise RaspImportError(_(u'В нитке {} нет привязки для первой станции.').format(self))

        if not supplier_rtstations[-1].station:
            raise RaspImportError(_(u'В нитке {} нет привязки для последней станции.').format(self))

    def need_to_skip_not_mapped_station(self, supplier_rtstation):
        if not supplier_rtstation.station:
            return True

    def need_to_skip_station_by_majority(self, supplier_rtstation):
        need_to_skip = (supplier_rtstation.station.majority_id == self.exclude_from_path_majority_id)

        if need_to_skip and supplier_rtstation.is_first:
            log.error(N_(u"Первая станция '%s' имеет majority=exclude_from_path."),
                      supplier_rtstation.station.title)

            # не пропускаем первую станцию, даже если ее majority = need_to_skip
            return False

        if need_to_skip and supplier_rtstation.is_last:
            log.error(N_(u"Исключаем последнюю станцию '%s' (majority=exclude_from_path)."),
                      supplier_rtstation.station.title)

        return need_to_skip


class MarkThreadWithInvalidGeometryHidden(SkipOnRaspImportErrorFilter):
    def process_item(self, xml_thread):
        filter_ = xml_thread.factory.get_filter('check_path_geometry')
        xml_thread = filter_.apply(xml_thread)

        yield xml_thread
