# coding: utf-8

import logging

from django.utils.translation import gettext_noop as N_

from travel.rasp.admin.timecorrection.correction import CorrectionFunctions
from travel.rasp.admin.timecorrection.models import PermanentPathData
from travel.rasp.admin.timecorrection.models_data_access import PathSpanProxy
from travel.rasp.admin.timecorrection.path_checkers import GeometryCheck, is_path_approved
from travel.rasp.admin.timecorrection.utils import n_wize


log = logging.getLogger(__name__)


class Filter(object):
    """
    Корректировка времени прибытия и отправления
    """
    def __init__(self, params):
        self.params = params

    def apply(self, rtstations, cysix_rts_parser):
        if self.params is not None:
            log.info(u'Запускаем фильтр корректировки времени по данным карт'.center(70, '='))

            is_correction_available = self.can_correct_path(rtstations, cysix_rts_parser)

            if is_correction_available:

                log_stations(rtstations, N_(u'До коррекции времен отправления и прибытия'))

                self.path_span_list = PathSpanProxy.get_pathspan_list(rtstations)

                rtstations = self.prepare_arrival_and_departure(rtstations, cysix_rts_parser.standard_stop_time)

                rtstations = self.correct_rtstations(rtstations, cysix_rts_parser.standard_stop_time)

                log_stations(rtstations, N_(u'После коррекции времен отправления и прибытия'))

            log.info(u'Отработал фильтр корректировки времени по данным карт'.center(70, '='))

        return rtstations

    def can_correct_path(self, rtstations, cysix_rts_parser):
        """
        Проверка возможности применения корректировки к нитке
        """
        if cysix_rts_parser.thread.hidden:
            log.info(u'Нитка скрыта. Корректировка не производится')
            return

        stations = [rts.station for rts in rtstations]
        if not cysix_rts_parser.two_stage_package.can_use_filter('check_path_geometry'):
            log.info(u'Фильтр check_path_geometry отключен. Проверяем геометрию')
            is_path_correct = GeometryCheck.is_path_correct(stations)
            permanent_path_data = PermanentPathData.get_or_create_from_rtstations(
                rtstations=rtstations,
                defaults={
                    'path_correct': is_path_correct
                })
            if not permanent_path_data.path_correct:
                log.info(u'Геометрия нитки невалидна. Корректировка не производится')
                return

        if not self.params['correct_not_approved_path'] and not is_path_approved(stations):
            log.info(u'Маршрут не геокодирован. Корректировка не производится')
            return

        return True

    def correct_rtstations(self, rtstations, standard_stop_time):
        supplier_duration_list = get_supplier_duration_list(rtstations, standard_stop_time)
        correction_function = CorrectionFunctions.get_correction_function_for_import(rtstations, supplier_duration_list)
        diff_array = list(correction_function(self.path_span_list, supplier_duration_list))

        if not any(diff_array):
            log.info(N_(u'Маршрут не требует корректировки'))
        elif sum(round(abs(diff)) for diff in diff_array) < self.params['min_correction_time']:
            log.info(N_(u'Сумма корректировки меньше {} мин. Маршрут не корректируем').format(
                self.params['min_correction_time']))
        else:
            rtstations = apply_diff_array(diff_array, rtstations)

        return rtstations

    def prepare_arrival_and_departure(self, rtstations, standard_stop_time):
        """
        Подготовка данных нитки к корректировке.
        """
        self.stab_invalid_arrival_and_departure(rtstations, standard_stop_time)

        if self.params['verbose_log']:
            log_stations(rtstations, N_(u'После операции stab'))

        self.share_time_with_invalid_stations(rtstations)

        if self.params['verbose_log']:
            log_stations(rtstations, N_(u'После операции share'))

        return rtstations

    @staticmethod
    def stab_invalid_arrival_and_departure(rtstations, standard_stop_time):
        """
        Замена всех None значений arrival и departure (кроме первой и последней станции) на stab
        Если из пары arrival и departure None только одно, значение устанавливается +/- standard_stop_time
        """
        stab = 0

        # first rts
        rtstations[0].arrival = None
        rtstations[0].departure = 0

        # last rts
        if rtstations[-1].arrival is None:
            log.info(u'не проставлено время прибытия на конечную станцию')
            if rtstations[-1].departure is not None:
                rtstations[-1].arrival = rtstations[-1].departure
                rtstations[-1].departure = None
            else:
                rtstations[-1].arrival = stab

        # any other rts
        for rts in rtstations[1:-1]:
            if rts.arrival is None and rts.departure is None:
                rts.arrival = stab
                rts.departure = stab + standard_stop_time
            elif rts.departure is None:
                rts.departure = rts.arrival + standard_stop_time
            elif rts.arrival is None:
                rts.arrival = rts.departure - standard_stop_time

    def share_time_with_invalid_stations(self, rtstations):
        segment_start_index = 0
        for current_station_index, rts in enumerate(rtstations[1:], 1):
            if rts.arrival:
                if current_station_index - segment_start_index > 1:
                    self.proportional_split_time(rtstations, segment_start_index, current_station_index)

                segment_start_index = current_station_index

    def proportional_split_time(self, rtstations, start, end):
        """Передается rtstations и индексы участка от N до М
                где N,M - Станции у которой заполнены arrival и departure
                N+1...M-1 - Станции у которой отсутствуют arrival и departure
        Время станций от start до end устанавливается пропорционально времени соответствующих участков PathSpan.
        """
        if start == 0:
            start_time = rtstations[start].departure
        else:
            start_time = rtstations[start].arrival

        span_duration = rtstations[end].arrival - start_time

        path_duration_by_map = sum(path_span.duration for path_span in self.path_span_list[start:end])

        for index in xrange(start + 1, end):
            ratio = self.path_span_list[index - 1].duration / path_duration_by_map
            start_time += span_duration * ratio
            rtstations[index].arrival += round(start_time)
            rtstations[index].departure += round(start_time)


def get_supplier_duration_list(rtstations, standard_stop_time):
    supplier_duration_list = []
    for rts_from, rts_to in n_wize(rtstations):
        if rts_from.arrival is None:
            shift_time = rts_to.arrival - rts_from.departure
            stop_time = 0
        else:
            shift_time = rts_to.arrival - rts_from.arrival
            stop_time = rts_from.departure - rts_from.arrival

        if stop_time != standard_stop_time:
            shift_time -= stop_time

        supplier_duration_list.append(shift_time)
    return supplier_duration_list


def apply_diff_array(diff_array, rtstations):
    for rts, diff in zip(rtstations, diff_array):
        diff = round(diff)
        rts.arrival = None if rts.arrival is None else rts.arrival + diff
        rts.departure = None if rts.departure is None else rts.departure + diff
        rts.is_fuzzy = bool(diff)

    return rtstations


def log_stations(rtstations, msg):
    log.info(msg.center(70, '-'))

    for rts in rtstations:
        log.info(u"{station_id:^15} {station_title:^20}: '{arrival}' - '{departure}' fuzzy {fuzzy}".format(
            station_id=rts.station.id, station_title=rts.station.title, arrival=rts.arrival,
            departure=rts.departure, fuzzy=rts.is_fuzzy))
