# -*- coding: utf-8 -*-

import logging
import math
from functools import partial

from django.utils.translation import ugettext as _, gettext_noop as N_

from cysix.filters.distance_calculator import (calculate_geo_distance, calculate_geo_and_combine_distance,
                                               correct_supplier_distance)
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.scripts.schedule.utils import RaspImportError


log = logging.getLogger(__name__)


MINUTES_IN_DAY = 24 * 60


class Filter(object):
    def __init__(self, params):
        self.params = params

    def apply(self, rtstations, standard_stop_time):
        if self.params:
            rtstations = self.fill_can_add_day(rtstations)

            rtstations = self.correct_departure_and_arrival(rtstations, standard_stop_time)

        return rtstations

    def correct_departure_and_arrival(self, rtstations, standard_stop_time):
        if self.params:
            if self.params['do_verbose_logging']:
                self.log_stations(rtstations, _(u'До коррекции времен отправления и прибытия'))

            if self.params['only_add_day']:
                rtstations = self.only_add_day(rtstations, standard_stop_time)

            else:
                self.calculate_distance_if_needed(rtstations)

                self.correct_supplier_distance_if_needed(rtstations)

                correction = DepartureAndArrivalCorrection(self.params)
                rtstations = correction.mark_invalid_rtstations(rtstations, self.params)

                if self.params['do_verbose_logging']:
                    self.log_stations(rtstations, _(u'После проверки времен отправления и прибытия'))

                rtstations = correction.correct_invalid_rtstations(rtstations, self.params,
                                                                   standard_stop_time)
                if self.params['mark_invalid_as_fuzzy']:
                    mark_invalid_as_fuzzy(rtstations)

            if self.params['do_verbose_logging']:
                self.log_stations(rtstations, _(u'После коррекции времен отправления и прибытия'))

        return rtstations

    def calculate_distance_if_needed(self, rtstations):
        if 'by_combine_distance' in (self.params['check'], self.params['correction']):
            calculate_geo_and_combine_distance(rtstations, self.params)

        elif 'by_our_distance' in (self.params['check'], self.params['correction']):
            calculate_geo_distance(rtstations)

    def correct_supplier_distance_if_needed(self, rtstations):
        if 'by_supplier_distance' in (self.params['check'], self.params['correction']):
            correct_supplier_distance(rtstations)

    def log_stations(self, rtstations, msg):
        log.info(u"-------- %s --------", msg)

        combine_distance_str = u''
        has_combine_distance = hasattr(rtstations[-1], 'combine_distance')

        geo_distance_str = u''
        has_geo_distance = hasattr(rtstations[-1], 'geo_distance')

        invalid_str = u''
        has_invalid = hasattr(rtstations[-1], 'invalid')

        for rts in rtstations:
            if has_combine_distance:
                combine_distance_str = u", comb_s:'{}'".format(rts.combine_distance)

            if has_geo_distance:
                geo_distance_str = u", geo_s:'{}'".format(rts.geo_distance)

            if has_invalid:
                invalid_str = u", invalid:'{}'".format(rts.invalid)

            log.info(u"%30s a-d: '%s' - '%s', s: '%s'%s%s%s",
                     str(rts.station.id) + rts.station.title, rts.arrival, rts.departure,
                     rts.distance, geo_distance_str, combine_distance_str,
                     invalid_str)

    def only_add_day(self, rtstations, standard_stop_time):
        # возможна ситуация близко расположенных станций,
        # например 11/12 - 11/12 - 12/13: поэтому
        # если последнее хорошее время отправления, то берем его за вычетом standard_stop_time

        rtstations = self.drop_rtstations_without_times(rtstations)

        last_good_time = (rtstations[0].departure or 0) - standard_stop_time

        for rts in rtstations[1:]:
            # arrival
            if rts.arrival is not None:
                if rts.can_add_day_to_arrival:
                    while rts.arrival < last_good_time:
                        add_one_day_to_arrival(rts)

                if rts.arrival > last_good_time:
                    last_good_time = rts.arrival

            # departure
            if rts.departure is not None:
                if rts.can_add_day_to_departure:
                    while rts.departure < last_good_time:
                        add_one_day_to_departure(rts)

                if rts.departure - standard_stop_time > last_good_time:
                    last_good_time = rts.departure - standard_stop_time

        return rtstations

    def drop_rtstations_without_times(self, rtstations):
        """ Выкидываем станции следования без времен RASPADMIN-537 """

        if not rtstations:
            return rtstations

        new_rtstations = [rtstations[0]]

        for rts in rtstations[1:]:
            if rts.departure is None and rts.arrival is None:
                log.warning(N_(u"Фильтр коррекции времен (только добавлять сутки): "
                               u"выкидываем станцию без времен '%s'"), rts.station.title)

                continue

            new_rtstations.append(rts)

        return new_rtstations

    def fill_can_add_day(self, rtstations):
        if not self.params['trust_day_shift']:
            return self.set_can_add_day_to_true(rtstations)

        for rts in rtstations:
            rts.can_add_day = rts.supplier_rtstation.can_add_day
            rts.can_add_day_to_departure = rts.supplier_rtstation.can_add_day_to_departure
            rts.can_add_day_to_arrival = rts.supplier_rtstation.can_add_day_to_arrival

        return rtstations

    def set_can_add_day_to_true(self, rtstations):
        for rts in rtstations:
            rts.can_add_day = True
            rts.can_add_day_to_arrival = True
            rts.can_add_day_to_departure = True

        return rtstations


class CheckByTime(object):
    def __call__(self, rtstations, params):
        self.t_min = params['t_min']
        self.t_max = params['t_max']
        self.t_super_max = params['t_super_max']

        rtstations[0].invalid = False
        self.last_good_rts = rtstations[0]
        self.last_good = 0

        for i, rts in enumerate(rtstations[1:], 1):
            if rts.arrival is None:
                rts.invalid = True
                continue

            rts_times_backup = RTSTimesBackup(rts)

            self.add_days(rts)

            rts.invalid = self.is_invalid_time(rts, i)

            if rts.invalid:
                rts_times_backup.restore(rts)

            else:
                self.last_good_rts = rts
                self.last_good = i

        return rtstations

    def add_days(self, rts):
        if rts.can_add_day:
            while (rts.arrival - self.last_good_rts.departure) < self.t_min:
                add_one_day_to_rts(rts)

    def is_invalid_time(self, rts, i):
        if rts.arrival > self.t_super_max:
            log.warning(N_(u"Превышено максимально допустимое время всего рейса, станция '%s'."),
                        rts.station.title)
            return True

        duration = rts.arrival - self.last_good_rts.departure
        t_max_now = self.t_max * (i - self.last_good)

        return not self.t_min <= duration <= t_max_now


class CheckByDistance(object):
    def prepare_to_check(self, rtstations, params, distance_func):
        self.rtstations = rtstations
        self.v_min = params['v_min']
        self.v_max = params['v_max']
        self.calc_velocity_from = params['calc_velocity_from']
        self.distance_func = distance_func

        for rts in self.rtstations:
            if self.distance_func(rts) is None:
                raise RaspImportError(_(u"Нет расстояния для станции '{}'").format(rts.station.title))

    def __call__(self, rtstations, params, distance_func):
        self.prepare_to_check(rtstations, params, distance_func)

        self.rtstations[0].invalid = False
        self.last_good_rts = self.rtstations[0]

        for rts in self.rtstations[1:]:
            if rts.arrival is None:
                rts.invalid = True

                continue

            rts_times_backup = RTSTimesBackup(rts)

            self.rts = rts

            self.add_days_by_time()

            self.distance = self.calc_distance_from()

            self.add_days_if_too_fast()

            rts.invalid = self.is_invalid_time()

            if rts.invalid:
                rts_times_backup.restore(rts)

            else:
                self.last_good_rts = rts

        return self.rtstations

    def add_days_by_time(self):
        if self.rts.can_add_day:
            while self.rts.arrival < self.last_good_rts.departure:
                add_one_day_to_rts(self.rts)

    def add_days_if_too_fast(self):
        if self.rts.can_add_day:
            while self.distance > self.max_possible_distance_from():
                add_one_day_to_rts(self.rts)

    def is_invalid_time(self):
        if self.rts.arrival < self.last_good_rts.departure:
            return True

        good = (self.min_possible_distance_from() <= self.distance <= self.max_possible_distance_from())

        return not good

    def max_possible_distance_from(self):
        t = self.calc_time_from() or 1  # Если время в пути 0 минут, то допускаем маленькое расстояние
        return t * self.v_max / 60.

    def min_possible_distance_from(self):
        t = self.calc_time_from()
        return t * self.v_min / 60.

    def calc_distance_from(self):
        if self.calc_velocity_from == 'first':
            distance = self.distance_func(self.rts)

        elif self.calc_velocity_from == 'previous':
            distance = self.distance_func(self.rts) - self.distance_func(self.last_good_rts)

        else:
            raise RaspImportError(_(u"Неизвестное значение '{}' опции 'calc_velocity_from'")
                                  .format(self.calc_velocity_from))

        return distance

    def calc_time_from(self):
        t = None

        if self.calc_velocity_from == 'first':
            t = self.rts.arrival

        elif self.calc_velocity_from == 'previous':
            t = self.rts.arrival - self.last_good_rts.departure

        else:
            raise RaspImportError(_(u"Неизвестное значение '{}' опции 'calc_velocity_from'")
                                  .format(self.calc_velocity_from))

        return t


class CheckByDistanceKeepLast(CheckByDistance):
    """
    Класс проверяет времена для станций нитки, отмечает невалидные.

    Основная особенность - время прибытия на последнюю станцию считается более важным,
    чем времена промежуточных станций.

    Алгоритм проверки:

    1. Сначала проверяется время для последней станции по минимальной и максимальной скорости,
       возможно с прибавлением суток.

    2. Если время последней станции невалидно, - то для проверки валидности используем CheckByDistance

    3. Времена промежуточных станций проверяем по условиям:

        3.1. Время от начала рейса (от последней валидной) до текущей станции должно быть таким,
             чтобы скорость на этом участке укладывалась в интервал от минимальной до максимальной скорости

        3.2. Время от текущей станции до конечной станции должно быть таким,
             чтобы скорость на этом участке укладывалась в интервал от минимальной до максимальной скорости

    Вместо сравнения по скорости в коде используется сравнение по расстоянию, чтобы исключить деление на ноль.
    """
    class CheckByDistanceKeepLastError(SimpleUnicodeException):
        pass

    def prepare_to_check(self, rtstations, params, distance_func):
        super(CheckByDistanceKeepLast, self).prepare_to_check(rtstations, params, distance_func)

        self.first_rts = self.rtstations[0]
        self.last_rts = self.rtstations[-1]

        self.all_distance = self.distance_func(self.last_rts)

        if self.all_distance <= 0:
            raise self.CheckByDistanceKeepLastError(_(u"Неправильное расстояние от первой до последней станции '{}'")
                                                    .format(self.all_distance))

    def __call__(self, rtstations, params, distance_func):
        try:
            self.prepare_to_check(rtstations, params, distance_func)

            self.add_days_and_check_last_station()

            self.calc_and_check_all_time()

        except self.CheckByDistanceKeepLastError, e:
            log.error(N_(u'Используем обычную проверку, т.к. %s'), e)

            return CheckByDistance()(rtstations, params, distance_func)

        for rts in self.rtstations[1:-1]:
            if rts.arrival is None:
                rts.invalid = True

                continue

            rts_times_backup = RTSTimesBackup(rts)

            self.rts = rts

            self.add_days_by_time()

            self.add_days_if_too_fast_middle_rts()

            rts.invalid = self.is_invalid_time_middle_rts()

            if rts.invalid:
                rts_times_backup.restore(rts)

            else:
                self.last_good_rts = rts

        return self.rtstations

    def add_days_and_check_last_station(self):
        if self.last_rts.arrival is None:
            raise self.CheckByDistanceKeepLastError(_(u"Не указано время прибытия для последней станции."))

        self.first_rts.invalid = False

        self.last_good_rts = self.first_rts

        rts_times_backup = RTSTimesBackup(self.last_rts)

        self.rts = self.last_rts
        self.distance = self.calc_distance_from()

        self.add_days_by_time()

        self.add_days_if_too_fast()

        last_rts_invalid = self.is_invalid_time()

        if last_rts_invalid:
            rts_times_backup.restore(self.last_rts)
            raise self.CheckByDistanceKeepLastError(_(u'Указано невалидное время для последней станции.'))

        self.last_rts.invalid = last_rts_invalid

    def is_invalid_time_middle_rts(self):
        if self.rts.arrival < self.last_good_rts.departure:
            return True

        if self.rts.departure > self.last_rts.arrival:
            return True

        good_from = (self.min_possible_distance_from() <= self.calc_distance_from() <=
                     self.max_possible_distance_from())

        good_to_last = (self.min_possible_distance_to_last() <= self.calc_distance_to_last() <=
                        self.max_possible_distance_to_last())

        good = (good_from and good_to_last)

        return not good

    def add_days_if_too_fast_middle_rts(self):
        """
        По логике работы для промежуточных станций проверяем условия:

        (1) min_possible_distance_to_last <= distance_to_last <= max_possible_distance_to_last
        (2) min_possible_distance_from <= distance_from <= max_possible_distance_from

        Если условия не выполняются, то возможно прибавление суток поможет.

        Когда добавление суток может помочь?

        Когда нарушается одно из условий, то есть если
        (1) distance_to_last < min_possible_distance_to_last или distance_to_last > max_possible_distance_to_last
            Понятно, что добавлять сутки имеет смысл только в одном из этих вариантов

            При добавлении суток время до последней уменьшается и
            расчетное расстояние до конечной min(max)_possible_distance_to_last может только уменьшиться

            И значит, добавлять сутки имеет смысл, если distance_to_last < min_possible_distance_to_last

        (2) При добавлении суток время от первой(последней хорошей) увеличивается и
            расчетное расстояние от первой(последней хорошей) может только увеличиться

            И значит, добавлять сутки имеет смысл, если distance_from > max_possible_distance_from
        """

        if self.rts.can_add_day:
            distance_to_last = self.calc_distance_to_last()
            distance_from = self.calc_distance_from()

            while (distance_to_last < self.min_possible_distance_to_last() or
                   distance_from > self.max_possible_distance_from()):
                add_one_day_to_rts(self.rts)

    def calc_and_check_all_time(self):
        self.all_time = self.last_rts.arrival - self.first_rts.departure

        if self.all_time <= 0:
            raise self.CheckByDistanceKeepLastError(_(u"Неправильное время от первой до последней станции '{}'")
                                                    .format(self.all_time))

    def max_possible_distance_to_last(self):
        t = self.calc_time_to_last() or 1  # Если время в пути 0 минут, то допускаем маленькое расстояние
        return t * self.v_max / 60.

    def min_possible_distance_to_last(self):
        t = self.calc_time_to_last()
        return t * self.v_min / 60.

    def calc_distance_to_last(self):
        return self.distance_func(self.last_rts) - self.distance_func(self.rts)

    def calc_time_to_last(self):
        return self.last_rts.arrival - self.rts.departure


class CorrectByTime(object):
    def __call__(self, rtstations, params, standard_stop_time):
        fill_last_stations = params['fill_last_stations']
        fill_internal_stations = params['fill_internal_stations']
        use_on_time_correction = params['use_on_time_correction']

        self.do_not_import_invalid_threads = params['ignore_invalid_threads']
        self.const_t = params['t_const']
        self.standard_stop_time = standard_stop_time

        self.rts_invalid_ranges, self.rts_invalid_tail = get_invalids_inside_and_tail(rtstations, standard_stop_time)

        if fill_last_stations:
            self.correct_tail(rtstations)

        if fill_internal_stations:
            if use_on_time_correction == 'const_time':
                for rts_invalid_range in self.rts_invalid_ranges:
                    self.correct_internal_by_const_time(rtstations, rts_invalid_range)

            elif use_on_time_correction == 'equal_time':
                for rts_invalid_range in self.rts_invalid_ranges:
                    self.correct_internal_by_equal_time(rtstations, rts_invalid_range)

            else:
                raise NotImplementedError()

        return rtstations

    def correct_tail(self, rtstations):
        for i in self.rts_invalid_tail.indexes():
            rtstations[i].arrival = rtstations[i - 1].departure + self.const_t
            rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

    def correct_internal_by_const_time(self, rtstations, rts_invalid_range):
        time_needed = len(rts_invalid_range) * (self.const_t + self.standard_stop_time)

        if time_needed < rts_invalid_range.get_all_time_without_stops():
            for i in rts_invalid_range.indexes():
                rtstations[i].arrival = rtstations[i - 1].departure + self.const_t
                rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

        else:
            self.correct_internal_fallback(rtstations, rts_invalid_range)

    def correct_internal_by_equal_time(self, rtstations, rts_invalid_range):
        all_time_without_stops = rts_invalid_range.get_all_time_without_stops()
        n_of_runs = len(rts_invalid_range) + 1
        time_between = float(all_time_without_stops) / n_of_runs

        if time_between >= 0:
            last_departure = rts_invalid_range.good_before_rts.departure

            for i in rts_invalid_range.indexes():
                duration = last_departure + time_between

                rtstations[i].arrival = int(math.floor(duration))
                rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

                last_departure += time_between + self.standard_stop_time
        else:
            self.correct_internal_fallback(rtstations, rts_invalid_range)

    def correct_internal_fallback(self, rtstations, rts_invalid_range):
        log.info(N_(u"Нехватает времени между '%s' и '%s' для правильной расстановки времен."),
                 rts_invalid_range.good_before_rts.station.title,
                 rts_invalid_range.good_after_rts.station.title)

        if self.do_not_import_invalid_threads:
            raise RaspImportError(_(u"Не получилось скорректировать времена."))

        else:
            n_of_runs = len(rts_invalid_range) + 1
            time_between = float(rts_invalid_range.get_all_time()) / n_of_runs

            last_departure = rts_invalid_range.good_before_rts.departure

            for i in rts_invalid_range.indexes():
                duration = last_departure + time_between

                rtstations[i].arrival = int(math.floor(duration))
                rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

                last_departure += time_between


class CorrectByDistance(object):
    def __call__(self, rtstations, params, standard_stop_time, distance_func):
        fill_last_stations = params['fill_last_stations']
        fill_internal_stations = params['fill_internal_stations']
        use_on_time_correction = params['use_on_distance_correction']

        self.do_not_import_invalid_threads = params['ignore_invalid_threads']
        self.distance_func = distance_func
        self.standard_stop_time = standard_stop_time
        self.avg_speed = params['v_avg']

        for i, rts in enumerate(rtstations):
            if distance_func(rts) is None:
                raise RaspImportError(_(u"Нет расстояния для станции {}").format(rts.station.title))

        self.rts_invalid_ranges, self.rts_invalid_tail = get_invalids_inside_and_tail(rtstations, standard_stop_time)

        if fill_last_stations:
            self.correct_tail(rtstations)

        if fill_internal_stations:
            if use_on_time_correction == 'const_speed':
                for rts_invalid_range in self.rts_invalid_ranges:
                    self.correct_internal_by_const_speed(rtstations, rts_invalid_range)

            elif use_on_time_correction == 'proportional_to_distance':
                for rts_invalid_range in self.rts_invalid_ranges:
                    self.correct_internal_by_proportional_distance(rtstations, rts_invalid_range)

            else:
                raise NotImplementedError()

        return rtstations

    def correct_tail(self, rtstations):
        for i in self.rts_invalid_tail.indexes():
            ds = self.distance_func(rtstations[i]) - self.distance_func(rtstations[i - 1])

            additional_time = float(ds) / self.avg_speed * 60.

            rtstations[i].arrival = rtstations[i - 1].departure + int(math.floor(additional_time))
            rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

    def correct_internal_by_const_speed(self, rtstations, rts_invalid_range):
        all_time = rts_invalid_range.get_all_time()
        all_distance = rts_invalid_range.get_all_distance(self.distance_func)
        summary_stop_time = rts_invalid_range.get_summary_stop_time()

        time_needed = float(all_distance) / self.avg_speed + summary_stop_time

        if time_needed < all_time:
            last_good_distance = self.distance_func(rts_invalid_range.good_before_rts)

            summary_stop_time = 0
            for i in rts_invalid_range.indexes():
                ds = self.distance_func(rtstations[i]) - last_good_distance

                duration = float(ds) / self.avg_speed * 60.
                duration = int(math.floor(duration))

                rtstations[i].arrival = rts_invalid_range.good_before_rts.departure + duration + summary_stop_time
                rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

                summary_stop_time += self.standard_stop_time

        else:
            self.correct_internal_fallback(rtstations, rts_invalid_range)

    def correct_internal_by_proportional_distance(self, rtstations, rts_invalid_range):
        all_time_without_stops = rts_invalid_range.get_all_time_without_stops()

        if all_time_without_stops > 0:
            all_distance = rts_invalid_range.get_all_distance(self.distance_func)

            last_good_distance = self.distance_func(rts_invalid_range.good_before_rts)

            summary_stop_time = 0
            for i in rts_invalid_range.indexes():
                duration = 0
                try:
                    distance = self.distance_func(rtstations[i]) - last_good_distance

                    duration = float(all_time_without_stops) * distance / all_distance
                    duration = int(math.ceil(duration))
                except ZeroDivisionError:
                    pass

                rtstations[i].arrival = rts_invalid_range.good_before_rts.departure + duration + summary_stop_time
                rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time

                summary_stop_time += self.standard_stop_time

        else:
            self.correct_internal_fallback(rtstations, rts_invalid_range)

    def correct_internal_fallback(self, rtstations, rts_invalid_range):
        log.info(N_(u"Нехватает времени между '%s' и '%s' для правильной расстановки времен."),
                 rts_invalid_range.good_before_rts.station.title,
                 rts_invalid_range.good_after_rts.station.title)

        if self.do_not_import_invalid_threads:
            raise RaspImportError(_(u"Не получилось скорректировать времена."))

        else:
            all_time = rts_invalid_range.get_all_time()

            all_distance = rts_invalid_range.get_all_distance(self.distance_func)

            last_good_distance = self.distance_func(rts_invalid_range.good_before_rts)

            for i in rts_invalid_range.indexes():

                duration = 0
                try:
                    distance = self.distance_func(rtstations[i]) - last_good_distance

                    duration = float(all_time) * distance / all_distance
                    duration = int(math.ceil(duration))
                except ZeroDivisionError:
                    pass

                rtstations[i].arrival = rts_invalid_range.good_before_rts.departure + duration
                rtstations[i].departure = rtstations[i].arrival + self.standard_stop_time


def get_invalids_inside_and_tail(rtstations, standard_stop_time):
    invalids_lists = []
    invalids_now = []
    for i, rts in enumerate(rtstations):
        if rts.invalid:
            invalids_now.append(i)

        else:
            if invalids_now:
                invalids_lists.append(invalids_now)
                invalids_now = []

    invalids_tail = invalids_now

    rts_invalid_ranges = []
    for invalids in invalids_lists:
        rts_invalid_ranges.append(RTSInvalidRange(rtstations, invalids[0], invalids[-1], standard_stop_time))

    return rts_invalid_ranges, RTSInvalidTail(rtstations, invalids_tail)


class RTSInvalidRange(object):
    def __init__(self, rtstations, first_index, last_index, standard_stop_time):
        self.rtstations = rtstations
        self.first_index = first_index
        self.last_index = last_index
        self.standard_stop_time = standard_stop_time

        self.good_before = first_index - 1
        self.good_after = last_index + 1

        self.good_before_rts = rtstations[self.good_before]
        self.good_after_rts = rtstations[self.good_after]

    def __len__(self):
        return self.good_after - self.first_index

    def indexes(self):
        return xrange(self.first_index, self.good_after)

    def get_summary_stop_time(self):
        return len(self) * self.standard_stop_time

    def get_all_time(self):
        return self.good_after_rts.arrival - self.good_before_rts.departure

    def get_all_time_without_stops(self):
        return self.get_all_time() - self.get_summary_stop_time()

    def get_all_distance(self, distance_func):
        return distance_func(self.good_after_rts) - distance_func(self.good_before_rts)


class RTSInvalidTail(object):
    def __init__(self, rtstations, indexes):
        self.rtstations = rtstations

        self.first_index = len(self.rtstations)
        if indexes:
            self.first_index = indexes[0]

    def indexes(self):
        return xrange(self.first_index, len(self.rtstations))


def add_one_day_to_rts(rts):
    rts.arrival += MINUTES_IN_DAY
    rts.departure += MINUTES_IN_DAY
    log.info(N_(u"Фильтр коррекции времен: добавляем сутки к временам для станции '%s'"), rts.station.title)


def add_one_day_to_arrival(rts):
    rts.arrival += MINUTES_IN_DAY
    log.info(N_(u"Фильтр коррекции времен: добавляем сутки к времени прибытия для станции '%s'"),
             rts.station.title)


def add_one_day_to_departure(rts):
    rts.departure += MINUTES_IN_DAY
    log.info(N_(u"Фильтр коррекции времен: добавляем сутки к времени отправления для станции '%s'"),
             rts.station.title)


def mark_invalid_as_fuzzy(rtstations):
    for rts in rtstations:
        if hasattr(rts, 'invalid') and rts.invalid:
            rts.is_fuzzy = True


class DepartureAndArrivalCorrection(object):
    CHECKS = {
        'by_time': CheckByTime(),
        'by_supplier_distance': partial(CheckByDistance(), distance_func=lambda x: x.distance),
        'by_our_distance': partial(CheckByDistance(), distance_func=lambda x: x.geo_distance),
        'by_combine_distance': partial(CheckByDistance(), distance_func=lambda x: x.combine_distance),
    }

    CHECKS_KEEP_LAST = {
        'by_time': CheckByTime(),
        'by_supplier_distance': partial(CheckByDistanceKeepLast(), distance_func=lambda x: x.distance),
        'by_our_distance': partial(CheckByDistanceKeepLast(), distance_func=lambda x: x.geo_distance),
        'by_combine_distance': partial(CheckByDistanceKeepLast(), distance_func=lambda x: x.combine_distance),
    }

    CORRECTIONS = {
        'by_time': CorrectByTime(),
        'by_supplier_distance': partial(CorrectByDistance(), distance_func=lambda x: x.distance),
        'by_our_distance': partial(CorrectByDistance(), distance_func=lambda x: x.geo_distance),
        'by_combine_distance': partial(CorrectByDistance(), distance_func=lambda x: x.combine_distance),
    }

    def __init__(self, params):
        self.correct_invalid_rtstations = self.CORRECTIONS.get(params['correction'], correct_nothing)
        self.mark_invalid_rtstations = self.CHECKS.get(params['check'], check_nothing)

        if params['keep_time_for_last_station']:
            self.mark_invalid_rtstations = self.CHECKS_KEEP_LAST.get(params['check'], check_nothing)


def check_nothing(rtstations, params):
    return rtstations


def correct_nothing(rtstations, params, standard_stop_time):
    return rtstations


class RTSTimesBackup(object):
    def __init__(self, rts):
        self.arrival = rts.arrival
        self.departure = rts.departure

    def restore(self, rts):
        rts.arrival = self.arrival
        rts.departure = self.departure
