# -*- coding: utf-8 -*-

import logging
import json

from django.utils.translation import gettext_noop as N_
from lxml import etree

from travel.rasp.admin.www.models.geo import RoutePath


log = logging.getLogger(__name__)


class Filter(object):
    def __init__(self, params):
        self.params = params

    def apply(self, route, thread):
        if self.params:
            write_geometry(route, thread, self.params['replace_existing_geometry'])


def write_geometry(route, thread, replace):
    geometry_el = route.thread_el.find('./geometry')

    if not geometry_el:
        return

    log.info(N_(u'Разбираем геометрию'))

    paths_between_stations = build_paths_between_stations(thread, geometry_el)

    if not paths_between_stations:
        log.warning(N_(u'Не нашли валидной геометрии.'))

        return

    for (station_from, station_to), path in paths_between_stations.iteritems():
        already_exists = bool(RoutePath.get(station_from, station_to))

        if already_exists and not replace:
            log.info(N_(u'Уже есть геометрия между станциями %s и %s. Пропускаем.'),
                     station_from, station_to)

            continue

        path_data = json.dumps(path)
        RoutePath.save_route_path(station_from, station_to, path_data, for_two_directions=False)

        log.info(N_(u'Записали новую геометрию между станциями %s и %s.'),
                 station_from, station_to)


def build_paths_between_stations(thread, geometry_el):
    supplier_station_code_to_station = build_supplier_station_code_to_station(thread)

    paths_between_stations = dict()

    prev_station = None
    points = []

    for point_el in geometry_el.findall('./point'):
        try:
            longitude = float(point_el.get('lon'))
            latitude = float(point_el.get('lat'))

        except (ValueError, TypeError):
            log.error(N_(u'Пропускаем точку. Ошибка разбора координат %s'), etree.tostring(point_el))

            continue

        point = (longitude, latitude)
        points.append(point)

        station_code = point_el.get('station_code')

        if station_code:
            station = supplier_station_code_to_station.get(station_code)

            if not station:
                log.warning(N_(u'Неопознанный код станции "%s" для точки %s'),
                            station_code, etree.tostring(point_el))

                continue

            if not prev_station:
                prev_station = station

                continue

            paths_between_stations[(prev_station, station)] = points

            points = [point]

    return paths_between_stations


def build_supplier_station_code_to_station(thread):
    supplier_station_code_to_station = dict()

    for rtstation in thread.rtstations:
        supplier_station_code = rtstation.supplier_rtstation.supplier_station.station_code
        station = rtstation.supplier_rtstation.station

        supplier_station_code_to_station[supplier_station_code] = station

    return supplier_station_code_to_station
