# -*- coding: utf-8 -*-

import logging

from django.utils.translation import gettext_noop as N_

log = logging.getLogger(__name__)


class Filter(object):
    def __init__(self, params):
        self.params = params

    def apply(self, rtstations):
        if self.params is not None:
            self.do_verbose_logging = self.params['do_verbose_logging']
            rtstations = self.sort_by_supplier_distance(rtstations)

        return rtstations

    def sort_by_supplier_distance(self, rtstations):
        if not rtstations:
            log.error(N_(u"Пропускаем сортировку по расстоянию. Нет станций для сортировки."))

            return rtstations

        for rts in rtstations:
            if rts.distance is None:
                log.error(
                    N_(
                        u"Пропускаем сортировку по расстоянию. "
                        u"Нет расстояния поставщика для станции '%s'"
                    ),
                    rts.supplier_station.title
                )

                return rtstations

        for rts in rtstations[1:]:
            if not rts.distance:
                log.error(
                    N_(
                        u"Пропускаем сортировку по расстоянию. "
                        u"Нулевое расстояние поставщика для станции '%s'"
                    ),
                    rts.supplier_station.title
                )

                return rtstations

        if need_to_sort(rtstations):
            if self.do_verbose_logging:
                log_rtstations(rtstations, N_(u"--------- До сортировки ---------"))

            new_rtstations = sort_rtstations(rtstations)

            if self.do_verbose_logging:
                log_rtstations(new_rtstations, N_(u"--------- После сортировки ---------"))

            return new_rtstations

        return rtstations


def need_to_sort(rtstations):
    prev_rts = rtstations[0]

    for rts in rtstations[1:]:
        if rts.distance < prev_rts.distance:
            log.info(
                N_(
                    u"Нужна сортировка. "
                    u"Расстояние '%s' для станции '%s' "
                    u"меньше расстояния '%s' для станции '%s'."
                ),
                rts.distance,
                rts.supplier_station.title,
                prev_rts.distance,
                prev_rts.supplier_station.title
            )

            return True

        prev_rts = rts

    return False


def sort_rtstations(rtstations):
    stops = [(rts.distance, i, rts) for i, rts in enumerate(rtstations)]

    stops.sort()

    new_rtstations = [stop[2] for stop in stops]

    return new_rtstations


def log_rtstations(rtstations, first_msg):
    log.info(first_msg)

    for rts in rtstations:
        log.info(N_(u"\tРасстояние: '%s', станция: '%s'."), rts.distance, rts.supplier_station.title)
