# -*- coding: utf-8 -*-

import logging
import json
import time as os_time
import heapq
from bisect import bisect_left
from datetime import timedelta, datetime
from itertools import islice, takewhile

from django.utils.translation import gettext_noop as N_
import pytz

from common.utils.date import timedelta2minutes, RunMask
from travel.rasp.library.python.common23.date import environment
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.www.utils import calendar_templates
from travel.rasp.admin.www.utils.calendar_templates import TemplateMatcherException, ThreadDataForExtrapolation


log = logging.getLogger(__name__)


class Filter(object):
    # длина известного расписания. по этому параметру строятся precalc_days_text для ниток и экстраполяции
    known_schedule_length = 10
    use_schedule_plan = False
    templates = 'buses'  # набор шаблонов
    extrapolate = True  # экстраполировать ли расписание
    min_match_days = 7  # какое минимальное количество совпадений должно случиться при сопоставлении шаблонов
    min_shedule_length = 14
    max_mismatch_days = 3  # Максимальное колечество несовпадений с шаблоном
    min_schedule_length_in_range = 7  # Длинна присланного расписания
    search_first_day_in_past_days = 0

    extrapolation_limit_length = 365 - 40

    save = True

    def __init__(self, params, max_forward_days):
        self.extrapolation_limit = None
        self.cysix_template_matcher = None
        self.today = environment.today()

        self.params = params

        self.enabled = bool(self.params)

        if self.params:
            self.known_schedule_length = self.params['known_schedule_length']
            self.templates = self.params['templates']
            self.extrapolate = self.params['extrapolate']
            self.min_match_days = self.params['min_match_days']
            self.max_mismatch_days = self.params['max_mismatch_days']
            self.extrapolation_limit_length = self.params['extrapolation_limit_length']
            self.min_schedule_length_in_range = self.params['min_schedule_length_in_range']
            self.search_first_day_in_past_days = self.params['search_first_day_in_past_days']

            # FIXME: неожиданно, но эти параметры друг от друга зависят
            self.extrapolation_limit_length = min(max_forward_days, self.extrapolation_limit_length)
            self.extrapolation_limit = self.today + timedelta(self.extrapolation_limit_length)

            self._init_matcher()

    def _init_matcher(self):
        start = os_time.time()
        self.cysix_template_matcher = self.prepare_cysix_template_matcher()
        log.info(N_(u"Инициализация фильтра %s"), os_time.time() - start)

    def apply(self, thread, save=True, extrapolate=True):
        if not self.params:
            return

        if thread.year_days == RunMask.EMPTY_YEAR_DAYS:
            log.error(N_(u"Экстраполяция: У нитки %s %s пустая маска дней хождения"), thread.uid, thread.title)
            return

        self.process(thread, save, extrapolate)

    def process(self, thread, save, extrapolate):
        dst_date = get_next_dst_date(thread, self.today)

        try:
            thread_data = build_thread_data_for_extrapolation(thread, dst_date, self.today, self.extrapolation_limit)
        except ExtrapolationError as e:
            log.error(e.msg_template, *e.msg_args)
            return

        if extrapolate and self.should_extrapolate_thread(thread):
            self.extrapolate_thread(thread, thread_data)
        else:
            log.info(N_(u"Экстраполяция: Не экстраполируем нитку %s %s"), thread.uid, thread.title)

        log.info(N_(u'Экстраполяция: Генерируем маску дней хождения для %s %s'), thread.uid, thread.title)
        self.precalc_days_texts(thread, dst_date)

        if save:
            thread.save()

    def should_extrapolate_thread(self, thread):
        """ Нужно ли в процессоре экстраполировать нитку """
        return self.extrapolate and thread.has_extrapolatable_mask

    def extrapolate_thread(self, thread, thread_data):
        """ Произвести экстраполяцию нитки """
        log.info(N_(u'Экстраполяция: Пробуем экстраполировать нитку %s %s'), thread.uid, thread.title)

        __, template, extrapolable = self.find_template(thread_data)
        if not extrapolable:
            log.info(N_(u'Экстраполяция: Не удалось подобрать шаблон для %s %s'), thread.uid, thread.title)
            return

        extrapolated_mask = thread.get_mask(today=self.today) | \
            (RunMask(days=template.days) & RunMask.range(thread_data.extrapolate_from, thread_data.extrapolate_to))

        if extrapolated_mask:
            thread.year_days = str(extrapolated_mask)
            if extrapolated_mask.difference(thread.get_mask(today=self.today)):
                log.info(N_(u'Экстраполяция: Экстраполировали нитку %s %s'), thread.uid, thread.title)
            else:
                log.info(N_(u'Экстраполяция: Маска уже достаточной длинны %s %s'), thread.uid, thread.title)
        else:
            log.error(
                N_(u'Экстраполяция: У шаблона %s не нашлось дней хождений в диапазоне %s %s; %s %s'),
                template.__class__.__name__,
                thread_data.extrapolate_from, thread_data.extrapolate_to,
                thread.uid, thread.title
            )

    def precalc_days_texts(self, thread, dst_date):
        days = get_number_of_days_in_run(thread, self.today)

        days_texts = []
        except_texts = []
        for shift in range(-1, days + 2):  # от -1 до days + 1
            try:
                thread_data = build_thread_data_for_extrapolation(thread, dst_date, self.today,
                                                                  self.extrapolation_limit, shift)
                text, __, __ = self.find_template(thread_data)
            except ExtrapolationError:
                text = None

            days_texts.append(text)
            except_texts.append(None)

        thread.translated_days_texts = json.dumps(days_texts, ensure_ascii=False)
        thread.translated_except_texts = json.dumps(except_texts, ensure_ascii=False)

    def find_template(self, thread_data):
        try:
            text, template, extrapolable = self.cysix_template_matcher.find_template(thread_data)
            return text, template, extrapolable
        except TemplateMatcherException as e:
            log.error(e.msg_template, *e.msg_args)
            return None, None, None

    def prepare_cysix_template_matcher(self):
        template_first_day = self.today - timedelta(30)  # Берем шаблоны начиная с 30 дней назад

        cysix_template_matcher = calendar_templates.get_cysix_template_matcher(
            generator_name=self.templates,
            today=self.today,
            start_date=template_first_day,
            length=365,
            schedule_length=self.known_schedule_length,
            extrapolation_limit=self.extrapolation_limit,
            min_match_days=self.min_match_days,
            max_mismatch_days=self.max_mismatch_days,
        )

        return cysix_template_matcher


class ExtrapolationError(SimpleUnicodeException):
    pass


DAYS_IN_HALF_YEAR = 180


def build_thread_data_for_extrapolation(thread, dst_date, today, extrapolation_limit, shift=0):
    # dst_date нужно вычислить для оригинальной нитки и потом сдвинуть на shift
    dst_date = dst_date + timedelta(days=shift) if dst_date else None
    extrapolation_limit += timedelta(days=shift)

    full_mask = thread.get_mask(today=today).shifted(shift)

    mask_is_extrapolatable = thread.has_extrapolatable_mask

    full_mask_days = full_mask.dates()
    last_date_of_full_mask = full_mask_days[-1] if full_mask_days else None

    try:
        extrapolate_from = (d for d in full_mask_days if d >= today).next()
    except StopIteration:
        raise ExtrapolationError(N_(u'Нитка %s %s не ходит после сегодняшнего дня'), thread.uid, thread.title)
    extrapolate_to = min(extrapolation_limit, dst_date or extrapolation_limit)

    # https://st.yandex-team.ru/EXRASP-5227
    # в связи с тем, что у нас циклический календарь, может возникнуть ситуация,
    # что последний день хождения нитки очень далеко
    # (например, сейчас январь и у нас расписание еще за прошлый год, т.е. последний день хождения - в декабре)
    # поэтому надо от маски отфильтровать такие дни - взять только первые полгода
    last_needed_day = extrapolate_from + timedelta(days=DAYS_IN_HALF_YEAR)
    run_days_in_half_year = list(takewhile(lambda d: d <= last_needed_day, full_mask_days))

    return ThreadDataForExtrapolation(
        run_days_in_half_year=run_days_in_half_year,
        last_date_of_full_mask=last_date_of_full_mask,
        mask_is_extrapolatable=mask_is_extrapolatable,
        dst_date=dst_date,
        extrapolate_from=extrapolate_from,
        extrapolate_to=extrapolate_to
    )


def get_next_dst_date(thread, today):
    mask = thread.get_mask(today=today)
    run_days = mask.dates()
    naive_start_dt = datetime.combine(run_days[-1], thread.tz_start_time)
    start_dt = thread.pytz.localize(naive_start_dt)

    return _get_next_dst_date(start_dt, thread.time_zone, list(thread.path))


def _get_next_dst_date(start_dt, thread_time_zone, thread_path):
    utc_start_dt = start_dt.astimezone(pytz.UTC).replace(tzinfo=None)
    zones = {rts.time_zone for rts in thread_path}
    zones.add(thread_time_zone)

    def _transition_iter(tz, utc_start_dt):
        if not hasattr(tz, '_utc_transition_times'):
            return []

        start_index = bisect_left(tz._utc_transition_times, utc_start_dt)
        return islice(tz._utc_transition_times, start_index, None)

    transitions_iter = heapq.merge(*[
        _transition_iter(pytz.timezone(zone), utc_start_dt)
        for zone in zones if zone
    ])

    for transition_dt in transitions_iter:
        return pytz.UTC.localize(transition_dt).astimezone(pytz.timezone(thread_time_zone)).date()


def get_number_of_days_in_run(thread, today):
    first_run = RunMask.first_run(thread.year_days, today) or today

    naive_start_dt = datetime.combine(first_run, thread.tz_start_time)
    start_dt = thread.pytz.localize(naive_start_dt)
    last_rts = thread.rtstation_set.order_by('-id')[0]
    arrival_dt = last_rts.pytz.localize(naive_start_dt + timedelta(minutes=last_rts.tz_arrival))
    duration = int(timedelta2minutes(arrival_dt - start_dt))

    return (duration + 1440 - 1) / 1440
