# -*- coding: utf-8 -*-

from django import forms
from django.utils.safestring import mark_safe

from cysix.filter_parameters import FilterParameters


class FilterParametersWidget(forms.Widget):
    def __init__(self, attrs=None):
        super(FilterParametersWidget, self).__init__(attrs)

        self.build_html_functions = {
            'begin_block': self.build_html_begin_block,
            'end_block': self.build_html_end_block,
            'bool': self.build_html_bool,
            'select': self.build_html_select,
            'int': self.build_html_common,
            'float': self.build_html_common,
            'string': self.build_html_string,
        }

        self.nested_level = 0

    def value_from_datadict(self, data, files, name):
        parameters = data.get('{}_all'.format(name), u'[]')
        parameters = FilterParameters(parameters)

        for param in parameters.get_parameters():
            if param['type'] in ['begin_block', 'end_block']:
                continue

            param_name = '{}_{}'.format(name, param['code'])
            value = data[param_name] if param_name in data else u''

            if param['type'] == 'select':
                param['value']['selected'] = value
            elif param['type'] == 'bool' and value == 'on':
                param['value'] = True
            else:
                param['value'] = value

        return parameters.get_string()

    def value_omitted_from_data(self, data, files, name):
        return False  # always in data

    def render(self, name, value, attrs=None, renderer=None):
        widget_html = u''

        if value is None:
            return widget_html

        parameters = value

        if isinstance(value, unicode):
            parameters = FilterParameters(value)

        name_all = '{}_all'.format(name)
        id_all = 'id_{}'.format(name)
        widget_html += forms.HiddenInput().render(name_all, parameters.get_string(),
                                                  attrs={'id': id_all}, renderer=renderer)

        self.nested_level = 0

        for param in parameters.get_parameters():
            widget_html += self.build_html_for_param(name, param)

        return mark_safe(widget_html)

    def build_html_for_param(self, name, param, renderer=None):
        widget_name = '{}_{}'.format(name, param['code'])
        widget_id = 'id_{}'.format(widget_name)

        return self.build_html_functions[param['type']](widget_name, param, widget_id, renderer)

    def build_html_begin_block(self, name, param, id_, renderer=None):
        self.nested_level += 1

        nested = (self.nested_level % 2) + 1
        title = mark_safe(unicode(param['title']))

        return u'<fieldset class="nested{}"> <legend>{}</legend>'.format(nested, title)

    def build_html_end_block(self, name, param, id_, renderer=None):
        self.nested_level -= 1

        return u'</fieldset><br />'

    def build_html_bool(self, name, param, id_, renderer=None):
        return (
            mark_safe(unicode(param['title'])) + u"&nbsp;" +
            forms.CheckboxInput().render(name, param['value'], attrs={'id': id_}, renderer=renderer) +
            u'<br />'
        )

    def build_html_select(self, name, param, id_, renderer=None):
        value = param[u'value'][u'selected']

        variants = param[u'value'][u'all']
        if isinstance(variants[0], (list, tuple)):
            choices = param[u'value'][u'all']
        else:
            choices = [(v, v) for v in param[u'value'][u'all']]

        return (
            mark_safe(unicode(param['title'])) + u"&nbsp;" +
            forms.Select(choices=choices).render(name, value, attrs={'id': id_}, renderer=renderer) +
            u'<br />'
        )

    def build_html_common(self, name, param, id_, renderer=None):
        return (
            mark_safe(unicode(param['title'])) + u"&nbsp;" +
            forms.TextInput().render(name, param['value'], attrs={'id': id_}, renderer=renderer) +
            u'<br />'
        )

    def build_html_string(self, name, param, id_, renderer=None):
        return (
            mark_safe(unicode(param['title'])) + u"&nbsp;" +
            forms.Textarea().render(
                name, param['value'], attrs={'id': id_, 'cols': '100', 'rows': '5'}, renderer=renderer
            ) + u'<br />'
        )


class CysixFilterAdminForm(forms.ModelForm):
    class Meta:
        widgets = {
            'parameters': FilterParametersWidget
        }
