# -*- coding: utf-8 -*-
import logging
import os.path
import re
from collections import defaultdict
from datetime import time
from itertools import combinations

from django.conf import settings
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext as _, gettext_noop as N_
from lxml import etree

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, StoppointBlock, ScheduleBlock
from common.models.geo import Country, Region, Settlement
from travel.rasp.admin.importinfo.models.validators import triangle_file_validator
from travel.rasp.admin.lib.import_bounds import import_bounds

from cysix.base import CriticalImportError
from cysix.tsi_converter import CysixTSIConverterFactory, CysixTSIConverterFileProvider
from travel.rasp.admin.importinfo.triangle_import.mask_parser import TriangleMaskTemplateParser
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_method_result
from travel.rasp.admin.lib.encoding import detect_file_encoding_by_path
from travel.rasp.admin.lib.text import unify_minuses
from common.utils.unicode_csv import UnicodeDictReader
from travel.rasp.admin.lib.xls import XlsDictParser
from travel.rasp.admin.scripts.schedule.utils import RaspImportError, TariffMatrix
from travel.rasp.admin.scripts.schedule.utils.file_providers import PackageFileProvider


START_ROUTE_BLOCK = (u'маршрут', u'route', u'Rota'.lower())
END_ROUTE_BLOCK = (u'конец', u'end', u'Bitiş'.lower(), u'Кінець'.lower())
NUMBER_RE = re.compile(
    u'(?:' + (u'|'.join(map(re.escape, START_ROUTE_BLOCK))) + u')' +
    u'\s*(?P<number>.*)',
    re.U + re.I)

FILE_FORMATS = ('csv', 'xls')
TIMETABLE_FIELDS = ('number', 'route_title', 'carrier_code', 'bus_type', 'station_title',
                    'station_code', 'distance')
STATION_FIELDS = ('code', 'title', 'country', 'region', 'region_code', 'settlement', 'settlement_code',
                  'latitude', 'longitude')
CARRIER_FIELDS = ('code', 'title', 'address', 'phone')

ROUTE_INFO_COLUMNS = ('number', 'route_title', 'carrier_code', 'bus_type')
STATION_INFO_COLUMNS = ('station_title', 'station_code', 'distance')


log = logging.getLogger(__name__)


class Triangle2CysixError(RaspImportError):
    pass


class TriangleParseError(RaspImportError):
    pass


class TimeTableParserError(RaspImportError):
    pass


class TriangleCysixFactory(CysixTSIConverterFactory):
    def __init__(self, package):
        super(TriangleCysixFactory, self).__init__(package)

    @cache_method_result
    def get_file_provider(self):
        if not self.package.package_file:
            raise RaspImportError(N_(u'Не нашли приложенный файл для треугольного пакета.'))

        log.info(N_(u'Файл приложен к треугольному пакету берем данные из него'))

        return self.get_converter_file_provider(TriangleDataProvider(self))

    def get_converter_file_provider(self, triangle_file_provider):
        return TriangleCysixFileProvider(self.package, triangle_file_provider)

    @cache_method_result
    def get_triangle_mask_parser(self):
        today = environment.today()

        start, end = import_bounds(today=today, forward=self.max_forward_days)

        return TriangleMaskTemplateParser(start, end, today, country=self.package.country)


class TriangleCysixFileProvider(CysixTSIConverterFileProvider):
    def convert_data(self, filepath):
        c = Converter(self.raw_file_provider)

        c.convert(filepath)


class Converter(object):
    def __init__(self, provider):
        self.provider = provider

        self.stations = dict()
        self.carriers = dict()

    def convert(self, filepath):
        channel_el = self.build_channel_element()

        with open(filepath, 'w') as f:
            f.write(etree.tostring(
                channel_el, xml_declaration=True, encoding='utf-8', pretty_print=True
            ))

    def build_channel_element(self):
        channel_block = ChannelBlock(
            t_type=self.provider.package.t_type.code,
            timezone='local',
            station_code_system='vendor',
            carrier_code_system='vendor',
            vehicle_code_system='local',

            country_code_system='IATA',
            region_code_system='yandex',
            settlement_code_system='yandex',
        )
        if self.provider.package.t_subtype:
            channel_block.t_subtype = self.provider.package.t_subtype.code

        if self.provider.package.t_subtype:
            channel_block.t_subtype = self.provider.package.t_subtype.code

        group_block = GroupBlock(channel_block, title='all', code='all')

        self.build_stations(group_block)
        self.build_carriers(group_block)

        self.build_threads(group_block)

        channel_block.add_group_block(group_block)

        return channel_block.get_element()

    def build_threads(self, group_block):
        for route in self.provider.triangle_routes:
            stations_and_distances = route.station_list.triangle_stations_and_distances

            carrier = None

            if route.route_info.company:
                carrier = self.carriers.get(route.route_info.company.code, None)

            fare_block = self.build_fare_block(group_block, route)

            thread_title = route.route_info.supplier_title.strip()
            use_supplier_title = False

            if thread_title.startswith(u'@'):
                use_supplier_title = True

                thread_title = unify_minuses(thread_title[1:])

                if u' - ' not in thread_title:
                    log.error(_(u"Нет разделителя ' - ' в названии %s"), thread_title)

            for schedule_period in route.schedule_periods:
                thread_block = group_block.add_thread_block(ThreadBlock(
                    group_block,
                    title=thread_title,
                    number=route.route_info.number,
                    carrier=carrier,
                    fare=fare_block,
                    use_supplier_title=use_supplier_title,
                ))

                thread_block.set_raw_data(route.get_raw_data(schedule_period.column_number))

                times = schedule_period.times

                for (station, distance), point_times in zip(stations_and_distances, times):
                    if not point_times:
                        continue

                    thread_block.add_stoppoint_block(StoppointBlock(
                        thread_block,
                        self.stations[station.code],
                        distance=distance,
                        departure_time=point_times.departure_time,
                        arrival_time=point_times.arrival_time,
                        departure_day_shift=point_times.departure_day_shift,
                        arrival_day_shift=point_times.arrival_day_shift,
                        is_fuzzy='1' if point_times.is_fuzzy else None
                    ))

                start_date = schedule_period.start_date.strftime('%Y-%m-%d') \
                    if schedule_period.start_date else None

                end_date = schedule_period.end_date.strftime('%Y-%m-%d') \
                    if schedule_period.end_date else None

                thread_block.add_schedule_block(ScheduleBlock(
                    thread_block,
                    days=schedule_period.days,
                    exclude_days=schedule_period.exclude_days,
                    period_start_date=start_date,
                    period_end_date=end_date,
                ))

    def build_stations(self, group_block):
        for station in self.provider.stations.stations.values():
            yandex_code = self.provider.station_mappings.get((station.title, station.code))

            if yandex_code:
                station_block = group_block.add_station(station.title, yandex_code, code_system='yandex')
                self.stations[station.code] = station_block
            else:
                station_block = group_block.add_station(station.title, station.code)

                station_block.country_code = station.country_code
                station_block.region_code = station.region_code
                station_block.settlement_code = station.settlement_code

                station_block.add_legacy_station(station.title, station.code)

                self.stations[station.code] = station_block

    def build_carriers(self, group_block):
        for carrier in self.provider.carriers.values():
            carrier_block = group_block.add_carrier(
                title=carrier.title,
                code=carrier.code,
                phone=carrier.phone,
                address=carrier.address,
                country_code=self.provider.country.code
            )

            self.carriers[carrier.code] = carrier_block

    def build_fare_block(self, group_block, route):
        if not route.tariff_matrix.has_tariff:
            return None

        triangle_stations = route.station_list.triangle_stations

        fare_block = group_block.add_local_fare()

        for from_index, to_index in combinations(xrange(len(triangle_stations)), 2):
            price = route.tariff_matrix.get_tariff(from_index, to_index)

            if price:
                fare_block.add_price_block(
                    price_value='%.2f' % price,
                    currency=self.provider.currency,
                    stop_from=self.stations[triangle_stations[from_index].code],
                    stop_to=self.stations[triangle_stations[to_index].code],
                    data=None
                )

        return fare_block


class TriangleDataProvider(object):
    def __init__(self, triangle_factory):
        self.package = triangle_factory.package

        self.currency = self.get_currency()
        self.country = self.package.country

        self.mask_parser = triangle_factory.get_triangle_mask_parser()

        self.file_provider = PackageFileProvider(self.package)

        self.validate_triangle_file()
        self.filemap = self.file_provider.get_unpack_map_with_trimmed_dirs_from_package_archive()

        self.station_mappings = self.get_station_mappings()
        self.stations = TriangleStations(self.get_stations())
        self.carriers = self.get_carriers()

        self.triangle_routes = self.get_triangle_routes()

    def get_triangle_routes(self):
        timetable_file_parser = get_file_parser(
            self.get_filepath_by_filename_without_extension('timetable'),
            TIMETABLE_FIELDS,
            encoding=self.package.encoding
        )

        return TriangleRoutesParser.parse_routes(timetable_file_parser, self.stations,
                                                 self.carriers, self.mask_parser)

    def get_stations(self):
        stations = dict()

        file_parser = get_file_parser(
            self.get_filepath_by_filename_without_extension('stations'),
            STATION_FIELDS,
            encoding=self.package.encoding
        )

        # skip titles
        file_parser.next()

        for rowdict in file_parser:
            if not rowdict['code']:
                continue

            triangle_station = TriangleStation.create_from_rowdict(rowdict)

            if triangle_station.code in stations:
                log.error(N_(u'Дублирование кода %s в файле станций, берем последнюю.'),
                          triangle_station.code)

            stations[triangle_station.code] = triangle_station

        return stations

    def get_carriers(self):
        carriers = dict()

        file_parser = get_file_parser(
            self.get_filepath_by_filename_without_extension('carriers'),
            CARRIER_FIELDS,
            encoding=self.package.encoding
        )

        # skip titles
        file_parser.next()

        for rowdict in file_parser:
            if not rowdict['code']:
                continue

            triangle_carrier = TriangleCarrier.create_from_rowdict(rowdict)

            if triangle_carrier.code in carriers:
                log.error(N_(u'Дублирование кода %s в файле перевозчиков, берем последнюю.'),
                          triangle_carrier.code)

            carriers[triangle_carrier.code] = triangle_carrier

        return carriers

    def get_filepath_by_filename_without_extension(self, filename):
        allowed_extensions = FILE_FORMATS

        for ext in allowed_extensions:
            if (filename + '.' + ext) in self.filemap:
                return self.filemap[(filename + '.' + ext)]

    def get_currency(self):
        if self.package.currency:
            return self.package.currency.code

        else:
            return 'RUR'

    def validate_triangle_file(self):
        try:
            triangle_file_validator(self.file_provider.fileobj)

        except ValidationError as e:
            raise CriticalImportError(u'\n'.join(e.messages))

    def get_station_mappings(self):
        station_mappings = dict()

        station_mapping_filepath = self.get_filepath_by_filename_without_extension('station_mapping')

        if not station_mapping_filepath:
            log.info(N_(u'Файла с соответствиями не найдено.'))

        else:
            log.info(N_(u'Разбираем соответсвия.'))

            file_parser = get_file_parser(
                station_mapping_filepath,
                fieldnames=None,
                encoding=self.package.encoding
            )

            for rowdict in file_parser:
                code = rowdict['code'].strip()
                title = rowdict['title'].strip()
                station_id = rowdict['station_id'].strip()

                if not code or not title or not station_id:
                    continue

                station_mappings[(title, code)] = station_id

        return station_mappings


class TriangleRoutesParser():
    @classmethod
    def parse_routes(cls, timetable_file_parser, stations, carriers, mask_parser):
        triangle_routes = []

        # skip 2 first rows
        timetable_file_parser.next()
        timetable_file_parser.next()

        while True:
            first_rowdict_of_route = cls.find_first_rowdict_of_route(timetable_file_parser)

            if not first_rowdict_of_route:
                break

            log.info(N_(u'Собираем данные для: %s'), first_rowdict_of_route)

            triangle_route_rowdicts = cls.get_triangle_route_rowdicts(timetable_file_parser,
                                                                      first_rowdict_of_route)

            if triangle_route_rowdicts is not None:
                triangle_routes.append(
                    TriangleRoute(triangle_route_rowdicts, stations, carriers, mask_parser)
                )

        return triangle_routes

    @classmethod
    def find_first_rowdict_of_route(cls, timetable_file_parser):
        for timetable_rowdict in (TimetableRowdict(rowdict) for rowdict in timetable_file_parser):
            if timetable_rowdict.is_route_data_start():
                if not timetable_rowdict.has_any_station_column():
                    return timetable_rowdict

                else:
                    log.error(N_(u'Пропускаем маршрут. В строке заголовка маршрута,'
                                 u' не должно быть данных о станции. %s'),
                              timetable_rowdict.get_station_info())

            elif timetable_rowdict.has_data():
                log.warning(N_(u'Неожиданные данные между маршрутами: %s'), timetable_rowdict)

    @classmethod
    def get_triangle_route_rowdicts(cls, timetable_file_parser, first_route_timetable_rowdict):
        try:
            return cls._get_triangle_route_rowdicts(timetable_file_parser, first_route_timetable_rowdict)

        except TriangleParseError as e:
            log.error(e.msg_template, *e.msg_args)

            return None

    @classmethod
    def _get_triangle_route_rowdicts(cls, timetable_file_parser, first_route_timetable_rowdict):
        route_data_rowdicts = [first_route_timetable_rowdict]

        for timetable_rowdict in (TimetableRowdict(rowdict) for rowdict in timetable_file_parser):
            route_data_rowdicts.append(timetable_rowdict)

            if timetable_rowdict.is_route_data_end():
                if timetable_rowdict.has_any_station_column():
                    raise TriangleParseError(
                        N_(u'Пропускаем маршрут. В строке конца маршрута не должно быть данных о станции. %s'),
                        timetable_rowdict.get_station_info()
                    )
                break

            elif timetable_rowdict.has_any_route_column():
                raise TriangleParseError(
                    N_(u'Пропускаем маршрут, в строчке станции, поля маршрут,'
                       u' название, переовзчик и автобус, должны быть пустыми. %s'),
                    timetable_rowdict.get_route_info()
                )

        if not route_data_rowdicts[-1].is_route_data_end():
            raise TriangleParseError(N_(u'У последнего маршрута в файле нет признака окончания'))

        return route_data_rowdicts


class TriangleRoute(object):
    def __init__(self, rowdicts, stations, carriers, mask_parser):
        self.raw_rowdicts = [r.rowdict for r in rowdicts]
        self.rowdicts = [r.rowdict for r in rowdicts[:-1] if r.has_data()]

        self.carriers = carriers
        self.stations = stations

        self._valid = True

        self.mask_parser = mask_parser

        self.parse_rowdicts()

    def parse_rowdicts(self):
        try:
            self.route_info = RouteInfo(self.rowdicts[0], self.carriers)
            self.station_list = TriangleRouteStationList(self.rowdicts[1:], self.stations)
            self.tariff_matrix = TriangleTariffMatrix(self.rowdicts, self.station_list.size)
            self.schedule_periods = self.get_schedule_periods()

        except Exception:
            log.exception(N_(u'Ошибка разбора маршрута'))

            self._valid = False

    def is_valid(self):
        if not self._valid:
            return False

        for row in self.rowdicts[1:]:
            if row['number'].lower().startswith(START_ROUTE_BLOCK):
                log.error(
                    N_(u'Неверные данные в маршруте %s %s неожиданное начало следующего маршрута.'),
                    self.route_info.supplier_number, self.route_info.supplier_title
                )

                return False

        return True

    def get_schedule_periods(self):
        schedule_periods = []

        for column in self.parse_mask_templates_and_times_columns():
            try:
                schedule_periods.extend(self.parse_mask_templates_and_times_column(column))

            except TimeTableParserError as e:
                log.error(e.msg_template, *e.msg_args)

        return schedule_periods

    def parse_mask_templates_and_times_columns(self):
        columns_count = self.get_mask_templates_and_times_columns_count()

        log.debug(N_(u'Количество расписаний %s'), columns_count)

        start_column_index = self.get_number_of_colums_before_schedule()
        columns = [ScheduleColumn(start_column_index + i) for i in range(columns_count)]

        for rowdict in self.rowdicts:
            tail = rowdict['tail']

            for index, column in enumerate(columns):
                column.data.append(tail[index])

        return columns

    def parse_mask_templates_and_times_column(self, column):
        times = self.get_times(column.data)

        template_text = column.data[0]

        schedule_periods = []

        parser = self.mask_parser

        try:
            for group in parser.split_template_text_to_groups(template_text):
                period_days = group.to_cysix(parser)

                schedule_periods.append(SchedulePeriod(
                    period_days['days'], period_days['exclude_days'],
                    times,
                    group.start_date, group.end_date,
                    column.column_number
                ))

        except TriangleMaskTemplateParser.ParseError as e:
            log.error(u'%s', e)

        return schedule_periods

    def get_times(self, column):
        times = []

        for time_str in column[1:]:
            times.append(self.parse_triangle_time(time_str))

        self.set_arrival_time_for_last_stop(times)

        return times

    def set_arrival_time_for_last_stop(self, times):
        def get_last_good_index_or_none():
            for i, t in enumerate(reversed(times)):
                if t is None:
                    continue

                if t.arrival_time is not None or t.departure_time is not None:
                    return len(times) - i - 1

            return None

        last_good_index = get_last_good_index_or_none()

        if last_good_index is None:
            return

        if times[last_good_index].arrival_time is None and times[last_good_index].departure_time is not None:
            times[last_good_index].arrival_time = times[last_good_index].departure_time
            times[last_good_index].departure_time = None

            times[last_good_index].arrival_day_shift = times[last_good_index].departure_day_shift
            times[last_good_index].departure_day_shift = None

    def parse_triangle_time(self, time_str):
        time_str = time_str.strip()
        if time_str in (u'', u'-'):
            return None

        is_fuzzy = False
        if u'?' in time_str or u'~' in time_str:
            time_str = time_str.replace(u'?', u'').replace(u'~', u'').strip()

            if not time_str:
                log.error(N_(u'У нечеткой станции не указано время, считаем что автобус станцию не посещает.'))

                return None

            is_fuzzy = True

        try:
            points_time = self.parse_points_time(time_str)
            points_time.is_fuzzy = is_fuzzy

            return points_time

        except Exception:
            raise TimeTableParserError(N_(u"Ошибка разбора времени '%s'"), time_str)

    def parse_points_time(self, time_str):
        if '/' in time_str:
            arrival_time_str, departure_time_str = time_str.split('/')
        else:
            departure_time_str = time_str
            arrival_time_str = None

        departure_day_shift, departure_time = self.parse_triangle_time_with_shift(departure_time_str)
        if arrival_time_str:
            arrival_day_shift, arrival_time = self.parse_triangle_time_with_shift(arrival_time_str)
        else:
            arrival_day_shift, arrival_time = None, None

        return PointTimes(departure_time, arrival_time, departure_day_shift, arrival_day_shift)

    def parse_triangle_time_with_shift(self, time_str):
        """
        10:00
        10:00+1  +1 день
        """

        shift = None

        if u'+' in time_str:
            time_str, shift = time_str.split(u'+')
            shift = int(shift)
            if shift < 1:
                raise TimeTableParserError(N_(u'Сдвиг меньше нуля'))

        return shift, time(*map(lambda t: int(t.strip()), time_str.split(u':')))

    # Английская и русская т
    mask_templates_end_re = re.compile(ur'^\s*t|т\s*|\s*$', re.U | re.I)

    def get_mask_templates_and_times_columns_count(self):
        tail = self.rowdicts[0]['tail']
        count = 0
        for count, value in enumerate(tail):
            if self.mask_templates_end_re.match(value):
                break
        else:
            log.debug(N_(u'Нет колонок с тарифами, значит текущая колонка последняя'))
            count += 1

        if not count:
            raise TimeTableParserError(N_(u'У маршрута нет колонок дней курсирования.'))

        return count

    def get_number_of_colums_before_schedule(self):
        return len(TIMETABLE_FIELDS)

    def get_raw_data(self, column_number):
        table = self.get_table_from_rowdict()

        column_separator = u'; '
        row_separator = u'\n'

        formatted_table = FormattedTable.build(table, column_separator=column_separator, row_separator=row_separator)

        column_selector_row = column_separator.join(
            (u'^' if i == column_number else u' ') * formatted_table.max_width_of_columns[i]
            for i in xrange(formatted_table.number_of_columns)
        )

        return formatted_table.formatted_table + row_separator + column_selector_row

    def get_table_from_rowdict(self):
        table = []

        for rowdict in self.rowdicts:
            row = []

            for key, value in rowdict.iteritems():
                if key == 'tail':
                    row.extend(value)

                else:
                    row.append(value)

            table.append(row)

        return table


class FormattedTable(object):
    """
    Класс строит форматированную текстовую таблицу из списка списков.
    Вызывать FormattedTable.build(...)
    """

    def __init__(self, formatted_table, number_of_columns, max_width_of_columns):
        self.formatted_table = formatted_table
        self.number_of_columns = number_of_columns
        self.max_width_of_columns = max_width_of_columns

    @classmethod
    def build(cls, table, column_separator=u'; ', row_separator=u'\n'):
        number_of_columns = max(len(row) for row in table)

        max_width_of_columns = defaultdict(lambda: 0)

        for row in table:
            for i, value in enumerate(row):
                max_width_of_columns[i] = max(max_width_of_columns[i], len(value))

        result_rows = []

        for row in table:
            result_row_parts = []

            for i, value in enumerate(row):
                result_row_parts.append(u'%s%s' % (u' ' * (max_width_of_columns[i] - len(value)), value))

            result_row_parts += [u'' for _k in xrange(len(result_row_parts), number_of_columns)]

            result_rows.append(column_separator.join(result_row_parts))

        formatted_table = row_separator.join(result_rows)

        return cls(formatted_table, number_of_columns, max_width_of_columns)


class ScheduleColumn(object):
    def __init__(self, column_number):
        self.data = []
        self.column_number = column_number


class SchedulePeriod(object):
    def __init__(self, days, exclude_days, times, start_date, end_date, column_number):
        self.days = days
        self.exclude_days = exclude_days
        self.times = times
        self.start_date = start_date
        self.end_date = end_date
        self.column_number = column_number


class PointTimes(object):
    def __init__(self, departure_time, arrival_time, departure_day_shift, arrival_day_shift):
        self.departure_time = departure_time
        self.arrival_time = arrival_time
        self.is_fuzzy = False
        self.departure_day_shift = departure_day_shift
        self.arrival_day_shift = arrival_day_shift

    def __repr__(self):
        return u'<PointTimes {!r}'.format(self.__dict__)


class RouteInfo(object):
    company = None
    number = u''
    supplier_title = u''
    supplier_number = u''

    def __init__(self, rowdict, carriers):
        self.carriers = carriers
        self.rowdict = rowdict

        self._parse()

    def _parse(self):
        self.parse_company()
        self.parse_number()

        self.supplier_title = self.rowdict['route_title']

    def parse_company(self):
        try:
            self.company = self.carriers[self.rowdict['carrier_code']]

        except KeyError:
            log.warning(N_(u'Перевозчик с кодом %s не найден, не указываем перевозчика.'),
                        self.rowdict['carrier_code'])

    def parse_number(self):
        self.supplier_number = self.rowdict['number']

        self.number = self.clean_number(self.supplier_number)

    @classmethod
    def clean_number(cls, number):
        if NUMBER_RE.match(number):
            number = NUMBER_RE.match(number).groupdict()['number']

            return number.strip()

        return u''


class TriangleRouteStationList(object):
    def __init__(self, rowdicts, stations):
        self.rowdicts = rowdicts
        self.stations = stations

        self.triangle_stations_and_distances = []

        self._parse()

    def _parse(self):
        for rowdict in self.rowdicts:
            triangle_station = self.stations.get_or_create_station(
                rowdict['station_title'],
                rowdict['station_code']
            )

            distance = None

            if rowdict['distance']:
                try:
                    distance = float(rowdict['distance'].replace(',', '.'))

                except ValueError:
                    log.warning(N_(u'Не верное значения расстояния %s'), rowdict['distance'])

            self.triangle_stations_and_distances.append((triangle_station, distance))

    @property
    def size(self):
        return len(self.triangle_stations_and_distances)

    @property
    def triangle_stations(self):
        return [s for s, _ in self.triangle_stations_and_distances]


class TriangleTariffMatrix(TariffMatrix):
    def __init__(self, rowdicts, stations_rows_count):
        super(TriangleTariffMatrix, self).__init__(log)

        self.rowdicts = rowdicts

        self.station_rows = rowdicts[1:]

        self.station_rows_count = stations_rows_count
        self.tariff_matrix_size = self.station_rows_count - 1

        self.has_tariff = False
        self._parse()

    def _parse(self):
        matrix_columns = self.get_tariff_matrix_columns()

        if matrix_columns is None:
            return

        for from_station_index in range(self.tariff_matrix_size):
            for to_station_index in range(from_station_index + 1, self.station_rows_count):
                tariff = self.get_float(matrix_columns[from_station_index][to_station_index])

                if tariff:
                    self.set_tariff(from_station_index, to_station_index, tariff)

                    self.has_tariff = True

    # Английская и русская т
    is_tariff_re = re.compile(ur'^\s*t|т\s*$', re.U | re.I)

    def get_tariff_start_index(self):
        tail = self.rowdicts[0]['tail']
        for count, value in enumerate(tail):
            if self.is_tariff_re.match(value):
                break
        else:
            count = None

        return count

    def get_tariff_matrix_columns(self):
        start_index = self.get_tariff_start_index()

        if start_index is not None:
            log.debug(N_(u'Отсчет тарифов с дополнительной колонки %s'), start_index)

        columns = [list() for _ in range(self.tariff_matrix_size)]

        if start_index is None:
            log.debug(N_(u'Тарифы не указаны'))

            return None

        for rowdict in self.station_rows:
            tail = rowdict['tail']
            matrix_tail = tail[start_index:]
            for index, column in enumerate(columns):
                try:
                    column.append(matrix_tail[index])
                except IndexError:
                    column.append(u'')

        return columns

    def get_float(self, tariff_str):
        tariff_str = tariff_str.replace(u',', u'.')

        try:
            return float(tariff_str)

        except (TypeError, ValueError):
            tariff_str = unify_minuses(tariff_str).replace(u'-', u'').strip()

            if tariff_str:
                log.error(N_(u"Неправильно заполнен тариф '%s'"), tariff_str)

            return None


class TimetableRowdict(object):
    def __init__(self, rowdict):
        self.rowdict = rowdict

    def is_route_data_end(self):
        return self.rowdict['number'].lower().startswith(END_ROUTE_BLOCK)

    def is_route_data_start(self):
        return self.rowdict['number'].lower().startswith(START_ROUTE_BLOCK)

    @cache_method_result
    def has_any_route_column(self):
        return any(self.rowdict[column] for column in ROUTE_INFO_COLUMNS)

    @cache_method_result
    def has_any_station_column(self):
        return any(self.rowdict[column] for column in STATION_INFO_COLUMNS)

    def get_route_info(self):
        return u' | '.join(self.rowdict[k] for k in ROUTE_INFO_COLUMNS)

    def get_station_info(self):
        return u' | '.join(self.rowdict[k] for k in STATION_INFO_COLUMNS)

    def has_data(self):
        for k in self.rowdict:
            if k != 'tail':
                if self.rowdict[k]:
                    return True

            else:
                if any(self.rowdict['tail']):
                    return True

        return False

    def __unicode__(self):
        return u', '.join(u'%s=%s' % (k, v) for k, v in self.rowdict.items())


class TriangleStations(object):
    def __init__(self, stations):
        self.stations = stations or dict()

    def get_or_create_station(self, title, code):
        if code not in self.stations:
            log.error(N_(u'Не нашли станцию title=%s code=%s в файле станций.'),
                      title, code)

            self.stations[code] = TriangleStation(title, code)

        return self.stations[code]


class TriangleStation(object):
    def __init__(self, title, code, region_code=None, settlement_code=None, country_code=None):
        self.title = title
        self.code = code
        self.region_code = region_code
        self.settlement_code = settlement_code
        self.country_code = country_code

    @classmethod
    def create_from_rowdict(cls, rowdict):
        country = None
        region = None
        settlement = None

        if rowdict['country']:
            for lang in [None] + settings.MODEL_LANGUAGES:
                field = 'title_' + lang if lang else 'title'
                try:
                    country = Country.objects.get(**{field: rowdict['country']})
                    break
                except (Country.DoesNotExist, Country.MultipleObjectsReturned):
                    pass
            else:
                log.error(N_(u'Не нашли страну по названию %s.'), rowdict['country'])

        if rowdict['region']:
            for lang in [None] + settings.MODEL_LANGUAGES:
                field = 'title_' + lang if lang else 'title'
                try:
                    region = Region.objects.get(**{field: rowdict['region']})
                    break
                except (Region.DoesNotExist, Region.MultipleObjectsReturned):
                    pass
            else:
                log.error(N_(u'Не нашли регион по названию %s.'), rowdict['region'])

        if rowdict['settlement']:
            for lang in [None] + settings.MODEL_LANGUAGES:
                field = 'title_' + lang if lang else 'title'
                try:
                    settlement = Settlement.objects.get(**{field: rowdict['settlement']})
                    break
                except (Settlement.DoesNotExist, Settlement.MultipleObjectsReturned):
                    pass
            else:
                log.error(N_(u'Не нашли город по названию %s.'), rowdict['settlement'])

            if settlement is None:
                log.info(N_(u'Пробуем отыскать отрубив г.'))
                title = rowdict['settlement'].replace(u'г.', u'').strip()
                for lang in [None, 'ru']:
                    field = 'title_' + lang if lang else 'title'
                    try:
                        settlement = Settlement.objects.get(**{field: title})
                        break
                    except (Settlement.DoesNotExist, Settlement.MultipleObjectsReturned):
                        pass
                else:
                    log.error(N_(u'Не нашли город по названию %s.'), title)

        return cls(
            title=rowdict['title'],
            code=rowdict['code'],
            settlement_code=settlement and unicode(settlement.id),
            region_code=region and unicode(region.id),
            country_code=country and unicode(country.code)
        )


class TriangleCarrier(object):
    def __init__(self, title, code, address=None, phone=None):
        self.title = title
        self.code = code
        self.address = address
        self.phone = phone

    @classmethod
    def create_from_rowdict(cls, rowdict):
        return cls(
            title=rowdict['title'],
            code=rowdict['code'],
            address=rowdict['address'] or None,
            phone=rowdict['phone'] or None
        )


def get_file_parser(filepath, fieldnames, encoding=None):
    _, ext = os.path.splitext(filepath)

    if ext == '.xls':
        return XlsDictParser(filepath, transform_to_text=True, fieldnames=fieldnames,
                             strip_values=True)
    elif ext == '.csv':
        if not encoding:
            encoding = detect_encoding(filepath)

        return UnicodeDictReader(open(filepath), fieldnames=fieldnames, encoding=encoding,
                                 restkey='tail', delimiter=';', strip_values=True)
    else:
        raise Triangle2CysixError(N_(u"%s not supported"), ext)


def detect_encoding(filepath):
    encoding = detect_file_encoding_by_path(filepath)

    if encoding is None:
        possible_encoding = detect_file_encoding_by_path(filepath, any_confidence=True)
        log.error(
            N_(u'Проблемы с определением кодировки файла %s возможная кодировка %s'),
            os.path.basename(filepath), possible_encoding
        )

        log.warning(N_(u"Берем cp1251 в таком случае для файла %s"), os.path.basename(filepath))

        encoding = 'cp1251'

    return encoding
