# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import, division, print_function
import base64
import errno
import heapq
import json
import logging
import os
import shutil
import urllib2
import urlparse
import zlib
from bisect import bisect_left
from collections import defaultdict
from copy import copy
from datetime import datetime, timedelta, time
from itertools import islice

import pytz
from django.conf import settings
from django.utils.encoding import smart_unicode
from django.utils.functional import cached_property
from django.utils.translation import ugettext as _, gettext_noop as N_
from lxml import etree

from common.models.geo import Station, StationMajority
from common.models.schedule import RTStation, Route
from common.models.tariffs import ThreadTariff
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_method_result, cache_method_result_with_exception
from common.utils.date import RunMask
from common.utils.date import timedelta2minutes
from cysix.base import CysixCompany, CysixTransportModel
from cysix.filters.apply_base_stations import apply_base_and_trusted_stations
from cysix.filters.clear_consecutive_duplicate_rtstations import clear_consecutive_duplicate_rtstations
from cysix.two_stage_import.mask_parser import CysixMaskParser
from travel.rasp.admin.importinfo.models.two_stage_import import TSIThreadStationFlag
from travel.rasp.admin.importinfo.two_stage_import.tariffs import TariffVariant
from travel.rasp.admin.lib.xmlutils import get_sub_tag_text, copy_lxml_element
from travel.rasp.admin.scripts.schedule.utils import SupplierRoute, SupplierPath
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.file_providers import XmlPackageFileProvider, PackageFileProvider
from travel.rasp.admin.scripts.schedule.utils.supplier_station import SupplierStation
from cysix.two_stage_import.utils import xsd_boolean, group_description


log = logging.getLogger(__name__)

define_setting('YT_FILE_HOST', default='hahn.yt.yandex-team.ru')


class CysixXmlPackageFileProvider(XmlPackageFileProvider):
    @cache_method_result_with_exception
    def get_cysix_file(self):
        files = self.get_schedule_files()

        if len(files) != 1:
            raise RaspImportError(_('Должен быть приложен ровно 1 файл, найдено %s') % repr(files))

        return files[0]

    def store_fileobj(self, fileobj, filename, subpath=None):
        filepath = self.get_package_filepath(filename)

        # filename may consists subdirs
        if not os.path.exists(os.path.dirname(filepath)):
            try:
                os.makedirs(os.path.dirname(filepath))
            except OSError as exc:  # Guard against race condition
                if exc.errno != errno.EEXIST:
                    raise

        fileobj.seek(0)
        with open(filepath, 'wb') as f:
            f.write(fileobj.read())

        return filepath


class CysixHTTPFileProvider(PackageFileProvider):
    def __init__(self, package):
        self.arhiv_extract = {
            'zip': self.unzip_files,
            'rar': self.unrar_files
        }

        super(CysixHTTPFileProvider, self).__init__(package)

        self.package = package

    @cache_method_result_with_exception
    def get_cysix_file(self):
        log.info(N_('Качаем файл по %s'), self.package.url)

        filepath = self.get_package_filepath('cysix.xml')

        req = self.build_request()

        arhiv_ext = self.get_arhive_ext_or_none(self.package.url)

        if arhiv_ext:
            self.download_arhive_and_extract_xml(req, arhiv_ext, filepath)

        else:
            self.download_file(req, filepath)

        return filepath

    def build_request(self):
        req = urllib2.Request(self.package.url)

        username = self.package.username
        password = self.package.password

        if urlparse.urlsplit(self.package.url).netloc == settings.YT_FILE_HOST:
            req.add_header('Authorization', 'OAuth {}'.format(settings.YT_TOKEN))
        elif username or password:
            base64string = base64.encodestring('{}:{}'.format(username, password))[:-1]
            req.add_header('Authorization', 'Basic {}'.format(base64string))

        return req

    def download_arhive_and_extract_xml(self, req, arhiv_ext, filepath):
        tmp_file = self.get_package_filepath('cysix.' + arhiv_ext)

        self.download_file(req, tmp_file)

        file_map = self.arhiv_extract[arhiv_ext](tmp_file)

        if len(file_map) != 1:
            raise RaspImportError(_('В архиве должен быть ровно 1 файл, найдено {}').format(len(file_map)))

        xml_filepath = file_map.values()[0]

        shutil.copy(xml_filepath, filepath)

    def get_arhive_ext_or_none(self, url):
        arhiv_ext = None

        is_archive_file = self.is_archive_file(url)

        if is_archive_file:
            arhiv_ext = is_archive_file.group(1)

        if arhiv_ext not in self.arhiv_extract:
            arhiv_ext = None

        return arhiv_ext


class CysixLegacyStation(SupplierStation):
    pass


class CysixStation(SupplierStation):
    station_code = None
    station_code_system = None
    group_el = None
    cysix_context = None
    settings = None
    data = None
    geocode_title = None

    def __unicode__(self):
        return "<StopPoint: station_title='%s' station_code='%s' station_code_system='%s' %s>" % (
            self.title, self.station_code, self.station_code_system, group_description(self.group_el)
        )

    @classmethod
    def create_station(cls, stoppoint_el, thread_el, group_el,
                       factory, stoppoint_context, settings):
        group_code = group_el.get('code', "").strip()

        station_code_system = stoppoint_context.station_code_system

        station_code = stoppoint_el.get('station_code', "").strip()

        if not station_code:
            raise RaspImportError(_('Не указан station_code'))

        if not station_code_system:
            raise RaspImportError(_('station_code_system не заполнена'))

        if not group_code:
            raise RaspImportError(_('group_code не заполнен'))

        ref_code = '_'.join([group_code, station_code_system, station_code])

        if station_code_system == 'local':
            code = '_'.join([group_code, station_code_system])
        else:
            code = ref_code

        simple_code = None

        if settings.use_thread_in_station_code:
            simple_code = code

            thread_code_parts = [
                code,
                thread_el.get('title', '').strip(),
                thread_el.get('number', '').strip()
            ]

            code = '_'.join(thread_code_parts)

        data_provider = factory.get_data_provider()

        station_el = data_provider.get_station_el_by_ref_code(ref_code)

        if station_el is None and station_code_system in ('local', 'vendor'):
            raise RaspImportError(
                _(
                    'Станции '
                    '<StopPoint: station_title="%s" station_code="%s" station_code_system="%s">'
                    ' нет в блоке stations'
                ) % (stoppoint_el.get('station_title', '').strip(), station_code, station_code_system)
            )

        station_title = station_el.get('title', '').strip() if station_el is not None else ''

        cysix_station = cls(station_title, code)
        cysix_station.settings = settings
        cysix_station.group_el = group_el
        cysix_station.cysix_context = stoppoint_context
        cysix_station.station_code = station_code
        cysix_station.station_code_system = station_code_system

        cysix_station.legacy_stations = []

        if station_el is not None:
            cysix_station.real_title = station_el.get('recommended_title', '') or None
            cysix_station.geocode_title = station_el.get('_geocode_title', '') or None
            latitude_text = station_el.get('lat', '').strip()
            longitude_text = station_el.get('lon', '').strip()

            if latitude_text and longitude_text:
                try:
                    latitude = float(latitude_text)
                    longitude = float(longitude_text)

                    cysix_station.latitude = latitude
                    cysix_station.longitude = longitude
                except ValueError:
                    log.error(N_('Ошибка разбора координат "%s" "%s" станции %s'),
                              latitude_text, longitude_text, cysix_station)
            elif latitude_text or longitude_text:
                log.warning(N_('Указана только одна из координат для станции %s'), cysix_station)

        if simple_code:
            cysix_station.legacy_stations.append(CysixLegacyStation(
                station_title,
                simple_code
            ))

        cysix_station.data = get_sub_tag_text(station_el, 'data', silent=True)

        return cysix_station

    def get_json_data(self):
        if self.data:
            return json.loads(self.data)


class CysixRoute(SupplierRoute):
    """
    Класс для импорта нитки в "основную базу"
    """

    ROUTE_GEN_UID_PARAMS = {'use_stations': True, 'use_company': True}

    def __init__(self, xml_thread, supplier_rtstations, cysix_schedule, factory):
        self.factory = factory

        self.thread_el = xml_thread.thread_el
        self.group_el = xml_thread.group_el
        self.xml_thread = xml_thread

        self.settings = self.xml_thread.get_settings()

        self.supplier_rtstations = supplier_rtstations
        self.cysix_schedule = cysix_schedule
        self.local_mask = cysix_schedule.local_mask
        self.local_start_time = cysix_schedule.local_start_time
        self.thread_pytz = cysix_schedule.pytz
        self.station_finder = self.factory.get_station_finder()
        self.company_finder = self.factory.get_company_finder()
        self.transport_model_finder = self.factory.get_transport_model_finder()
        self.context = cysix_schedule.schedule_context

        self.is_interval = cysix_schedule.is_interval
        self.interval_start_time = cysix_schedule.interval_start_time
        self.interval_end_time = cysix_schedule.interval_end_time
        self.period_int = cysix_schedule.period_int

        self.comment = cysix_schedule.comment
        self.density = cysix_schedule.density

        self.add_original_data()

        self.supplier_number = self.thread_el.get('number', '').strip()
        self.supplier_title = self.thread_el.get('title', '').strip()

    def get_path_key(self):
        return self.xml_thread.path_key

    @cached_property
    def path_key(self):
        return self.get_path_key()

    @cached_property
    def tsi_thread_setting(self):
        return self.xml_thread.tsi_thread_setting

    @cached_property
    def timezones(self):
        timezones = {self.thread_pytz.zone}

        first_station = self.supplier_rtstations[0].station

        # Если первая станция не привязана, то нитка не импортируеся.
        if first_station is not None:
            for rts in self.supplier_rtstations:
                # Если станция не привязана, то она все равно не импортируеся.
                if rts.station is not None:
                    timezones.add(rts.get_pytz(self.context).zone)

        return list(timezones)

    def add_original_data(self):
        self.raw_data = get_sub_tag_text(self.thread_el, 'raw', silent=True)

        if self.raw_data:
            self.raw_data = smart_unicode(zlib.decompress(base64.b64decode(self.raw_data)))

            tmp_thread_el = copy_lxml_element(self.thread_el)
            tmp_thread_el.remove(tmp_thread_el.find('raw'))

        else:
            tmp_thread_el = self.thread_el

        self.cysix_data = '\n'.join([
            etree.tounicode(self.group_el, pretty_print=True),  # Группа приходит сюда без детей
            etree.tounicode(tmp_thread_el, pretty_print=True)
        ] + [
            etree.tounicode(fare_el, pretty_print=True)
            for _mask, fare_el in self.get_fare_variants()
        ])

    def __unicode__(self):
        return unicode(self.xml_thread)

    def create_route(self):
        route = Route()
        route.supplier = self.factory.package.supplier
        route.two_stage_package = self.factory.package

        if not self.context.t_type:
            raise RaspImportError(_('Нигде не указан тип транспорта'))

        route.t_type = self.context.t_type

        self.route = route

        return route

    def build_thread(self):
        thread = self.create_thread()
        thread.t_subtype = self.context.subtype

        if thread.t_subtype_id and thread.t_subtype.t_type_id != thread.t_type_id:
            log.error(N_('Подтип транспорта не соответствует типу транспорта'))
            thread.t_subtype = None

        if not self.local_mask:
            raise RaspImportError(_('Пустая маска у маршрута'))

        if self.xml_thread.is_thread_hidden:
            log.info(N_('Ставим флаг hidden'))
            thread.hidden = True

        thread.mask = self.local_mask

        if self.is_interval:
            thread.type_id = 12

            thread.begin_time = self.interval_start_time
            thread.end_time = self.interval_end_time
            thread.period_int = self.period_int
            thread.density = self.density

        thread.show_in_alldays_pages = self.settings.show_in_alldays_pages

        thread.has_extrapolatable_mask = self.cysix_schedule.is_extrapolatable()

        thread.tz_start_time = self.local_start_time
        thread.time_zone = self.thread_pytz.zone

        thread.t_model = self.get_transport_model()

        thread.comment = self.comment

        thread.path_key = self.get_path_key()

        transition_expert = self.get_transition_expert()

        safe_start_dt = transition_expert.get_safe_dt_before_transition(out_tz=self.thread_pytz)
        naive_start_dt = safe_start_dt.replace(tzinfo=None)

        if transition_expert.is_different_duration_possible and thread.has_extrapolatable_mask:
            log.warning(
                N_('Экстраполяция разрешена, при этом после ближайщего перевода часов %s время в пути будет другим.'),
                transition_expert.get_first_transition_dt(out_tz=self.thread_pytz).date()
            )

        self.log_warning_if_thread_has_different_durations(transition_expert)

        self.build_rtstations(thread, naive_start_dt)

        filter_ = self.factory.get_filter('import_geometry')
        filter_.apply(self, thread)

        return thread

    def get_transition_expert(self):
        local_start_date = RunMask.first_run(str(self.local_mask), environment.today())
        naive_start_dt = datetime.combine(local_start_date, self.local_start_time)

        start_dt = self.thread_pytz.localize(naive_start_dt)

        return TransitionExpert(self.timezones, start_dt)

    def log_warning_if_thread_has_different_durations(self, transition_expert):
        """
        Варнинг пишем, если после перевода часов может изменится время в пути и
        есть даты хождения нитки после перевода часов
        """

        if transition_expert.is_different_duration_possible:
            local_transition_date = transition_expert.get_first_transition_dt(out_tz=self.thread_pytz).date()

            last_date = self.local_mask.dates()[-1]

            if last_date >= local_transition_date:
                log.warning(N_('После перевода времени %s изменится время в пути, например для %s'),
                            local_transition_date, last_date)

    def get_path(self):
        return SupplierPath([srts.supplier_station for srts in self.supplier_rtstations])

    def build_route(self):
        route = self.create_route()

        log.info(N_('Разбираем расписание'))

        thread = self.add_thread_to_route(route, self.get_thread_title_override())

        thread.company = self.get_company()
        thread.t_model = self.get_transport_model()

        if self.settings.set_number:
            thread.number = self.thread_el.get('number', '').strip()

        if self.settings.set_hidden_number:
            thread.hidden_number = self.thread_el.get('number', '').strip()

        return route

    def build_rtstations(self, thread, naive_start_dt):
        parser = CysixRTSParser(self, self.factory, thread, naive_start_dt, self.context, self.settings)

        parser.parse_rtstations()

        parser.add_rtstations_to_thread()

        for rts in thread.rtstations:
            if rts.is_combined:
                thread.is_combined = True

    @cache_method_result
    def get_tariffs_variants(self):
        variants = dict()

        for mask, fare_el in self.get_fare_variants():
            variant = TariffVariant(self.gen_prices(fare_el))

            if variant:
                variants[variant] = mask

        return variants

    def gen_prices(self, fare_el):
        group_context = self.xml_thread.group_context
        group_settings = self.xml_thread.group_settings

        for price_el in fare_el.findall('./price'):
            if not price_el.get('price', '').strip():
                continue

            stop_from_el = price_el.find('./stop_from')

            stop_from_context = copy(group_context)
            stop_from_context.update_from(stop_from_el)

            ss_from = CysixStation.create_station(stop_from_el, self.thread_el, self.group_el,
                                                  self.factory, stop_from_context, group_settings)

            stop_to_el = price_el.find('./stop_to')

            stop_to_context = copy(group_context)
            stop_to_context.update_from(stop_to_el)

            ss_to = CysixStation.create_station(stop_to_el, self.thread_el, self.group_el,
                                                self.factory, stop_from_context, group_settings)

            try:
                tariff = float(price_el.get('price', '').strip())
                currency = price_el.get('currency', '').strip()
                is_min_tariff = xsd_boolean(price_el.get('is_min_price'), False)

                order_data = self.get_order_data(price_el, stop_from_el, stop_to_el)

                fare = ThreadTariff.Fare.create(tariff, currency, order_data=order_data,
                                                is_min_tariff=is_min_tariff)

            except ValueError:
                log.warning(N_('Не смогли разобрать тариф %s %s'),
                            price_el.get('price', '').strip(),
                            self.get_order_data(price_el, stop_from_el, stop_to_el))

                continue

            try:
                station_from = self.station_finder.find_by_supplier_station(ss_from)
                station_to = self.station_finder.find_by_supplier_station(ss_to)

                yield (station_from.id, station_to.id, fare)

            except (Station.DoesNotExist, Station.MultipleObjectsReturned):
                pass

    @cache_method_result_with_exception
    def get_fare_variants(self):
        return list(self._get_fare_variants_iter())

    def _get_fare_variants_iter(self):
        def fare_ref_code(fare_code):
            return self.group_el.get('code', '').strip() + '_' + fare_code

        thread_fare_code = self.thread_el.get('fare_code', '').strip()

        fare_link_els = self.thread_el.findall('./fares/fare')

        if fare_link_els and thread_fare_code:
            log.warning(N_('Нельзя одновременно указывать тарифы через <fares><fare .. />'
                           ' и через аттрибут <thread fare_code="" >'))

        if not fare_link_els and thread_fare_code:
            ref_code = fare_ref_code(thread_fare_code)
            fare_el = self.factory.get_data_provider().get_fare_el(ref_code)

            if fare_el is None:
                log.error(N_('Не наши тарифа с кодом %s в группе %s'),
                          self.thread_el.get('fare_code', "").strip(),
                          self.group_el.get('code', "").strip())
                return

            mask = self.local_mask

            yield mask, fare_el

            return

        settings = self.settings
        mask_builder = self.factory.get_mask_builder(start_date=settings.start_date, end_date=settings.end_date)

        added_masks = RunMask(today=self.local_mask.today)

        # combine tariff masks by ref_code
        tariff_mask_by_ref_code = defaultdict(lambda: RunMask(today=self.local_mask.today))
        for fare_link_el in fare_link_els:
            ref_code = fare_ref_code(fare_link_el.get('code', '').strip())

            try:
                actual_tariff_mask = CysixMaskParser.parse_mask(fare_link_el, mask_builder)

            except RaspImportError, e:
                log.warning(
                    N_('Ошибка разбора маски тарифа %s %s'),
                    etree.tounicode(fare_link_el, pretty_print=True),
                    unicode(e)
                )

                continue

            actual_tariff_mask &= self.local_mask

            if not actual_tariff_mask:
                log.warning(N_('Ссылка на тариф не пересекается по дням хождения с ниткой %s'),
                            etree.tounicode(fare_link_el, pretty_print=True))
                continue

            if actual_tariff_mask & added_masks:
                log.warning(
                    N_('Ссылка на тариф пересекается с другими ссылками,'
                       ' какой тариф будет показан не известно %s'),
                    etree.tounicode(fare_link_el, pretty_print=True)
                )

            added_masks |= actual_tariff_mask

            tariff_mask_by_ref_code[ref_code] |= actual_tariff_mask

        for ref_code, tariff_mask in tariff_mask_by_ref_code.iteritems():
            fare_el = self.factory.get_data_provider().get_fare_el(ref_code)

            if fare_el is None:
                log.error(N_('Не наши тарифа с кодом %s в группе %s'),
                          self.thread_el.get('fare_code', "").strip(),
                          self.group_el.get('code', "").strip())
                continue

            yield tariff_mask, fare_el

    def get_order_data(self, price_el, stop_from_el, stop_to_el):
        order_data = None

        if self.need_to_import_order_data():

            order_data = get_sub_tag_text(price_el, 'data', silent=True).strip()

            if not order_data:
                order_data = self.get_default_order_data(stop_from_el, stop_to_el)

        return order_data

    def need_to_import_order_data(self):
        if not xsd_boolean(self.thread_el.get('sales'), default=True):
            return False

        if not self.factory.get_settings().import_order_data:
            return False

        if not self.factory.get_settings().filter_by_group:
            return True

        filter_for_group = self.factory.get_data_provider().get_filter_for_group(self.group_el)

        return filter_for_group.import_order_data

    def get_default_order_data(self, stop_from_el, stop_to_el):
        group_code = self.group_el.get('code', '').strip()
        station_from_code = stop_from_el.get('station_code', '')
        station_to_code = stop_to_el.get('station_code', '')

        order_data = {
            'group_code': group_code,
            'station_from_code': station_from_code,
            'station_to_code': station_to_code,
        }

        return json.dumps(order_data)

    @cache_method_result
    def get_company(self):
        supplier_company = self.get_supplier_company()

        if supplier_company:
            return self.company_finder.find(supplier_company)

    @cache_method_result
    def get_supplier_company(self):
        return CysixCompany.create_from_thread_el(self.thread_el, self.group_el,
                                                  self.factory, self.context, self.settings)

    @cache_method_result
    def get_transport_model(self):
        supplier_transport_model = self.get_supplier_transport_model()

        if supplier_transport_model:
            return self.transport_model_finder.find(supplier_transport_model)

    @cache_method_result
    def get_supplier_transport_model(self):
        return CysixTransportModel.create_from_thread_el(self.thread_el, self.group_el,
                                                         self.factory, self.context, self.settings)

    def get_thread_title_override(self):
        thread_supplier_title = self.thread_el.get('title', '').strip()

        if thread_supplier_title:
            if xsd_boolean(self.thread_el.get('_use_supplier_title'), default=False):
                return thread_supplier_title

        return None


class CysixRTSParser(object):
    """
    Класс используется для формирования станций следования нитки (RTStation).

    При этом нужно правильно заполнить атрибуты времени (возможно с коррекцией):
        tz_arrival, tz_departure, time_zone.

    Также нужно правильно расставить флаги:
        is_fuzzy, is_searchable_to, is_searchable_from, in_station_schedule, in_thread

    Важно!
    Возможна ситуация, когда времена в пути могут отличаться для летнего и зимнего периодов.

    Атрибуты arrival и departure выставляются для коррекции времен следования
    и коррекции времен стоянки, затем они ставятся в None (чтобы они не записывались в базу)
    """

    def __init__(self, supplier_route, factory, thread, naive_start_dt,
                 schedule_context, thread_settings):
        self.supplier_route = supplier_route
        self.factory = factory
        self.thread = thread
        self.path_key = thread.path_key

        self.supplier_rtstations = supplier_route.supplier_rtstations

        self.naive_start_dt = naive_start_dt
        self.start_dt = thread.pytz.localize(naive_start_dt)
        self.start_date = naive_start_dt.date()

        self.settings = thread_settings
        self.schedule_context = schedule_context

        self.two_stage_package = self.factory.package

        self.exclude_from_path_majority_id = StationMajority.objects.get(code='exclude_from_path').id

        self.standard_stop_time = self.factory.get_settings().standard_stop_time

        self.rtstations = None

    def parse_rtstations(self):
        self.init_rtstations()

        self.calculate_shift()

        # Ставим флаги определенные в Station и в TSIThreadStationFlag
        # Делаем это до коррекции т.к. во время коррекции проставляется is_fuzzy
        self.process_flags()

        last_station_old_arrival = self.rtstations[-1].arrival

        self.rtstations = clear_consecutive_duplicate_rtstations(self.rtstations)
        self.correct_stop_time()
        self.correct_departure_and_arrival()
        self.correct_last_rtstation()
        self.correct_arrival_time_by_map()

        self.copy_is_fuzzy_from_supplier_rtstation()
        self.process_fuzzy()
        self.apply_base_stations()

        if not self.is_valid_rtstations():
            raise RaspImportError(_('Не валидная нитка'))

        self.log_if_corrected()

        self.log_if_last_station_corrected(last_station_old_arrival)

        self.calc_tz_departure_and_tz_arrival()
        self.unset_departure_and_arrival()

        if not self.is_enough_visible_rtstations():
            raise RaspImportError(_('Недостаточно видимых станций'))

    def init_rtstations(self):
        self.rtstations = []

        for index, supplier_rtstation in enumerate(self.supplier_rtstations):
            is_middle_rts = 0 < index < (len(self.supplier_rtstations) - 1)

            rts = RTStation()
            rts.supplier_rtstation = supplier_rtstation
            rts.station = supplier_rtstation.station
            rts.distance = supplier_rtstation.distance

            if is_middle_rts:
                rts.is_combined = supplier_rtstation.is_combined()

            self.rtstations.append(rts)

        if len(self.rtstations) < 2:
            raise RaspImportError(_("В нитке '{}' меньше 2 актуальных(привязанных) станций")
                                  .format(self.supplier_route))

    def calculate_shift(self):
        """
        Вычисление вспомогательных параметров arrival и departure, а также сопутствующих для RTStation
        """
        from cysix.two_stage_import.xml_thread import RTSCalculator

        rts_calculator = RTSCalculator(self.schedule_context, self.start_dt, self.naive_start_dt,
                                       self.standard_stop_time, self.thread.time_zone)

        for rts in self.rtstations:
            calculator_result = rts_calculator.calculate_departure_and_arrival(rts.supplier_rtstation)

            rts.time_zone = calculator_result.time_zone
            rts.arrival = calculator_result.arrival
            rts.departure = calculator_result.departure

            # can_add_day_* используются в фильтрах из rts.supplier_rtstation
            rts.supplier_rtstation.can_add_day = calculator_result.can_add_day
            rts.supplier_rtstation.can_add_day_to_arrival = calculator_result.can_add_day_to_arrival
            rts.supplier_rtstation.can_add_day_to_departure = calculator_result.can_add_day_to_departure

    def correct_stop_time(self):
        filter_ = self.factory.get_filter('correct_stop_time')
        self.rtstations = filter_.apply(self.rtstations, self.standard_stop_time)

    def correct_departure_and_arrival(self):
        filter_ = self.factory.get_filter('correct_departure_and_arrival')
        self.rtstations = filter_.apply(self.rtstations, self.standard_stop_time)

    def correct_arrival_time_by_map(self):
        filter_ = self.factory.get_filter('correct_arrival_time_by_map')
        filter_.apply(self.rtstations, self)

    def process_flags(self):
        for rts in self.rtstations:
            tsi_thread_station_flag = self.get_fuzzy_flag(rts.supplier_rtstation.supplier_station)
            self.process_tsi_thread_station_flag(rts, tsi_thread_station_flag)
            self.process_nonstop(rts)
            self.process_technical(rts)

    def process_nonstop(self, rts):
        if rts.supplier_rtstation.is_nonstop():
            rts.is_searchable_to = False
            rts.is_searchable_from = False
            rts.in_station_schedule = False
            log.info('{} is_nonstop'.format(rts.station.title))

    def process_technical(self, rts):
        if rts.supplier_rtstation.is_technical():
            rts.is_searchable_to = False
            rts.is_searchable_from = False
            rts.in_station_schedule = False
            log.info('{} is_technical'.format(rts.station.title))

    def get_fuzzy_flag(self, supplier_station):
        key = supplier_station.get_key_without_context()
        try:
            return TSIThreadStationFlag.objects.get(station_key=key, path_key=self.path_key,
                                                    package=self.two_stage_package)
        except TSIThreadStationFlag.DoesNotExist:
            pass

    def process_tsi_thread_station_flag(self, rts, tsi_thread_station_flag):
        flag_names = ('is_fuzzy', 'is_searchable_to', 'is_searchable_from',
                      'in_station_schedule', 'in_thread')

        for flag_name in flag_names:
            log.debug('Station flag value %s %s %s', rts.station.title, flag_name, getattr(rts.station, flag_name))
            setattr(rts, flag_name, getattr(rts.station, flag_name))

        if tsi_thread_station_flag:
            log.debug(N_('Нашли переопределение флагов для %s'), tsi_thread_station_flag.station_key)

            for flag_name in flag_names:
                flag_value = getattr(tsi_thread_station_flag, flag_name)
                if flag_value is not None:
                    log.debug('Tsi station flags overriding: %s %s %s', rts.station.title, flag_name, flag_value)
                    setattr(rts, flag_name, flag_value)

        # is_fuzzy переопределяется позже
        for flag_name in flag_names[1:]:
            flag_value = getattr(rts.supplier_rtstation, flag_name)()
            if flag_value is not None:
                log.debug('Cysix flags overriding: %s %s %s', rts.station.title, flag_name, flag_value)
                setattr(rts, flag_name, flag_value)

    def process_fuzzy(self):
        """От флага могут отказаться, дублируем поведение через новые флаги"""
        for rts in self.rtstations:
            self.factory.get_filter('fuzzy_flag_interpretation').apply(rts)

    def apply_base_stations(self):
        supplier_route = self.supplier_route
        ignore_base_stations = (
            xsd_boolean(supplier_route.thread_el.get('_ignore_base_stations'), False) or
            not supplier_route.tsi_thread_setting.apply_base_stations
        )
        if ignore_base_stations:
            return

        return apply_base_and_trusted_stations(self.rtstations, self.two_stage_package,
                                               getattr(self.settings, 'current_tsi_group', None))

    def is_valid_rtstations(self):
        for rts in self.rtstations:
            if rts.arrival is None and rts.departure is None:
                return False
        return True

    def add_rtstations_to_thread(self):
        self.thread.rtstations = self.rtstations

    def correct_last_rtstation(self):
        last_rts = self.rtstations[-1]

        if last_rts.arrival is None and last_rts.departure is not None:
            last_rts.arrival = last_rts.departure

        last_rts.departure = None

    def log_if_corrected(self):
        for rts in self.rtstations:
            if hasattr(rts, 'invalid') and rts.invalid:
                log.info(N_('Времена следования были скорректированы.'))

                return

    def log_if_last_station_corrected(self, last_station_origin_arrival):
        rts = self.rtstations[-1]

        if hasattr(rts, 'invalid') and rts.invalid:
            if rts.arrival != last_station_origin_arrival:
                log.info(N_('Для последней станции были изменены времена.'))

    def calc_tz_departure_and_tz_arrival(self):
        """
        Вычисление tz_arrival и tz_departure исходя из arrival и departure
        """
        self.log_tz_times(self.rtstations, _('tz_times до коррекции времен отправления и прибытия'))

        replace_local_time_by_shift_filter = self.factory.get_package_filter_obj('replace_local_time_by_shift')
        if self.rtstations and replace_local_time_by_shift_filter and replace_local_time_by_shift_filter.use:
            self.replace_local_time_by_shift()
        else:
            for rts in self.rtstations:
                rts.tz_departure = self._get_tz_shift_from_absolute_shift(rts.departure, rts.pytz)
                rts.tz_arrival = self._get_tz_shift_from_absolute_shift(rts.arrival, rts.pytz)

        self.log_tz_times(self.rtstations, _('tz_times после коррекции времен отправления и прибытия'))

    def _get_tz_shift_from_absolute_shift(self, minutes_shift, pytz):
        if minutes_shift is None:
            return None

        dt = self.start_dt + timedelta(minutes=minutes_shift)
        tz_dt = dt.astimezone(pytz)
        naive_dt = tz_dt.replace(tzinfo=None)
        return int(timedelta2minutes(naive_dt - self.naive_start_dt))

    def replace_local_time_by_shift(self):
        time_zone = self.rtstations[0].time_zone
        for rts in self.rtstations:
            rts.tz_departure = rts.departure
            rts.tz_arrival = rts.arrival
            rts.time_zone = time_zone

    def log_tz_times(self, rtstations, msg):
        log.info('-------- %s --------', msg)

        for rts in rtstations:
            log.info("%30s a-d: '%s' - '%s', tz: '%s'",
                     rts.station.title, rts.tz_arrival, rts.tz_departure, rts.time_zone)

    def unset_departure_and_arrival(self):
        for rts in self.rtstations:
            rts.departure = None
            rts.arrival = None

    def copy_is_fuzzy_from_supplier_rtstation(self):
        for rts in self.rtstations:
            if not rts.is_fuzzy and rts.supplier_rtstation.is_fuzzy():
                rts.is_fuzzy = True

    def is_enough_visible_rtstations(self):
        visible_rts_count = 0

        for rts in self.rtstations:
            if rts.in_thread:
                visible_rts_count += 1

                if visible_rts_count == 2:
                    return True

        return False


class TransitionExpert(object):
    """
    Класс, который знает все о переводе часов для заданных временных зон
    на конкретную дату-время.
    """

    def __init__(self, timezones, start_dt):
        self.timezones = timezones
        self.start_dt = start_dt
        self.first_transition_dt = None

        if len(timezones) > 1:
            self.first_transition_dt = self._calc_first_transition_dt_or_none()

    @cached_property
    def is_different_duration_possible(self):
        if self.first_transition_dt is None:
            return False

        # TODO возможна ситуация, когда перевод часов будет через год или больше, она нам не интересна
        # даты до и после перевода часов
        before = self.get_safe_dt_before_transition()
        after = self.first_transition_dt + timedelta(days=90)

        before = datetime.combine(before.date(), time(0, 0))
        after = datetime.combine(after.date(), time(0, 0))

        def transition_has_effect(tz1, tz2):
            before_diff = int((tz1.localize(before) - tz2.localize(before)).total_seconds())
            after_diff = int((tz1.localize(after) - tz2.localize(after)).total_seconds())

            return before_diff != after_diff

        first_tz = pytz.timezone(self.timezones[0])

        for timezone in self.timezones[1:]:
            tz = pytz.timezone(timezone)

            if transition_has_effect(first_tz, tz):
                return True

        return False

    def get_safe_dt_before_transition(self, out_tz=None):
        """
        Нужна дата при таких же временных условиях ("до перевода часов"),
        такая, что прибытие на конечную будет заведомо до перевода часов
        """

        before = self.start_dt

        if self.first_transition_dt is not None:
            # нужна дата задолго до перевода часов
            if (self.first_transition_dt.date() - before.date()).days < 30:
                before -= timedelta(days=90)

        if out_tz:
            before = before.astimezone(out_tz)

        return before

    def get_first_transition_dt(self, out_tz=None):
        dt = self.first_transition_dt

        if dt is None:
            return dt

        if out_tz:
            return dt.astimezone(out_tz)

        return dt

    def _calc_first_transition_dt_or_none(self):
        transitions = list(islice(
            heapq.merge(*[self._next_transitions(tz) for tz in self.timezones]), 0, 1
        ))

        if not transitions:
            return None

        return pytz.UTC.localize(transitions[0])

    def _next_transitions(self, timezone):
        tz = pytz.timezone(timezone)

        if not hasattr(tz, '_utc_transition_times'):
            return []

        naive_utc_start_dt = self._dt_to_naive_utc(self.start_dt)

        start_index = bisect_left(tz._utc_transition_times, naive_utc_start_dt)

        return islice(tz._utc_transition_times, start_index, None)

    def _dt_to_naive_utc(self, dt):
        return dt.astimezone(pytz.UTC).replace(tzinfo=None)
