# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from copy import copy
from itertools import ifilter

from django.utils.translation import gettext_noop as N_

from common.utils.caching import cache_method_result_with_exception
from common.utils.progress import PercentageStatus
from cysix.base import safe_parse_xml
from cysix.filters.core.base import Pipeline
from cysix.filters.core.cysix_xml_thread import (SortSupplierRTStationsByDistance, FillCysixXmlThreadPathKey,
                                                 FilterRTStations, MarkThreadWithInvalidGeometryHidden,
                                                 SkipStationWithoutTimesConditional)
from travel.rasp.admin.importinfo.models.two_stage_import import CysixGroupFilter
from travel.rasp.admin.lib.logs import log_context
from travel.rasp.admin.lib.xmlutils import copy_lxml_element
from travel.rasp.admin.scripts.schedule.utils import RaspImportError


log = logging.getLogger(__name__)


class CysixDataProvider(object):
    def __init__(self, factory):
        self.factory = factory
        self.two_stage_package = factory.package
        self.file_provider = factory.get_file_provider()

        self.finder = factory.get_station_finder()

        self.mask_builder = factory.get_mask_builder()

    def get_xml_thread_iter_for_import(self):
        groups = list(self.get_groups_iter(self.is_group_accepted_for_import))
        total_thread_els = self.get_thread_el_count(groups)
        status = PercentageStatus(total_thread_els, log)

        pipeline = self.get_middle_import_pipeline() + Pipeline([
            FilterRTStations(),
            MarkThreadWithInvalidGeometryHidden(),
        ])

        def xml_thread_generator():
            for xml_thread in self._get_xml_thread_iter(groups):
                status.step()
                yield xml_thread

        return pipeline.process(xml_thread_generator())

    def get_middle_import_pipeline(self):
        pipeline = Pipeline()
        return pipeline + Pipeline([
            SkipStationWithoutTimesConditional(),
            SortSupplierRTStationsByDistance(),
            FillCysixXmlThreadPathKey()
        ])

    def get_raw_xml_thread_iter_for_middle_import(self):
        groups = list(self.get_groups_iter(self.is_group_accepted_for_middle_import))

        for xml_thread in self._get_xml_thread_iter(groups):
            yield xml_thread

    def get_xml_thread_iter_for_middle_import(self):
        groups = list(self.get_groups_iter(self.is_group_accepted_for_middle_import))
        total_thread_els = self.get_thread_el_count(groups)
        status = PercentageStatus(total_thread_els, log)

        def xml_thread_generator():
            for xml_thread in self._get_xml_thread_iter(groups):
                status.step()
                yield xml_thread

        pipeline = self.get_middle_import_pipeline()

        return pipeline.process(xml_thread_generator())

    def get_thread_el_count(self, groups_iter):
        thread_el_count = 0
        for group_el, group_context, group_settings in groups_iter:
            thread_els = group_el.findall('./threads/thread')
            thread_el_count += len(thread_els)

        return thread_el_count

    def _get_xml_thread_iter(self, groups_iter):
        from cysix.two_stage_import.xml_thread import CysixXmlThread

        for group_el, group_context, group_settings in groups_iter:
            group_el_only = copy_lxml_element(group_el, top_element_only=True)

            group_el_only.sourceline = group_el.sourceline

            thread_els = group_el.findall('./threads/thread')

            self.log_warning_if_group_empty(group_el, thread_els)

            for thread_el in thread_els:
                factory = self.factory.get_group_factory(group_el.get('code'))
                yield CysixXmlThread.create(thread_el, group_el_only, factory, group_context, group_settings)

    def get_groups_iter(self, is_group_accepted_func):
        log.info(N_('---------------- Импорт групп -----------------------'))

        accepted_groups = []

        for group_el, group_context, group_settings in self.get_group_iter():
            code = group_el.get('code', '').strip()
            title = group_el.get('title', '').strip()

            if is_group_accepted_func(group_el):
                accepted_groups.append((group_el, group_context, group_settings))
                log.info(N_('Импортируем группу %s: %s'), title, code)

                log.info(N_('\tИмпортировать данные для покупки билета %s'),
                         self.is_group_accepted_for_import_order_data(group_el))

                log.info(N_('\tИспользовать информацию о нитке в генерации кода станции %s'),
                         self.is_group_accepted_for_use_thread_in_station_code(group_el))

            else:
                log.info(N_('Пропускаем группу %s: %s'), title, code)

        log.info('----------------')

        for g in accepted_groups:
            yield g

    def is_group_accepted_for_import(self, group_el):
        settings = self.factory.get_settings()

        if settings.filter_by_group:
            group_filter = self.get_filter_for_group(group_el)

            return group_filter.tsi_import_available

        return True

    def is_group_accepted_for_middle_import(self, group_el):
        settings = self.factory.get_settings()

        if settings.filter_by_group:
            group_filter = self.get_filter_for_group(group_el)

            return group_filter.tsi_middle_available

        return True

    def is_group_accepted_for_import_order_data(self, group_el):
        settings = self.factory.get_settings()

        if settings.filter_by_group:
            group_filter = self.get_filter_for_group(group_el)

            return group_filter.import_order_data

        return True

    def is_group_accepted_for_use_thread_in_station_code(self, group_el):
        settings = self.factory.get_settings()

        if settings.filter_by_group:
            group_filter = self.get_filter_for_group(group_el)

            return group_filter.use_thread_in_station_code

        return True

    def log_warning_if_group_empty(self, group_el, thread_els):
        if len(thread_els) == 0:
            group_title = group_el.get('title', '').strip()
            group_code = group_el.get('code', '').strip()

            log.warning(N_("Файл общего xml-формата содержит пустую группу '%s' - '%s'"),
                        group_title, group_code)

    def get_filter_for_group(self, group_el):
        code = group_el.get('code', "").strip()
        title = group_el.get('title', "").strip()
        try:
            group_filter = CysixGroupFilter.objects.get(code=code, package=self.two_stage_package)

        except CysixGroupFilter.DoesNotExist:
            group_filter = CysixGroupFilter(code=code, package=self.two_stage_package, title=title)
            group_filter.save()

        return group_filter

    def get_supplier_path_iter(self):
        from travel.rasp.admin.scripts.schedule.utils import SupplierPath

        for xml_thread in ifilter(
            self.middle_base_entity_filter,
            self.get_xml_thread_iter_for_middle_import()
        ):
            path_for_middle_import = None

            with log_context(log, unicode(xml_thread)):
                try:
                    path_for_middle_import = SupplierPath([
                        srts.supplier_station
                        for srts in xml_thread.supplier_rtstations
                    ])

                except RaspImportError, e:
                    log.error(N_('Не смогли построить путь: %s'), unicode(e))

            if path_for_middle_import:
                yield path_for_middle_import

    def get_station_el_by_ref_code(self, ref_code):
        return self.get_station_el_cache().get(ref_code)

    @cache_method_result_with_exception
    def get_station_el_cache(self):
        station_els = {}

        for group_el, group_context, group_settings in self.get_group_iter():
            group_code = group_el.get('code', '').strip()

            for station_el in group_el.findall('./stations/station'):
                context = copy(group_context)
                context.update_from(station_el)

                station_el = copy_lxml_element(station_el)

                ref_code = '_'.join([group_code, context.station_code_system, station_el.get('code', '').strip()])

                station_els[ref_code] = station_el

        return station_els

    def get_carrier_el_by_ref_code(self, ref_code):
        return self.get_carrier_el_cache().get(ref_code)

    @cache_method_result_with_exception
    def get_carrier_el_cache(self):
        carrier_els = {}

        for group_el, group_context, group_settings in self.get_group_iter():
            group_code = group_el.get('code', '').strip()

            for carrier_el in group_el.findall('./carriers/carrier'):
                context = copy(group_context)
                context.update_from(carrier_el)

                carrier_el = copy_lxml_element(carrier_el)

                ref_code = '_'.join([group_code, context.carrier_code_system, carrier_el.get('code', '').strip()])

                carrier_els[ref_code] = carrier_el

        return carrier_els

    def get_vehicle_el_by_ref_code(self, ref_code):
        return self.get_vehicle_el_cache().get(ref_code)

    @cache_method_result_with_exception
    def get_vehicle_el_cache(self):
        vehicle_els = {}

        for group_el, group_context, group_settings in self.get_group_iter():
            group_code = group_el.get('code', '').strip()

            for vehicle_el in group_el.findall('./vehicles/vehicle'):
                context = copy(group_context)
                context.update_from(vehicle_el)

                vehicle_el = copy_lxml_element(vehicle_el)

                ref_code = '_'.join([group_code, context.vehicle_code_system, vehicle_el.get('code', '').strip()])

                vehicle_els[ref_code] = vehicle_el

        return vehicle_els

    def get_fare_el_with_context_and_settings(self, ref_code):
        return self.get_fare_el_cache().get(ref_code, (None, None, None))

    def get_fare_el(self, ref_code):
        return self.get_fare_el_with_context_and_settings(ref_code)[0]

    @cache_method_result_with_exception
    def get_fare_el_cache(self):
        fare_els = {}

        for group_el, group_context, group_settings in self.get_group_iter():
            group_code = group_el.get('code', '').strip()

            for fare_el in group_el.findall('./fares/fare'):
                fare_el = copy_lxml_element(fare_el)

                ref_code = '_'.join([group_code, fare_el.get('code', '').strip()])

                fare_els[ref_code] = fare_el, group_context, group_settings

        return fare_els

    def get_supplier_route_iter(self):
        for xml_thread in ifilter(self.import_entity_filter, self.get_xml_thread_iter_for_import()):
            supplier_routes = []

            with log_context(log, unicode(xml_thread)):
                log.info(N_('Начинаем разбор'))

                try:
                    supplier_routes = xml_thread.get_supplier_routes()

                except RaspImportError as e:
                    log.error(N_('Пропускаем маршрут: %s'), unicode(e))

                    continue

                except Exception:
                    log.exception(N_('Ошибка разбора маршрута'))

            for supplier_route in ifilter(self.supplier_route_filter, supplier_routes):
                yield supplier_route

    def get_group_iter(self):
        global_context = self.factory.get_context()
        global_settings = self.factory.get_settings()

        tree = safe_parse_xml(self.file_provider.get_cysix_file())

        channel = tree.getroot()

        global_context.update_from(channel)

        for group_el in channel.findall('./group'):
            group_settings = copy(global_settings)
            group_context = copy(global_context)

            group_context.update_from(group_el)

            if global_settings.filter_by_group:
                group_filter = self.get_filter_for_group(group_el)
                group_settings.current_tsi_group = group_filter
                group_settings.use_thread_in_station_code = group_filter.use_thread_in_station_code

            yield group_el, group_context, group_settings

    def supplier_route_filter(self, supplier_route):
        if supplier_route.tsi_thread_setting.allow_to_import:
            return True

        log.info(N_('%s: Не импортируем рейс с таким набором станций'), supplier_route)
        return False

    def import_entity_filter(self, xml_thread):
        return True

    def middle_base_entity_filter(self, xml_thread):
        return True
