# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from importlib import import_module

from django.utils.translation import ugettext as _, gettext_noop as N_

from cysix.base import CysixCompanyFinder, CysixTransportModelFinder, CysixImportSettings, CysixContext
from cysix.two_stage_import import CysixXmlPackageFileProvider, CysixHTTPFileProvider, CysixRoute
from cysix.two_stage_import.data_provider import CysixDataProvider
from cysix.two_stage_import.importer import CysixTwoStageImporter
from cysix.two_stage_import.route_importer import CysixTSIRouteImporter
from travel.rasp.admin.importinfo.two_stage_import import TwoStageImportStationFinder
from common.utils.caching import cache_method_result
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.mask_builders import MaskBuilder
from travel.rasp.admin.lib.import_bounds import import_bounds


log = logging.getLogger(__name__)


class CysixTSIFactory(object):
    def __init__(self, package):
        self.package = package
        self.max_forward_days = package.tsisetting.max_forward_days

    @cache_method_result
    def get_two_stage_importer(self):
        return CysixTwoStageImporter(self)

    @cache_method_result
    def get_company_finder(self):
        return CysixCompanyFinder(self.package.supplier, self.get_settings())

    @cache_method_result
    def get_transport_model_finder(self):
        return CysixTransportModelFinder(self.package.supplier)

    @cache_method_result
    def get_data_provider(self):
        return CysixDataProvider(self)

    def get_package_file_provider(self):
        log.info(N_(u'Берем данные из приложенного файла'))

        return CysixXmlPackageFileProvider(self.package)

    def get_download_file_provider(self):
        log.info(N_(u'Берем данные по %s'), self.package.url)

        return CysixHTTPFileProvider(self.package)

    @cache_method_result
    def get_settings(self):
        return CysixImportSettings(self.package.tsisetting)

    @cache_method_result
    def get_context(self):
        return CysixContext()

    def get_route_importer(self):
        return CysixTSIRouteImporter(self)

    def get_supplier_route_class(self):
        return CysixRoute

    @cache_method_result
    def get_filter(self, filter_code, *args, **kwargs):
        try:
            module = import_module('cysix.filters.{}'.format(filter_code))

        except ImportError:
            raise RaspImportError(_(u'Не нашли фильтр с кодом {}').format(filter_code))

        params = self.package.get_filter_params(filter_code)

        return module.Filter(params, *args, **kwargs)

    @cache_method_result
    def get_package_filter_obj(self, filter_code):
        return self.package.get_filter(filter_code)

    @cache_method_result
    def get_station_finder(self):
        return TwoStageImportStationFinder(self.package.supplier)

    @cache_method_result
    def get_file_provider(self):
        if self.package.package_file:
            log.info(N_(u'Файл приложен к пакету берем данные из него'))

            file_provider = self.get_package_file_provider()
        else:
            log.info(N_(u'Файл не приложен к пакету пробуем скачать данные'))

            file_provider = self.get_download_file_provider()

        if not file_provider:
            raise RaspImportError(_(u'Файл не приложен и скачивание не поддерживается. '
                                    u'Невозможно взять данные.'))

        return file_provider

    @cache_method_result
    def get_mask_builder(self, start_date=None, end_date=None):
        default_start_date, default_end_date = import_bounds(forward=self.max_forward_days)

        start_date = start_date or default_start_date
        if start_date < default_start_date:
            start_date = default_start_date

        end_date = end_date or default_end_date
        if end_date > default_end_date:
            end_date = default_end_date

        return MaskBuilder(start_date, end_date, log=log)

    @cache_method_result
    def get_group_factory(self, group_code):
        return CysixTSIGroupFactory(self, group_code)


class CysixTSIGroupFactory(object):
    def __init__(self, base_factory, group_code):
        self.base_factory = base_factory
        self.group_code = group_code
        self.package = base_factory.package
        self.max_forward_days = base_factory.max_forward_days

    def get_two_stage_importer(self):
        return self.base_factory.get_two_stage_importer()

    def get_company_finder(self):
        return self.base_factory.get_company_finder()

    def get_transport_model_finder(self):
        return self.base_factory.get_transport_model_finder()

    def get_data_provider(self):
        return self.base_factory.get_data_provider()

    def get_package_file_provider(self):
        return self.base_factory.get_package_file_provider()

    def get_download_file_provider(self):
        return self.base_factory.get_download_file_provider()

    def get_settings(self):
        return self.base_factory.get_settings()

    def get_context(self):
        return self.base_factory.get_context()

    def get_route_importer(self):
        return self.base_factory.get_route_importer()

    def get_supplier_route_class(self):
        return self.base_factory.get_supplier_route_class()

    def _get_package_group_filter(self, filter_code):
        from cysix.models import PackageGroupFilter
        return PackageGroupFilter.objects.filter(
            cysix_group_filter__use_package_group_filters=True,
            cysix_group_filter__code=self.group_code,
            cysix_group_filter__package=self.package,
            filter__code=filter_code).first()

    @cache_method_result
    def get_filter(self, filter_code, *args, **kwargs):
        package_group_filter = self._get_package_group_filter(filter_code)
        if package_group_filter:
            log.info('Переопределяем настройки пакета для группы {}. Фильтр: {}'.format(self.group_code, filter_code))

            try:
                module = import_module('cysix.filters.{}'.format(filter_code))
            except ImportError:
                raise RaspImportError(_(u'Не нашли фильтр с кодом {}').format(filter_code))

            params = package_group_filter.parameters.get_parameters_as_dict() if package_group_filter.use else None

            return module.Filter(params, *args, **kwargs)

        return self.base_factory.get_filter(filter_code, *args, **kwargs)

    @cache_method_result
    def get_package_filter_obj(self, filter_code):
        return self._get_package_group_filter(filter_code) or self.base_factory.get_package_filter_obj(filter_code)

    def get_station_finder(self):
        return self.base_factory.get_station_finder()

    def get_file_provider(self):
        return self.base_factory.get_file_provider()

    def get_mask_builder(self, start_date=None, end_date=None):
        return self.base_factory.get_mask_builder(start_date=start_date, end_date=end_date)

    def get_group_factory(self, group_code):
        if group_code == self.group_code:
            return self

        raise RuntimeError("get_group_factory can't be used with CysixTSIGroupFactory")
