# coding: utf-8

from __future__ import unicode_literals

import re
from functools import reduce

from django.utils.translation import ugettext as _

from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.to_python_parsers import get_date


class CysixMaskParser(object):
    @classmethod
    def parse_mask(cls, mask_defining_el, mask_builder):
        period_start_date = get_date(mask_defining_el.get('period_start_date', u'').strip(),
                                     silent=True)
        period_end_date = get_date(mask_defining_el.get('period_end_date', u'').strip(),
                                   silent=True)

        positive_mask = cls.parser_mask_text(
            mask_defining_el.get('days', u'').strip(),
            period_start_date,
            period_end_date,
            mask_builder
        )

        negative_mask = cls.parser_mask_text(
            mask_defining_el.get('exclude_days', u'').strip(),
            period_start_date,
            period_end_date,
            mask_builder
        )

        result_mask = positive_mask - negative_mask

        return result_mask

    @classmethod
    def parser_mask_text(cls, mask_text, start_date, end_date, mask_builder):
        parts = mask_text.replace(u'|', u';').split(u';')

        return reduce(lambda x, y: x | y, [
            cls.parser_mask_text_part(part, start_date, end_date, mask_builder)
            for part in parts
        ])

    date_mask_re = re.compile(r'^\d{4}-\d\d-\d\d$', re.U + re.I)
    days_of_week_mask_re = re.compile(r'^[0-7]+$', re.U + re.I)
    every_number_day_re = re.compile(r'^через\s+(\d+).*$', re.U + re.I)
    whitespace_re = re.compile(r'\s+', re.U + re.I)

    @classmethod
    def parser_mask_text_part(cls, mask_text, start_date, end_date, mask_builder):
        mask_text = mask_text.strip().lower().replace(u'ё', u'е')

        mask_text = cls.whitespace_re.sub(u' ', mask_text)

        if not mask_text:
            return mask_builder.empty_mask()

        if mask_text in ('ежедневно', 'daily'):
            return mask_builder.daily_mask(start_date, end_date)

        elif mask_text in ('четные', 'even'):
            return mask_builder.even_mask(start_date, end_date)

        elif mask_text in ('нечетные', 'odd'):
            return mask_builder.odd_mask(start_date, end_date)

        elif mask_text in ('рабочие', 'рабочие дни', 'будние'):
            return mask_builder.ycal_workdays_mask(start_date, end_date)

        elif mask_text in ('выходные дни', 'выходные'):
            return mask_builder.ycal_weekends_and_holidays_mask(start_date, end_date)

        elif mask_text == 'через день':
            if not start_date:
                raise RaspImportError(_('Через день всегда должен сопровождаться модификатором "с <дата>"'))

            return mask_builder.through_the_day_mask(start_date, end_date)

        elif cls.every_number_day_re.match(mask_text):
            day_in_gap = int(cls.every_number_day_re.match(mask_text).group(1))

            if not start_date:
                raise RaspImportError(_('Через n дней всегда должен сопровождаться модификатором "с <дата>"'))

            return mask_builder.gap_days_mask(day_in_gap, start_date, end_date)

        elif cls.days_of_week_mask_re.match(mask_text):
            mask_text = mask_text.replace('0', '7')

            return mask_builder.mask_from_days_of_week(mask_text, start_date, end_date)

        elif cls.date_mask_re.match(mask_text):
            return mask_builder.one_day_mask(get_date(mask_text), start_date, end_date)

        else:
            raise RaspImportError(_('Не поддеживаемый формат, маски дней хождений "%s"'), mask_text)
