# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
import smtplib
from collections import defaultdict
from itertools import combinations, product, islice

import six
from django.core.urlresolvers import reverse
from django.db.models import Q
from django.utils.http import urlencode
from django.utils.translation import gettext_noop as N_

from common.models.schedule import RThread, Route
from common.models.tariffs import ThreadTariff
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_method_result
from common.utils.progress import PercentageStatus
from cysix.base import CriticalImportError
from cysix.two_stage_import.utils import SettlementIdsByStationIdGetter
from travel.rasp.admin.importinfo.models import BlackList, OriginalThreadData
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.report import ScheduleImportReporter
from travel.rasp.admin.scripts.schedule.utils.route_loader import get_simple_schedule_validator, PackageRouteUpdater
from travel.rasp.admin.www.utils.mysql import fast_delete_tariffs_by_uids


log = logging.getLogger(__name__)


class CysixTSIRouteImporter(object):
    def __init__(self, factory):
        self.factory = factory
        self.data_provider = self.factory.get_data_provider()

        self.updater = self.create_updater()
        self.validator = self.get_schedule_validator()
        self.updater.add_presave_thread_hook(self.validator.validate_thread)

        self.tariffs = defaultdict(dict)

    def create_updater(self):
        package = self.factory.package
        remove_old_routes = not package.tsisetting.update_threads

        return PackageRouteUpdater.create_route_updater(package, remove_old_routes=remove_old_routes,
                                                        update_fields=('hidden', 'has_extrapolatable_mask'))

    def do_import(self):
        self.clean_original_data()

        self.load_routes()

        self.finalize_loading()

        self.bind_original_data()

        self.update_tariffs()

        self.after_import()

        self.log_success()

        self.send_report()

    def load_routes(self):
        other_packages_route_uids = set(Route.objects.filter(
            (~Q(two_stage_package=self.factory.package) | Q(two_stage_package__isnull=True)) &
            (Q(rthread__isnull=True) | Q(rthread__supplier=self.factory.package.supplier))
        ).values_list('route_uid', flat=True))

        for supplier_route, route in self.get_supplier_route_iter():
            try:
                if route.route_uid in other_packages_route_uids:
                    otheroute = Route.objects.get(route_uid=route.route_uid)
                    if otheroute.two_stage_package_id:
                        raise CriticalImportError('Маршрут %s обнаружен в другом пакете %s %s с id %s',
                                                  route.route_uid, otheroute.two_stage_package_id,
                                                  otheroute.two_stage_package.title,
                                                  otheroute.id)
                    else:
                        raise CriticalImportError('Маршрут %s обнаружен беспакетным с id %s',
                                                  route.route_uid, otheroute.id)

                self.updater.add_route(route)

                thread_uid = route.threads[0].uid

                self.update_orig_data(thread_uid, supplier_route)

                if self.factory.get_settings().import_tariffs:
                    try:
                        self.collect_tariffs(thread_uid, supplier_route)
                    except RaspImportError as e:
                        log.error(N_('Пропускаем тарифы для %s %s'), supplier_route, e)

            except RaspImportError as e:
                self.log_rasp_import_error(supplier_route, e)
            except CriticalImportError:
                raise
            except Exception:
                log.exception('Неожиданная ошибка разбора маршрута')

    def finalize_loading(self):
        self.updater.update()

    def get_affected_routes(self):
        return Route.objects.filter(two_stage_package=self.factory.package)

    def update_tariffs(self):
        from travel.rasp.admin.importinfo.two_stage_import.tariffs import gen_tariffs_from_variants

        log.info(N_('Обновляем тарифы'))
        log.info(N_('Очищаем все старые тарифы'))

        old_tariff_thread_uids = list(set(self.tariffs.keys()) | self.updater.get_affected_thread_uids())

        fast_delete_tariffs_by_uids(old_tariff_thread_uids)

        log.info(N_('Удалили все старые тарифы'))

        log.info(N_('Загружаем новые тарифы для %s ниток'), len(self.tariffs))

        self._tariff_count = 0

        def tariff_generator():
            for uid, tariff_variants in self.tariffs.iteritems():
                for variant, mask in tariff_variants.iteritems():
                    try:
                        thread = RThread.objects.get(uid=uid)
                    except RThread.DoesNotExist:
                        continue

                    for t in gen_tariffs_from_variants(thread, variant, mask):
                        self._tariff_count += 1
                        yield t

        tariff_gen = tariff_generator()
        status = PercentageStatus(len(self.tariffs), log, title='Загрузка тарифов')

        while True:
            tariff_bulk = list(islice(tariff_gen, 5000))
            if not tariff_bulk:
                break

            status.step(len(tariff_bulk))

            ThreadTariff.objects.bulk_create(tariff_bulk)

        log.info(N_('Загрузили новые тарифы %s'), self._tariff_count)

    def collect_tariffs(self, thread_uid, supplier_route):
        tariff_variants = supplier_route.get_tariffs_variants()

        if not tariff_variants:
            return

        log.info(N_('Собираем цены для %s'), thread_uid)

        collected_variants = self.tariffs[thread_uid]
        for tariff_variant, mask in tariff_variants.iteritems():
            if tariff_variant in collected_variants:
                collected_variants[tariff_variant] |= mask
            else:
                collected_variants[tariff_variant] = mask

    def after_import(self):
        self.extrapolate()

        self.set_comment()

        self.fill_actual_direction()

    def extrapolate(self):
        routes = self.get_affected_routes().filter(script_protected=False)

        filter_ = self.factory.get_filter('text_schedule_and_extrapolation',
                                          max_forward_days=self.factory.max_forward_days)

        if filter_.enabled:
            log.info(N_('===== Запускаем фильтр подсчета дней хождений и экстраполяцию'))

            for thread in RThread.objects.filter(route__in=routes):
                filter_.apply(thread)

            log.info(N_('===== Отработал фильтр подсчета дней хождений и экстраполяцию'))
        else:
            log.info(N_('===== Пропускаем фильтр подсчета дней хождений и экстраполяцию'))

    def set_comment(self):
        filter_ = self.factory.get_filter('thread_comment')

        if filter_.params is not None:
            threads = RThread.objects.filter(route__two_stage_package=self.factory.package)

            log.info(N_('Добавляем комментарий ко всем %s ниткам пакета'), threads.count())

            for thread in threads:
                filter_.apply(thread)

                thread.save()

    def fill_actual_direction(self):
        from order.models import Partner, ActualDirection

        if self.factory.package.supplier.code not in ActualDirection.supplier_to_partner:
            return

        log.info(N_('Начинаем обновление таблицы актуальных направлений.'))

        partner_code = ActualDirection.supplier_to_partner[self.factory.package.supplier.code]
        partner = Partner.objects.get(code=partner_code)

        directions = set()
        stations = set()

        threads = RThread.objects.filter(route__two_stage_package_id=self.factory.package.id)\
                                 .only('id')\
                                 .order_by()

        for thread in threads:
            thread_stations = thread.get_stations()

            stations |= set(thread_stations)

            directions |= set(combinations(thread_stations, 2))

        log.info('  Обработали нитки')

        settlement_ids_by_station_id = SettlementIdsByStationIdGetter(stations, default=[None])

        log.info('  Достали города из базы')

        ActualDirection.objects.filter(partner__code=partner_code).delete()

        log.info('  Почистили ActualDirection')

        for station_from, station_to in directions:
            settlement_from_ids = settlement_ids_by_station_id.get(station_from.id)
            settlement_to_ids = settlement_ids_by_station_id.get(station_to.id)

            # Самый распрастраненный случай можно обработать бысто, без вложенных циклов
            if len(settlement_from_ids) == 1 and len(settlement_to_ids) == 1:
                ActualDirection(
                    partner=partner,
                    station_from_id=station_from.id,
                    station_to_id=station_to.id,
                    settlement_from_id=settlement_from_ids[0],
                    settlement_to_id=settlement_to_ids[0]
                ).save()

            else:
                for settlement_from_id, settlement_to_id in product(settlement_from_ids, settlement_to_ids):
                    ActualDirection(
                        partner=partner,
                        station_from_id=station_from.id,
                        station_to_id=station_to.id,
                        settlement_from_id=settlement_from_id,
                        settlement_to_id=settlement_to_id
                    ).save()

        log.info(N_('Таблица актуальных направлений успешно обновлена'))

    def get_supplier_route_iter(self):
        for supplier_route in self.data_provider.get_supplier_route_iter():
            try:
                if self.in_black_list(supplier_route):
                    log.info('Маршрут %s попал в черный список', supplier_route)
                    continue

                yield supplier_route, supplier_route.get_route()

            except RaspImportError as e:
                self.log_rasp_import_error(supplier_route, e)

            except Exception:
                log.exception('Неожиданная ошибка разбора маршрута')

    def in_black_list(self, supplier_route):
        for bl in self.get_black_list_entries():
            if bl.match_supplier_route(supplier_route):
                return True

        return False

    @cache_method_result
    def get_black_list_entries(self):
        return list(BlackList.objects.filter(supplier=self.factory.package.supplier))

    def get_schedule_validator(self):
        return get_simple_schedule_validator(log)

    def send_report(self):
        reporter = ScheduleImportReporter.create_import_reporter_by_tsi_package(self.factory.package)

        self.updater.add_report_to_reporter(reporter)
        self.validator.add_report_to_reporter(reporter)
        self.factory.get_station_finder().add_report_to_reporter(reporter)

        try:
            reporter.send_report()

        except smtplib.SMTPException as e:
            log.exception(N_('Ошибка при отправке сообщения. %s'), e)

            self.send_only_link_to_log()

    def send_only_link_to_log(self):
        reporter = ScheduleImportReporter.create_import_reporter_by_tsi_package(self.factory.package)

        request = environment.get_request()

        log_code = request.GET.get('_show_log', 'import_package')

        show_log_url = reverse('admin:importinfo_twostageimportpackage_show_log',
                               args=[self.factory.package.id])

        show_log_url = request.build_absolute_uri(show_log_url) + '?' + urlencode({'log_code': log_code})

        save_log_url = reverse('admin:importinfo_twostageimportpackage_save_log',
                               args=[self.factory.package.id])

        save_log_url = request.build_absolute_uri(save_log_url) + '?' + urlencode({'log_code': log_code})

        reporter.add_report('Не удалось отправить сообщение со всеми ошибками.',
                            'Лог файл можно посмотреть здесь: {}.\n'
                            'Или скачать {}.'.format(show_log_url, save_log_url))

        reporter.send_report()

    def clean_original_data(self):
        OriginalThreadData.fast_delete_by_package_id(self.data_provider.two_stage_package.id, log)

    def bind_original_data(self):
        uid_to_id = {
            id_: uid
            for id_, uid in (
                RThread.objects
                .filter(route__two_stage_package=self.data_provider.two_stage_package)
                .values_list('id', 'uid')
            )
        }

        for id_, uid in uid_to_id.iteritems():
            OriginalThreadData.objects.filter(thread_uid=uid).update(thread=id_)

    def update_orig_data(self, thread_uid, supplier_route):
        orig_data, __ = OriginalThreadData.objects.get_or_create(
            thread_uid=thread_uid,
            package=self.data_provider.two_stage_package
        )

        if hasattr(supplier_route, 'raw_data'):
            orig_data.raw = (orig_data.raw or "") + supplier_route.raw_data
        if hasattr(supplier_route, 'cysix_data'):
            orig_data.cysix = (orig_data.cysix or "") + supplier_route.cysix_data

        if orig_data.raw or orig_data.cysix:
            orig_data.save()

    def log_rasp_import_error(self, supplier_route, exception):
        log.error(N_('Пропускаем %s: %s'), supplier_route, six.text_type(exception))

    def log_success(self):
        log.info(N_('Успешно импортировали пакет %s %s от поставщика %s %s'),
                 self.factory.package.id, self.factory.package.title,
                 self.factory.package.supplier.title, self.factory.package.supplier.code)
