# coding: utf-8
from datetime import timedelta

from django.utils.functional import cached_property
from django.utils.translation import ugettext as _

from travel.rasp.library.python.common23.date import environment
from cysix.two_stage_import import CysixMaskParser
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from travel.rasp.admin.scripts.schedule.utils.to_python_parsers import get_date, get_time


class CysixSchedule(object):
    def __init__(self, xml_thread, schedule_el, local_start_time, schedule_context, schedule_pytz, comment):
        self.xml_thread = xml_thread
        self.schedule_el = schedule_el
        self.is_interval = False

        self.local_start_time = local_start_time
        self.schedule_context = schedule_context

        self.period_int = None
        self.interval_start_time = None
        self.interval_end_time = None

        self.comment = comment or schedule_context.schedule_comment
        self.density = u''

        self.pytz = schedule_pytz

        self.period_end_date = get_date(schedule_el.get('period_end_date', u'').strip(), silent=True)

    def is_extrapolatable(self):
        if not bool(self.period_end_date):
            return True

        last_mask_day = (environment.today() + timedelta(self.xml_thread.get_settings().max_forward_days))

        return self.period_end_date >= last_mask_day

    @cached_property
    def local_mask(self):
        settings = self.xml_thread.get_settings()

        mask_builder = self.xml_thread.factory.get_mask_builder(start_date=settings.start_date,
                                                                end_date=settings.end_date)

        return CysixMaskParser.parse_mask(self.schedule_el, mask_builder)

    @classmethod
    def day_based_schedule_iter(cls, xml_thread, schedule_el, schedule_context, supplier_rtstations):
        schedule_pytz = supplier_rtstations[0].get_pytz(schedule_context)

        comment = schedule_el.get('comment', u'').strip()

        time_strings = schedule_el.get('times', u'').strip().split(';')
        time_strings = [t for t in time_strings if t]

        if not cls.has_times(supplier_rtstations):
            for time_str in time_strings:
                local_start_time = get_time(time_str)

                yield cls(xml_thread, schedule_el, local_start_time, schedule_context, schedule_pytz, comment)

        else:
            first_timezone = supplier_rtstations[0].get_pytz(schedule_context).zone

            if schedule_pytz.zone != first_timezone:
                raise RaspImportError(_(
                    u'Когда хотя бы для одной станции следования задано время в ЧЧ:ММ, '
                    u'то временные зоны нитки(расписания) и первой станции должны совпадать'
                ))

            if len(time_strings) > 1:
                raise RaspImportError(_(
                    u'Когда хотя бы для одной станции следования задано время в ЧЧ:ММ, '
                    u'то должно быть не более одного времени в times.'
                ))

            elif len(time_strings) == 0:
                local_start_time = supplier_rtstations[0].departure_time

                if local_start_time is None:
                    local_start_time = supplier_rtstations[0].arrival_time

                if local_start_time is None:
                    raise RaspImportError(_(
                        u'Когда хотя бы для одной станции следования задано время в ЧЧ:ММ, '
                        u'и не задано times, '
                        u'то должно быть задано время для первой станции.'
                    ))

                yield cls(xml_thread, schedule_el, local_start_time, schedule_context, schedule_pytz, comment)

            else:  # len(time_strings) == 1
                local_start_time = get_time(time_strings[0])

                if supplier_rtstations[0].has_time():
                    if not cls.supplier_rtstation_has_this_time(supplier_rtstations[0], local_start_time):
                        raise RaspImportError(_(
                            u'Время отправления с первой станции '
                            u'должно совпадать с временем в times.'
                        ))

                yield cls(xml_thread, schedule_el, local_start_time, schedule_context, schedule_pytz, comment)

    @classmethod
    def create_interval_schedule(cls, xml_thread, schedule_el, schedule_context, supplier_rtstations):
        schedule_pytz = supplier_rtstations[0].get_pytz(schedule_context)

        period_int = schedule_el.get('period_int', u'').strip()

        try:
            period_int = int(period_int)

        except ValueError:
            raise RaspImportError(
                _(u'Не удалось разобрать расписание, не правильно указан period_int "{}".')
                .format(period_int)
            )

        interval_start_time = schedule_el.get('period_start_time', u'').strip()
        interval_end_time = schedule_el.get('period_end_time', u'').strip()

        if not interval_start_time or not interval_end_time:
            raise RaspImportError(_(
                u'Для интервального рейса должны быть указаны '
                u'period_start_time и period_end_time.'
            ))

        if cls.has_times(supplier_rtstations):
            raise RaspImportError(_(
                u'Для интервального рейса нельзя указывать времена '
                u'для станций следования в ЧЧ:ММ.'
            ))

        interval_start_time = get_time(interval_start_time)
        interval_end_time = get_time(interval_end_time)

        comment = schedule_el.get('comment', u'').strip()
        schedule = cls(xml_thread, schedule_el, interval_start_time, schedule_context, schedule_pytz, comment)

        schedule.is_interval = True

        schedule.period_int = period_int
        schedule.interval_start_time = interval_start_time
        schedule.interval_end_time = interval_end_time

        schedule.density = schedule_el.get('density', u'').strip() or schedule_context.schedule_density

        return schedule

    @classmethod
    def has_times(cls, supplier_rtstations):
        for s_rts in supplier_rtstations:
            if s_rts.has_time():
                return True

        return False

    @classmethod
    def supplier_rtstation_has_this_time(cls, supplier_rtstation, time):
        arrival = supplier_rtstation.arrival_time
        departure = supplier_rtstation.departure_time

        if departure is not None:
            return departure == time

        else:
            return arrival == time
