# -*- coding: utf-8 -*-

from collections import defaultdict

from common.models.geo import Station2Settlement


def group_description(group_el):
    return u'<Group title="%s" code="%s" sourceline=%s>' % (
        group_el.get('title', u'').strip(),
        group_el.get('code', u'').strip(),
        group_el.sourceline
    )


_non_specified = object()


def xsd_boolean(text, default=_non_specified):
    text = (text or u'').strip()

    if text in (u'1', u'true'):
        return True

    if text in (u'0', u'false'):
        return False

    if default is not _non_specified:
        return default

    raise ValueError(u'XSD Boolean valid values are "0", "1", "false", "true"')


class SettlementIdsByStationIdGetter(object):
    """
    Класс получает набор станций при инициализации и
    позволяет получить id города по id станции.
    Связь через Station2Settlement тоже учитывается.
    """

    def __init__(self, stations, default=None):
        self.default = default

        temp_settlement_ids_by_station_id = defaultdict(set)

        station_ids = []

        for station in stations:
            station_ids.append(station.id)

            if station.settlement_id:
                temp_settlement_ids_by_station_id[station.id].add(station.settlement_id)

        # order_by() для того, чтобы сбросить сортировку по-умолчанию и избавится от JOIN
        station2settlement_query = Station2Settlement.objects.filter(station_id__in=station_ids)\
                                                             .order_by()\
                                                             .values_list('station_id', 'settlement_id')

        for station_id, settlement_id in station2settlement_query:
            temp_settlement_ids_by_station_id[station_id].add(settlement_id)

        self.settlement_ids_by_station_id = {
            station_id: list(settlement_ids)
            for station_id, settlement_ids in temp_settlement_ids_by_station_id.iteritems()
        }

    def get(self, station_id):
        return self.settlement_ids_by_station_id.get(station_id, self.default)
