# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from copy import copy
from datetime import datetime, timedelta

import pytz
from django.utils.functional import cached_property
from django.utils.translation import ugettext as _, gettext_noop as N_

from cysix.two_stage_import import CysixStation, xsd_boolean
from travel.rasp.admin.importinfo.models.two_stage_import import TSIThreadSetting
from common.utils.caching import cache_method_result
from travel.rasp.admin.scripts.schedule.utils import RaspImportError
from common.models.geo import Station
from travel.rasp.admin.scripts.schedule.utils.to_python_parsers import get_time
from cysix.two_stage_import.schedule import CysixSchedule
from cysix.two_stage_import.utils import group_description
from common.utils.date import timedelta2minutes


log = logging.getLogger(__name__)


class CysixXmlThread(object):
    group_el = None
    path_key = None

    def __init__(self, thread_el, group_el, factory, group_context, group_settings):
        self.supplier_rtstations = []
        self.factory = factory
        self.thread_el = thread_el

        self.group_settings = group_settings
        self.group_context = group_context

        self.group_el = group_el

        self.is_thread_hidden = False

        self.CysixRouteClass = factory.get_supplier_route_class()

    def __unicode__(self):
        return '<CysixXmlThread: %s>' % self.get_description()

    def get_description(self):
        return 'title="%s" number="%s" sourceline=%s %s' % (
            self.thread_el.get('title', '').strip(),
            self.thread_el.get('number', '').strip(),
            self.thread_el.sourceline,
            group_description(self.group_el)
        )

    @cache_method_result
    def get_settings(self):
        settings = copy(self.group_settings)
        return settings

    @cached_property
    def settings(self):
        return self.get_settings()

    @cache_method_result
    def get_context(self):
        context = copy(self.group_context)

        context.update_from(self.thread_el)

        return context

    @cached_property
    def context(self):
        return self.get_context()

    @classmethod
    def create(cls, thread_el, group_el, factory, group_context, group_settings):
        cysix_xml_thread = cls(thread_el, group_el, factory, group_context, group_settings)

        cysix_xml_thread.supplier_rtstations = [
            CysixRTStation(cysix_xml_thread, stoppoint_el, thread_el, group_el, factory)
            for stoppoint_el in thread_el.findall('./stoppoints/stoppoint')
        ]

        return cysix_xml_thread

    def get_supplier_routes(self):
        if len(self.supplier_rtstations) < 2:
            raise RaspImportError(_('Не хватает станций для маршрута'))

        routes = []

        for cysix_schedule in self.schedule_iter(self.supplier_rtstations):
            routes.append(self.CysixRouteClass(
                self,
                self.supplier_rtstations,
                cysix_schedule,
                self.factory,
            ))

        return routes

    def schedule_iter(self, supplier_rtstations):
        for schedule_el in self.thread_el.findall('./schedules/schedule'):
            if schedule_el.get('canceled', '0').strip() == '1':
                log.info(N_('Пропускаем отмененное расписание.'))

                continue

            schedule_context = copy(self.get_context())
            schedule_context.update_from(schedule_el)

            if schedule_el.get('period_int', '').strip():
                yield CysixSchedule.create_interval_schedule(
                    self, schedule_el, schedule_context, supplier_rtstations
                )

            else:
                for cysix_schedule in CysixSchedule.day_based_schedule_iter(
                    self, schedule_el, schedule_context, supplier_rtstations
                ):
                    yield cysix_schedule

    @cached_property
    def tsi_thread_setting(self):
        package = self.factory.package

        try:
            return TSIThreadSetting.objects.get(package=package, path_key=self.path_key)
        except TSIThreadSetting.DoesNotExist:
            return TSIThreadSetting(package=package, path_key=self.path_key)


class RTSCalculator(object):
    class RTSCalcuatorResult(object):
        def __init__(self, time_zone):
            self.time_zone = time_zone

            self.arrival = None
            self.departure = None

            self.can_add_day = None
            self.can_add_day_to_departure = None
            self.can_add_day_to_arrival = None

        @cached_property
        def pytz(self):
            return pytz.timezone(self.time_zone)

    def __init__(self, schedule_context, start_dt, naive_start_dt, standard_stop_time, thread_time_zone):
        self.schedule_context = schedule_context
        self.start_dt = start_dt
        self.naive_start_dt = naive_start_dt
        self.standard_stop_time = standard_stop_time
        self.thread_time_zone = thread_time_zone

    def calculate_departure_and_arrival(self, binded_supplier_rts):
        result = self.RTSCalcuatorResult(binded_supplier_rts.get_pytz(self.schedule_context).zone)

        result.can_add_day = binded_supplier_rts.calc_initial_can_add_day()
        result.can_add_day_to_departure = binded_supplier_rts.calc_initial_can_add_day_to_departure()
        result.can_add_day_to_arrival = binded_supplier_rts.calc_initial_can_add_day_to_arrival()

        if binded_supplier_rts.is_first:
            result.departure = 0
            return result

        if binded_supplier_rts.has_seconds():
            self.calculate_from_shift(binded_supplier_rts, result)

        elif binded_supplier_rts.has_time():
            self.calculate_from_time(binded_supplier_rts, result)

        else:
            return result

        self.correct_empty_departure_or_arrival(result)

        if binded_supplier_rts.is_nonstop():
            result.departure = result.arrival
            result.can_add_day_to_departure = result.can_add_day_to_arrival

        elif result.arrival is not None and result.departure == result.arrival:
            result.departure = result.arrival + self.standard_stop_time
            result.can_add_day_to_departure = result.can_add_day_to_arrival

        return result

    def _get_shift_from_supplier_shift(self, supplier_shift, pytz):
        if supplier_shift is None:
            return None

        naive_dt = self.naive_start_dt + timedelta(minutes=supplier_shift)
        dt = pytz.localize(naive_dt)
        return timedelta2minutes(dt - self.start_dt)

    def calculate_from_shift(self, binded_supplier_rts, result):
        """
        Вычисление параметров arrival и departure, исходя из минутных сдвигов, заданных в исходном файле
        """
        result.arrival = self._get_shift_from_supplier_shift(binded_supplier_rts.arrival_minutes_shift, result.pytz)
        result.departure = self._get_shift_from_supplier_shift(binded_supplier_rts.departure_minutes_shift, result.pytz)

    def _get_shift_from_supplier_time(self, supplier_time, supplier_day_shift, pytz):
        if supplier_time is None:
            return None

        day_shift = supplier_day_shift or 0
        naive_date = self.naive_start_dt.date() + timedelta(days=day_shift)
        naive_dt = datetime.combine(naive_date, supplier_time)
        dt = pytz.localize(naive_dt)
        return int(timedelta2minutes(dt - self.start_dt))

    def calculate_from_time(self, binded_supplier_rts, result):
        """
        Вычисление параметров arrival и departure, исходя из времен, заданных в исходном файле
        """
        result.arrival = self._get_shift_from_supplier_time(
            binded_supplier_rts.arrival_time, binded_supplier_rts.arrival_day_shift , result.pytz
        )
        result.departure = self._get_shift_from_supplier_time(
            binded_supplier_rts.departure_time, binded_supplier_rts.departure_day_shift, result.pytz
        )

    def correct_empty_departure_or_arrival(self, result):
        if result.arrival is None and result.departure is None:
            return

        if result.arrival is None:
            result.arrival = result.departure - self.standard_stop_time
            result.can_add_day_to_arrival = result.can_add_day_to_departure

        elif result.departure is None:
            result.departure = result.arrival + self.standard_stop_time
            result.can_add_day_to_departure = result.can_add_day_to_arrival


class CysixRTStation(object):
    """
    Получается, что класс используется в двух ситуациях:

    1. При импорте "в промежуточную базу", когда нужна информация
    о последовательности станций и расстоянии поставщика (для сортировки)

    2. При импорте "в основную базу", когда нужна еще информация о временах

    Поэтому в __init__ находим:
    станцию поставщика (self.supplier_station)
    станцию (self.station) - может быть None - например, при первом импорте в промежуточную базу
    расстояние

    Атрибуты времени, парсим при необходимости (при импорте в основную базу)
    """

    _is_times_calculated = False

    def __init__(self, xml_thread, stoppoint_el, thread_el, group_el, factory):
        self.xml_thread = xml_thread
        self.factory = factory

        self.stoppoint_el = stoppoint_el
        self.thread_el = thread_el
        self.group_el = group_el
        self.thread_settings = xml_thread.settings
        self.context = self._get_context(xml_thread.context)

        self._distance = self.parse_distance(stoppoint_el)

    @cached_property
    def supplier_station(self):
        return CysixStation.create_station(
            self.stoppoint_el, self.thread_el, self.group_el,
            self.factory, self.context, self.thread_settings
        )

    @cached_property
    def finder(self):
        return self.factory.get_station_finder()

    @cached_property
    def station(self):
        supplier_station = self.supplier_station
        try:
            return self.finder.find_by_supplier_station(supplier_station)
        except Station.MultipleObjectsReturned:
            return self.finder.find_first_by_supplier_station(supplier_station)

        except Station.DoesNotExist:
            log.error(N_("Не нашли привязку для '{}'").format(supplier_station))

    @cached_property
    def stoppoint_timezone(self):
        return self.stoppoint_el.get('timezone', u'').strip()

    def _calc_times(self):
        if self._is_times_calculated:
            return

        self._arrival_time = self.parse_arrival_time(self.stoppoint_el)
        self._departure_time = self.parse_departure_time(self.stoppoint_el)

        self._arrival_minutes_shift = self.parse_arrival_minutes_shift(self.stoppoint_el)
        self._departure_minutes_shift = self.parse_departure_minutes_shift(self.stoppoint_el)

        self._arrival_day_shift = self.parse_arrival_day_shift(self.stoppoint_el)
        self._departure_day_shift = self.parse_departure_day_shift(self.stoppoint_el)

        self._is_times_calculated = True

        self._log_strange_case()

    def _get_context(self, thread_context):
        context = copy(thread_context)
        context.update_from(self.stoppoint_el)

        return context

    def has_seconds(self):
        return self.arrival_minutes_shift is not None or self.departure_minutes_shift is not None

    def has_time(self):
        return self.arrival_time is not None or self.departure_time is not None

    def calc_initial_can_add_day(self):
        if not self.has_seconds():
            if self.arrival_time is not None and self.arrival_day_shift is None:
                return True

            if self.departure_time is not None and self.departure_day_shift is None:
                return True

        return False

    def calc_initial_can_add_day_to_departure(self):
        if self.departure_minutes_shift is not None:
            return False

        return self.departure_time is not None and self.departure_day_shift is None

    def calc_initial_can_add_day_to_arrival(self):
        if self.arrival_minutes_shift is not None:
            return False

        return self.arrival_time is not None and self.arrival_day_shift is None

    @cache_method_result
    def is_nonstop(self):
        return self.stoppoint_el.get('is_nonstop', '').strip() == '1'

    @cache_method_result
    def is_technical(self):
        return self.stoppoint_el.get('is_technical', '').strip() == '1'

    @cache_method_result
    def is_fuzzy(self):
        return xsd_boolean(self.stoppoint_el.get('is_fuzzy'), False)

    @cache_method_result
    def is_combined(self):
        return xsd_boolean(self.stoppoint_el.get('is_combined'), False)

    @cache_method_result
    def in_station_schedule(self):
        return xsd_boolean(self.stoppoint_el.get('_in_station_schedule'), None)

    @cache_method_result
    def is_searchable_to(self):
        return xsd_boolean(self.stoppoint_el.get('_is_searchable_to'), None)

    @cache_method_result
    def is_searchable_from(self):
        return xsd_boolean(self.stoppoint_el.get('_is_searchable_from'), None)

    @cache_method_result
    def in_thread(self):
        return xsd_boolean(self.stoppoint_el.get('_in_thread'), None)

    @property
    def arrival_time(self):
        if not self._is_times_calculated:
            self._calc_times()

        return self._arrival_time

    @property
    def departure_time(self):
        if not self._is_times_calculated:
            self._calc_times()

        return self._departure_time

    @property
    def arrival_minutes_shift(self):
        if not self._is_times_calculated:
            self._calc_times()

        return self._arrival_minutes_shift

    @property
    def departure_minutes_shift(self):
        if not self._is_times_calculated:
            self._calc_times()

        return self._departure_minutes_shift

    @property
    def arrival_day_shift(self):
        if not self._is_times_calculated:
            self._calc_times()

        return self._arrival_day_shift

    @property
    def departure_day_shift(self):
        if not self._is_times_calculated:
            self._calc_times()

        return self._departure_day_shift

    @classmethod
    def parse_distance(cls, stoppoint_el):
        try:
            return float(stoppoint_el.get('distance', '').strip())

        except (TypeError, ValueError):
            return None

    @classmethod
    def parse_departure_time(cls, stoppoint_el):
        raw_departure_time = stoppoint_el.get('departure_time', '').strip()

        if raw_departure_time:
            try:
                return get_time(raw_departure_time)

            except (ValueError, TypeError):
                log.warning(N_("Ошибка разбора времени отправления (departure_time) '%s'"), raw_departure_time)

    @classmethod
    def parse_arrival_time(cls, stoppoint_el):
        raw_arrival_time = stoppoint_el.get('arrival_time', '').strip()

        if raw_arrival_time:
            try:
                return get_time(raw_arrival_time)

            except (ValueError, TypeError):
                log.warning(N_("Ошибка разбора времени прибытия (arrival_time) '%s'"), raw_arrival_time)

    @classmethod
    def parse_arrival_day_shift(cls, stoppoint_el):
        raw_arrival_day_shift = stoppoint_el.get('arrival_day_shift', '').strip()

        if raw_arrival_day_shift:
            try:
                return int(raw_arrival_day_shift)

            except (ValueError, TypeError):
                log.warning(N_("Ошибка разбора смещения в днях прибытия '%s'"), raw_arrival_day_shift)

    @classmethod
    def parse_departure_day_shift(cls, stoppoint_el):
        raw_departure_day_shift = stoppoint_el.get('departure_day_shift', '').strip()

        if raw_departure_day_shift:
            try:
                return int(raw_departure_day_shift)

            except (ValueError, TypeError):
                log.warning(N_("Ошибка разбора смещения в днях отправления '%s'"), raw_departure_day_shift)

    @classmethod
    def parse_arrival_minutes_shift(cls, stoppoint_el):
        raw_arrival_shift = stoppoint_el.get('arrival_shift', "").strip()

        if raw_arrival_shift:
            try:
                return int(float(raw_arrival_shift) / 60)

            except ValueError:
                log.warning(N_("Не смогли разобрать время прибытия (arrival_shift) '%s'"), raw_arrival_shift)

    @classmethod
    def parse_departure_minutes_shift(cls, stoppoint_el):
        raw_departure_shift = stoppoint_el.get('departure_shift', '').strip()

        if raw_departure_shift:
            try:
                return int(float(raw_departure_shift) / 60)

            except ValueError:
                log.warning(N_("Не смогли разобрать время отправления (departure_shift) '%s'"), raw_departure_shift)

    def _log_strange_case(self):
        if self.arrival_time is not None and self.departure_time is not None:

            if self.departure_day_shift is None and self.arrival_day_shift is not None:
                log.warning(
                    N_(
                        "Для станции '%s' заданы: "
                        "абсолятное время прибытия, смещение суток для прибытия "
                        "и абсолютное время отправления, "
                        "но не задано смещение суток для отправления."
                    ),
                    self.station.title
                )

            if self.arrival_day_shift is None and self.departure_day_shift is not None:
                # может быть случай когда departure_day_shift=1, а arrival_day_shift 0 и потому не указан.
                if self.departure_day_shift not in [0, 1]:
                    log.warning(
                        N_(
                            "Для станции '%s' заданы: "
                            "абсолятное время отправления, смещение суток для отправления "
                            "и абсолютное время прибытия, "
                            "но не задано смещение суток для прибытия."
                        ),
                        self.station.title
                    )

            # нужно проверить, что departure_day_shift >= arrival_day_shift и разница не больше 1.
            arrival_day_shift = self.arrival_day_shift or 0
            departure_day_shift = self.departure_day_shift or 0

            if (departure_day_shift - arrival_day_shift) > 1:
                log.warning(
                    N_(
                        "Для станции '%s': "
                        "смещение суток для отправления значительно превышает "
                        "смещение суток для прибытия."
                    ),
                    self.station.title
                )

    @property
    def is_first(self):
        return self is self.xml_thread.supplier_rtstations[0]

    @property
    def is_last(self):
        return self is self.xml_thread.supplier_rtstations[-1]

    @property
    def start_station(self):
        return self.xml_thread.supplier_rtstations[0].station

    @property
    def end_station(self):
        return self.xml_thread.supplier_rtstations[-1].station

    def get_pytz(self, schedule_context):
        """
        Правильно всегда передавать контекст, который меняет вывод метода неизменяющегося объекта.
        """

        tsi_thread_setting = self.xml_thread.tsi_thread_setting

        if tsi_thread_setting.timezone_override and tsi_thread_setting.timezone_override != 'none':
            tz = self.get_timezone(tsi_thread_setting.timezone_override)
            if tz is not None:
                return tz

        filter_ = self.factory.get_filter('timezone_override')
        tz = filter_.apply(self)
        if tz is not None:
            return tz

        tz = self.get_timezone(self.stoppoint_timezone)
        if tz is not None:
            return tz

        tz = self.get_timezone(schedule_context.timezone)
        if tz is not None:
            return tz

        raise RaspImportError(_('Не известное значение временной зоны "%s"'), schedule_context.timezone)

    def get_timezone(self, timezone):
        if timezone == 'start_station':
            return self.start_station.pytz

        elif timezone == 'end_station':
            return self.end_station.pytz

        elif timezone == 'local':
            return self.station.pytz

        elif timezone in pytz.all_timezones_set:
            return pytz.timezone(timezone)

    @property
    def distance(self):
        if self._distance:
            return self._distance

        if self.is_first:
            return 0.0

        return None
