# -*- coding: utf-8 -*-

import logging
import urllib2

from contextlib import contextmanager

from django.core.files.uploadhandler import TemporaryFileUploadHandler
from django.utils.html import escape

from cysix.filters.cysix_xml_validation import validate_cysix_xml, CysixCheckError
from django.views.decorators.csrf import csrf_exempt, csrf_protect
from travel.rasp.admin.lib.logs import get_collector, remove_collector
from travel.rasp.admin.lib.tmpfiles import get_tmp_filepath, clean_temp
from templates.bemhtml.validate import Template, TemplateResult


USER_CYSIX_FILE_DOWNLOAD_TIMEOUT = 120  # 2 минуты

log = logging.getLogger(__name__)


def index(request):
    context = {}

    return Template.render(request, context)


@csrf_exempt
@clean_temp
def validate(request):
    # Всегда временно сохраняем файл, даже маленький
    request.upload_handlers = [TemporaryFileUploadHandler()]

    with get_log_collector() as log_collector:
        return _validate(request, log_collector)


@csrf_protect
def _validate(request, log_collector):
    context = {}

    if request.method == 'POST':
        current_tab = request.POST.get('current-tab', None)

        context = {}

        if current_tab == 'tab-file':
            if not request.FILES:
                context['error_code'] = 'NO_FILE'

            else:
                file_ = request.FILES['file']

                filename = file_.temporary_file_path()

                _validate_cysix_xml(filename, log_collector, context)

        elif current_tab == 'tab-url':
            url = request.POST.get('url', None)

            if not url:
                context['error_code'] = 'NO_URL'

            else:
                try:
                    data = urllib2.urlopen(url, timeout=USER_CYSIX_FILE_DOWNLOAD_TIMEOUT).read()

                    tmp_file_name = get_tmp_filepath('for_cysix_validation.xml')

                    with open(tmp_file_name, 'w') as f:
                        f.write(data)
                        f.flush()

                        _validate_cysix_xml(tmp_file_name, log_collector, context)

                except (urllib2.URLError, ValueError):
                    context['error_code'] = 'CANT_OPEN_URL'

        elif current_tab == 'tab-text':
            text = request.POST.get('text', u'')

            text = text.encode('utf-8')

            tmp_file_name = get_tmp_filepath('for_cysix_validation.xml')

            with open(tmp_file_name, 'w') as f:
                f.write(text)
                f.flush()

                _validate_cysix_xml(tmp_file_name, log_collector, context)

    context['log_text'] = escape(log_collector.get_collected(clean=False))

    return TemplateResult.render(request, context)


def _validate_cysix_xml(filename, log_collector, context):
    try:
        validate_cysix_xml(filename)

        if log_collector.get_collected(clean=False):
            context['state'] = 'mediocre'

        else:
            context['state'] = 'good'

    except CysixCheckError, e:
        context['state'] = 'bad'

        context['cysix_check_error'] = unicode(e)


@contextmanager
def get_log_collector():
    log_collector = get_collector('', format=u'%(message)s')
    log_collector.setLevel(logging.INFO)
    try:
        yield log_collector

    finally:
        remove_collector('', log_collector)
