# -*- coding: utf-8 -*-

import os
import subprocess
import sys
import time
import traceback

from django import forms
from django.conf import settings
from django.contrib import admin, messages
from django.core.exceptions import PermissionDenied
from django.http import HttpResponseRedirect
from django.shortcuts import redirect
from django.utils.encoding import force_text
from django.utils.translation import ugettext_lazy as _

from travel.rasp.admin.admin.admin_tasks import TaskLog, get_task_log_blocks
from travel.rasp.admin.importinfo.admin.trusted_station import TrustedStationAdmin  # noqa
from travel.rasp.admin.importinfo.afdbchanges.actions import store
from common.importinfo.models import Express2Country
from common.importinfo.models import YaCalendarXml
from common.models.schedule import ScheduleExclusion
from common.models.transport import TransportModel
from travel.rasp.library.python.common23.date import environment

from travel.rasp.admin.importinfo.admin.e_traffic import EtrafficMainBusStationInGroupAdmin
from travel.rasp.admin.importinfo.admin.ipektur import IpekturStopsOrderAdmin
from travel.rasp.admin.importinfo.admin.slices import PackageDirectionsSliceAdmin, DirectionSliceAdmin
from travel.rasp.admin.importinfo.afdbchanges.utils import run_apply_af_db_changes_as_task
from travel.rasp.admin.importinfo.models import (
    Route2Terminal, TariffFile, ImportReport, CompanySynonym, TabloManualStatus, BlackList,
    TabloImportSynonym, BlackListThreadStation, IataCorrection, OriginalThreadData, CodeshareNumber)
from travel.rasp.admin.importinfo.models.af import (
    AFScheduleFile, AFUpdateFile, AFSubwayFile, AFMaskText, AFDBChangesFile, AFTrainTurnoverFile,
    AFTrainTurnoverUpdateFile
)
from travel.rasp.admin.importinfo.models.bus import MTASettlementMapping, MTADistrictMapping, MTAInternalNumberBlackList
from travel.rasp.admin.importinfo.models.content_manager import ContentManager
from travel.rasp.admin.importinfo.models.e_traffic import EtrafficMainBusStationInGroup
from travel.rasp.admin.importinfo.models.ipektur import IpekturStopsOrder
from travel.rasp.admin.importinfo.models.mappings import StationMapping, MappingContext, CompanyMapping, TransportModelMapping
from travel.rasp.admin.importinfo.models.two_stage_import import (
    TwoStageImportStation, TwoStageImportThread, TSIThreadStationFlag, TwoStageImportThreadStation,
    TwoStageImportPackage, PackageDirectionsSlice, DirectionSlice, CysixGroupFilter)
from travel.rasp.admin.importinfo.two_stage_import.admin import TwoStageImportPackageAdmin, CysixGroupFilterAdmin
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin, OrderingOnlyOnSmallQuerysetModelAdminMixin
from travel.rasp.admin.lib.adminutils import get_rasp_admin_url


class Route2TerminalAdmin(RaspExportModelAdmin):
    raw_id_fields = ('terminal',)
    list_display = ('number', 'terminal')


class TariffFileAdminForm(forms.ModelForm):
    def clean(self):
        if not (self.cleaned_data.get('tariff_file') or self.cleaned_data.get('tariff_rich_file')):
            raise forms.ValidationError(_(u'Нужно приложить файл с тарифами'))

        return super(TariffFileAdminForm, self).clean()


class TariffFileAdmin(RaspExportModelAdmin):
    form = TariffFileAdminForm

    def _reimport_tariff_files(self, tarifffiles):
        flock_command = [
            'flock', '--close', '--nonblock',
            '/tmp/rasp-{}-import_suburban.lock'.format(settings.INSTANCE_ROLE.code)
        ]

        process_env = os.environ.copy()
        process_env.update({
            'Y_PYTHON_ENTRY_POINT': 'travel.rasp.admin.scripts.tariffs.import_suburban'
        })

        process = subprocess.Popen(
            flock_command + [
                sys.executable, '--tarifffile-ids'
            ] + [
                str(tarifffile.id) for tarifffile in tarifffiles
            ],
            cwd=settings.PROJECT_PATH,
            env=process_env
        )

        time.sleep(0.1)  # ждем код возврата flock
        returncode = process.poll()

        return returncode in (None, 0)

    def _reimport_tariff_file(self, tarifffile):
        return self._reimport_tariff_files([tarifffile])

    def reimport_tariff_files(self, request, queryset):
        if self._reimport_tariff_files(queryset):
            messages.success(request, _(u"Запущен импорт файлов с тарифами"))
        else:
            messages.error(request, _(u"Не удалось запустить импорт файлов с тарифами"))

    reimport_tariff_files.short_description = _(u"Импортировать тарифы")

    actions = ['reimport_tariff_files']

    def change_view(self, request, object_id, **kwargs):
        if request.method == 'POST' and 'reimport' in request.POST:
            tariffile = TariffFile.objects.get(id=object_id)

            if self._reimport_tariff_file(tariffile):
                messages.success(request, _(u"Запущен импорт файла с тарифами"))
            else:
                messages.error(request, _(u"Не удалось запустить импорт файла с тарифами"))

            return HttpResponseRedirect(request.path)

        return super(TariffFileAdmin, self).change_view(request, object_id, **kwargs)

    def save_model(self, request, obj, form, change):
        obj.load_log = u"Импорт файла с тарифами запущен {}...".format(environment.now())
        obj.save()  # сохраняем поле перед импортом, чтобы избежать конфликта

        if not self._reimport_tariff_file(obj):
            obj.load_log = u"Не удалось запустить импорт файла с тарифами"
            obj.save()

    list_display = ('__unicode__', 'added', 'imported')
    fieldsets = (
        ('', {
            'fields': ('tariff_rich_file', 'imported')
        }),
        (u'Старое поле файла', {
            'fields': ('tariff_file_name', 'tariff_file'),
            'classes': ('collapse',)
        }),
    )


class ImportReportAdmin(RaspExportModelAdmin):
    list_display = ('__unicode__', 'route_count', 'imported')
    list_filter = ('supplier',)


class AFUpdateFileAdminForm(forms.ModelForm):
    class Meta:
        model = AFUpdateFile
        exclude = []

    def clean(self):
        from travel.rasp.admin.lib.maintenance.flags import flags
        # Идет работа с базой, не загружать обновления
        if flags['maintenance']:
            raise forms.ValidationError(u"Идет работа с базой, нельзя загружать обновления")

        return super(AFUpdateFileAdminForm, self).clean()


class AFUpdateFileAdmin(RaspExportModelAdmin):
    form = AFUpdateFileAdminForm
    list_display = ('update_file_name', 'added', 'loaded')
    fieldsets = (
        ('Profile field', {
            'fields': ('update_file_name', 'update_file', 'loaded', 'recount_schedule_on_the_fly')
        }),
    )


class AFScheduleFileAdmin(RaspExportModelAdmin):
    search_fields = ('region__title', 'schedule_file_name')
    raw_id_fields = ('region',)
    list_display = ('schedule_file_name', 'region', 'added', 'imported')
    fieldsets = (
        ('Profile field', {
            'fields': ('schedule_file_name', 'schedule_file', 'region', 'imported')
        }),
    )


class AFSubwayFileAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement',)
    search_fields = ('settlement__title', 'subway_file_name')
    list_display = ('subway_file_name', 'settlement', 'imported')
    exclude = ('load_log',)


@admin.register(AFTrainTurnoverFile)
class AFTrainTurnoverFileAdmin(RaspExportModelAdmin):
    form = AFUpdateFileAdminForm
    list_display = ('schedule_file_name', 'created_at', 'loaded')
    readonly_fields = ('loaded',)
    fieldsets = (
        ('', {
            'fields': ('schedule_file_name', 'schedule_file', 'loaded')
        }),
    )


@admin.register(AFTrainTurnoverUpdateFile)
class AFTrainTurnoverUpdateFileAdmin(RaspExportModelAdmin):
    form = AFUpdateFileAdminForm
    list_display = ('update_file_name', 'created_at', 'loaded')
    readonly_fields = ('loaded',)
    fieldsets = (
        ('', {
            'fields': ('update_file_name', 'update_file', 'loaded')
        }),
    )

    def save_model(self, request, obj, form, change):
        from travel.rasp.admin.importinfo.turnovers import load_turnovers_from_file
        from travel.rasp.admin.lib.maintenance.flags import flags
        from travel.rasp.admin.lib.maintenance.scripts import job
        if flags['maintenance']:
            raise Exception(u'Идет работа с базой, нельзя обновления')
        try:
            flags['maintenance'] = job.AF_TRAIN_TURNOVER.flag_value
            obj.load_log = u''
            last_log = load_turnovers_from_file(obj.get_encoded_data(), obj.update_file_name)
            obj.load_log = last_log
            obj.loaded = True
        except Exception:
            exc_info = traceback.format_exc()
            obj.load_log += force_text(exc_info)
        finally:
            flags['maintenance'] = 0
        super(AFTrainTurnoverUpdateFileAdmin, self).save_model(request, obj, form, change)


class TitleMappingForm(forms.ModelForm):
    def clean(self):
        super(TitleMappingForm, self).clean()
        title = self.cleaned_data['title'] or u""
        code = self.cleaned_data['code'] or u""
        supplier = self.cleaned_data['supplier']
        context = self.cleaned_data['context']

        model = self._meta.model

        similar_mappins = model.objects.exclude(pk=self.instance.id).filter(supplier=supplier,
                                                                            title=title, code=code,
                                                                            context=context)

        if similar_mappins:
            raise forms.ValidationError(u"Уже есть соответствие с таким названием, кодом и контекстом")

        return self.cleaned_data


class StationMappingAdmin(RaspExportModelAdmin):
    form = TitleMappingForm

    def get_station_title(self, obj):
        from travel.rasp.admin.www.admin import StationAdmin
        return StationAdmin.station_full_path(obj.station, include_station=True)

    get_station_title.allow_tags = True
    get_station_title.short_description = _(u"АвтоСтанция")

    fieldsets = (
        ('Profile field', {
            'fields': ('supplier', 'title', 'code', 'station', 'context')
        }),
    )

    raw_id_fields = ('station', 'region', 'context')
    search_fields = ('title', 'station__title', 'code')
    list_display = ('title', 'code', 'supplier', 'context', 'get_station_title')
    list_filter = ('supplier',)
    ordering = ('-context__priority',)

    def save_model(self, request, obj, form, change):
        obj.save()

        obj.update_tsi_stations()


class CompanyMappingAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('supplier', 'title', 'code', 'company')
        }),
    )
    raw_id_fields = ('company', 'supplier')
    search_fields = ('title', 'code', 'company__title', 'company__title_en', 'company__title_ru')


class TransportModelInline(admin.TabularInline):
    model = TransportModel


class TransportModelMappingAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('supplier', 'title', 'code', 't_model', 'get_transportmodel_url')
        }),
    )

    readonly_fields = ('get_transportmodel_url',)
    raw_id_fields = ('t_model', 'supplier')
    search_fields = ('title', 'supplier__title', 'supplier__code', 't_model__title')
    list_filter = ('supplier',)
    list_display = ('supplier', 'title', 'code', 't_model')

    def get_transportmodel_url(self, tmm):
        t_model = TransportModel.objects.get(pk=tmm.t_model.pk)

        url = get_rasp_admin_url(t_model, relative_only=True)

        return u'<a href="{}">{}</a>'.format(url, t_model.title)

    get_transportmodel_url.allow_tags = True
    get_transportmodel_url.short_description = _(u'Ссылка на модель транспорта')


class MappingContextAdmin(RaspExportModelAdmin):
    list_filter = ('supplier',)
    search_fields = ('route_number', 'route_title', 'first_station_code', 'first_station_title',
                     'last_station_code', 'last_station_title')
    list_display = ('priority', 'route_number', 'route_title', 'first_station_code',
                    'first_station_title', 'last_station_code', 'last_station_title')


class CompanySynonymAdmin(RaspExportModelAdmin):
    raw_id_fields = ('company',)
    search_fields = ('title', 'company__title')
    list_display = ('__unicode__', 'title', 'company', 't_type')
    list_filter = ('t_type',)


class TabloManualStatusAdmin(RaspExportModelAdmin):
    actions = ['delete_selected']
    raw_id_fields = ('station',)
    search_fields = ('station__title',)
    list_display = ('station', 'event', 'number', 'planned', 'status', 'real')
    list_filter = ('event', 'status')


class BlackListThreadStationInline(admin.StackedInline):
    model = BlackListThreadStation
    raw_id_fields = ('station',)
    extra = 1


class BlackListAdmin(RaspExportModelAdmin):
    inlines = [BlackListThreadStationInline]
    list_display = ('comment', 'supplier', 't_type', 'number')
    search_fields = ('number', 'comment')
    list_filter = ('t_type', 'supplier')
    raw_id_fields = ('supplier', 'start_station', 'finish_station')
    fieldsets = (
        (None, {
            'fields': ('comment', 'supplier', 't_type', 'number', 'thread_type')
        }),
        ('Special', {
            'fields': ('supplier_number', 'supplier_title', 'start_time_start', 'start_time_end', 'finish_time_start',
                       'finish_time_end', 'is_moscow_time', 'start_station', 'finish_station')
        })
    )


class BusStationImportFileAdmin(RaspExportModelAdmin):
    list_display = ('station_file_name', 'added', 'supplier')
    list_filter = ('supplier',)
    fieldsets = (
        ('Profile field', {
            'fields': ('station_file_name', 'station_file', 'supplier', 'test_run')
        }),
    )


class Express2CountryAdmin(RaspExportModelAdmin):
    list_display = ('code_re', 'country', 'time_zone', 'priority', 'comment')


class YaCalendarXmlAdmin(RaspExportModelAdmin):
    raw_id_fields = ('country',)
    list_display = ('country', 'get_year')


class MTASettlementMappingAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement', 'region')
    list_display = ('stop_city', 'settlement', 'region')


class MTADistrictMappingAdmin(RaspExportModelAdmin):
    raw_id_fields = ('region', 'district')
    list_display = ('area_name', 'area_code', 'district', 'region')


class TwoStageImportThreadAdmin(OrderingOnlyOnSmallQuerysetModelAdminMixin, RaspExportModelAdmin):
    search_fields = ('title',)
    list_display = ('title', )
    list_filter = ('package',)


class TwoStageImportThreadStationAdmin(RaspExportModelAdmin):
    list_display = ('get_order', 'get_station_title_link', 'get_station_code')
    list_filter = ('thread__package',)
    raw_id_fields = ('thread', 'station',)
    search_fields = ('thread__title', 'station__title',)
    ordering = ('station__code',)

    def get_station_title_link(self, thread_station):
        return u"<a href=\"%s\">%s</a>" % (get_rasp_admin_url(thread_station.station), thread_station.station.title)

    get_station_title_link.short_description = _(u"Название станции")
    get_station_title_link.allow_tags = True

    def get_station_code(self, thread_station):
        return thread_station.station.code

    get_station_code.short_description = _(u"Код станции")

    def get_order(self, thread_station):
        zero_based_order_number = list(thread_station.thread.threadstations.all()).index(thread_station)
        return zero_based_order_number + 1

    get_order.short_description = _(u"Порядковый номер")


class ScheduleExclusionAdmin(RaspExportModelAdmin):
    raw_id_fields = ('start_station', 'end_station', 'exclude_station')


class TwoStageImportStationAdmin(RaspExportModelAdmin):
    search_fields = ('title', 'code')
    list_display = ('title', 'code')
    list_filter = ('package',)
    readonly_fields = ('code', 'title', 'real_title', 'additional_info', 'package',
                       'station_mapping', 'latitude', 'longitude')

    def save_form(self, request, form, change):
        raise PermissionDenied


class KVSBusStationAdmin(RaspExportModelAdmin):
    search_fields = ('title', 'code')
    list_display = ('title', 'code', 'blue_middle_available', 'blue_import_available', 'has_buy')


class MTAInternalNumberBlackListAdmin(RaspExportModelAdmin):
    list_display = ('number', 'comment')


class TSIThreadStationFlagAdmin(RaspExportModelAdmin):
    list_display = ('station_key', 'package',
                    'is_fuzzy', 'is_searchable_from', 'is_searchable_to',
                    'in_station_schedule', 'in_thread')
    readonly_fields = ('package', 'path_key', 'station_key',)


class IataCorrectionAdmin(RaspExportModelAdmin):
    raw_id_fields = ('company',)


class OriginalThreadDataAdmin(RaspExportModelAdmin):
    readonly_fields = ('thread', 'thread_uid', 'cysix', 'raw')


admin.site.register(TariffFile, TariffFileAdmin)
admin.site.register(AFUpdateFile, AFUpdateFileAdmin)
admin.site.register(AFScheduleFile, AFScheduleFileAdmin)
admin.site.register(AFSubwayFile, AFSubwayFileAdmin)

admin.site.register(StationMapping, StationMappingAdmin)
admin.site.register(CompanyMapping, CompanyMappingAdmin)
admin.site.register(TransportModelMapping, TransportModelMappingAdmin)
admin.site.register(MappingContext, MappingContextAdmin)

admin.site.register(CompanySynonym, CompanySynonymAdmin)
admin.site.register(BlackList, BlackListAdmin)
admin.site.register(TabloManualStatus, TabloManualStatusAdmin)
admin.site.register(Route2Terminal, Route2TerminalAdmin)

admin.site.register(ImportReport, ImportReportAdmin)

admin.site.register(MTASettlementMapping, MTASettlementMappingAdmin)
admin.site.register(MTADistrictMapping, MTADistrictMappingAdmin)
admin.site.register(MTAInternalNumberBlackList, MTAInternalNumberBlackListAdmin)

admin.site.register(YaCalendarXml, YaCalendarXmlAdmin)

admin.site.register(TwoStageImportStation, TwoStageImportStationAdmin)
admin.site.register(TwoStageImportThread, TwoStageImportThreadAdmin)
admin.site.register(TwoStageImportThreadStation, TwoStageImportThreadStationAdmin)
admin.site.register(TwoStageImportPackage, TwoStageImportPackageAdmin)
admin.site.register(CysixGroupFilter, CysixGroupFilterAdmin)

admin.site.register(Express2Country, Express2CountryAdmin)
admin.site.register(ScheduleExclusion, ScheduleExclusionAdmin)

admin.site.register(TSIThreadStationFlag, TSIThreadStationFlagAdmin)

admin.site.register(IataCorrection, IataCorrectionAdmin)

admin.site.register(OriginalThreadData, OriginalThreadDataAdmin)


class AFMaskTextAdmin(RaspExportModelAdmin):
    list_display = ('code', 'short_text_ru', 'long_text_ru', 'prev_code', 'next_code', 'description')

admin.site.register(AFMaskText, AFMaskTextAdmin)


class TabloImportSynonymForm(forms.ModelForm):
    def clean(self):
        super(TabloImportSynonymForm, self).clean()
        station = self.cleaned_data['station']
        settlement = self.cleaned_data['settlement']

        if not (station or settlement):
            raise forms.ValidationError(u'Нужно указать станцию или город')

        return self.cleaned_data


class TabloImportSynonymAdmin(RaspExportModelAdmin):
    form = TabloImportSynonymForm
    raw_id_fields = ('station', 'settlement', 'supplier')

admin.site.register(TabloImportSynonym, TabloImportSynonymAdmin)


class ContentManagerAdmin(RaspExportModelAdmin):
    list_display = ('id', 'name', 'email')

admin.site.register(ContentManager, ContentManagerAdmin)

admin.site.register(IpekturStopsOrder, IpekturStopsOrderAdmin)


class CodeshareNumberAdmin(RaspExportModelAdmin):
    list_display = ('number_re', 'comment', 'exclude_from_tablo_import')


admin.site.register(CodeshareNumber, CodeshareNumberAdmin)

admin.site.register(DirectionSlice, DirectionSliceAdmin)
admin.site.register(PackageDirectionsSlice, PackageDirectionsSliceAdmin)
admin.site.register(EtrafficMainBusStationInGroup, EtrafficMainBusStationInGroupAdmin)


@admin.register(AFDBChangesFile)
class AFDBChangesFileAdmin(RaspExportModelAdmin):
    list_display = ('__unicode__', 'imported_at')
    MAX_LOGS_TO_SHOW = 5

    def save_model(self, request, obj, form, change):
        super(AFDBChangesFileAdmin, self).save_model(request, obj, form, change)

        if change:
            return

        run_apply_af_db_changes_as_task(obj)

        self.message_user(request, u'Запустили импорт изменений объектов')

    def get_urls(self):
        from django.conf.urls import url

        urlpatterns = super(RaspExportModelAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(.+)/reimport/$',
                self.admin_site.admin_view(self.reimport_view),
                name='%s_%s_reimport' % info),
        ] + urlpatterns

        return urlpatterns

    def reimport_view(self, request, db_change_id):
        afdbchangesfile = self.model.objects.get(pk=db_change_id)

        run_apply_af_db_changes_as_task(afdbchangesfile)

        self.message_user(request, u'Запустили импорт изменений объектов')

        return redirect('admin:importinfo_afdbchangesfile_change', db_change_id)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        afdbchangesfile = self.model.objects.get(pk=object_id)
        extra_context = extra_context or {}

        task_name = TaskLog.get_object_task_name(afdbchangesfile, 'import')
        task_log_blocks = get_task_log_blocks(task_name)

        extra_context['task_log_blocks'] = task_log_blocks
        extra_context['documentation'] = [
            {
                'model': ma['model']._meta.verbose_name,
                'type': ma['action_type'],
                'help_text': ma['help_text']
            }
            for ma in store.model_actions.values()
        ]

        return self.changeform_view(request, object_id, form_url, extra_context)
