# -*- coding: utf-8 -*-

import json

from django import forms
from django.utils.translation import ugettext_lazy as _

from travel.rasp.admin.importinfo.models.ipektur import IpekturStopsOrder
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin


class IpekturStopsOrderAdminForm(forms.ModelForm):
    class Meta:
        model = IpekturStopsOrder
        exclude = []

    def clean_original_order(self):
        original_order = self.cleaned_data['original_order'].strip()

        try:
            json.loads(original_order)

            return original_order

        except ValueError:
            # возможно передали как в данных от ipektur
            from travel.rasp.admin.scripts.schedule.bus.ipektur import IpekturStops

            titles = IpekturStops(original_order).get_canonical_titles()

            if len(titles) > 1:
                return json.dumps(titles, encoding='utf-8', ensure_ascii=False)

            # Тогда названия станций идут через перевод строки
            titles = self.get_titles_separated_by_new_line(original_order)

            if len(titles) > 1:
                return json.dumps([title.title for title in titles],
                                  encoding='utf-8', ensure_ascii=False)

            raise forms.ValidationError(_(u'Не получается разобрать порядок станций.'))

    def clean_right_order(self):
        right_order = self.cleaned_data['right_order'].strip()

        if not right_order:
            return ''

        try:
            json.loads(right_order)

            return right_order

        except ValueError:
            titles = self.get_titles_separated_by_new_line(right_order)

            if len(titles) > 1:
                return json.dumps([title.title for title in titles],
                                  encoding='utf-8', ensure_ascii=False)

            raise forms.ValidationError(_(u'Не получается разобрать порядок станций.'))

    def clean_use(self):
        if not self.cleaned_data['right_order'].strip():
            return False

        return self.cleaned_data['use']

    @staticmethod
    def get_titles_separated_by_new_line(text):
        from travel.rasp.admin.scripts.schedule.bus.ipektur import IpekturStationTitle

        titles = []

        for title in text.split(u'\n'):
            title = title.strip()

            if not title:
                continue

            titles.append(IpekturStationTitle(title))

        return titles


class IpekturStopsOrderAdmin(RaspExportModelAdmin):
    form = IpekturStopsOrderAdminForm

    list_display = ('id', 'use', 'original_order', 'right_order')
