# -*- coding: utf-8 -*-

from functools import update_wrapper

from django.contrib import messages
from django.utils.translation import ugettext_lazy as _
from django.http import HttpResponseRedirect, HttpResponseBadRequest
from django.utils.http import urlencode
from django.shortcuts import get_object_or_404, render

from common.utils.date import uni_strftime
from travel.rasp.admin.importinfo.models.two_stage_import import PackageDirectionsSlice, DirectionSlice
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin, OrderingOnlyOnSmallQuerysetModelAdminMixin


class DirectionSliceAdmin(OrderingOnlyOnSmallQuerysetModelAdminMixin, RaspExportModelAdmin):
    list_display = ('package_slice', 'direction_title', 'last_mask_date', 'starts_quantity')
    readonly_fields = ('direction_title', 'last_mask_date', 'starts_quantity')
    ordering = ('-package_slice__creation_dt', 'direction_title')

    actions = None

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False


class PackageDirectionsSliceAdmin(RaspExportModelAdmin):
    list_display = ('package', 'creation_dt')
    readonly_fields = ('package', 'creation_dt')
    raw_id_fields = ('package',)

    search_fields = ('package__title', 'package__supplier__title', 'package__supplier__code')
    list_filter = ('package__t_type', 'package__content_manager',
                   'package__package_type', 'package__supplier')

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def compare(self, request, queryset):
        slices = list(queryset)

        if len(slices) != 2:
            messages.error(request, _(u'Для сравнения нужно выбрать 2 среза'))

            return

        return HttpResponseRedirect(request.path + 'compare/?' + urlencode({
            'first_id': slices[0].id,
            'second_id': slices[1].id,
        }))

    compare.short_description = _(u'Сравнить 2 среза')

    actions = [compare]

    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)

            return update_wrapper(wrapper, view)

        urlpatterns = super(PackageDirectionsSliceAdmin, self).get_urls()

        urlpatterns = [
            url(r'^compare/', wrap(self.compare_slices)),
        ] + urlpatterns

        return urlpatterns

    def compare_slices(self, request):
        try:
            first_id = int(request.GET.get('first_id'))
            second_id = int(request.GET.get('second_id'))
        except (TypeError, ValueError):
            return HttpResponseBadRequest(
                u'%s' % _(u'Для сравнения нужно выбрать два среза данных.'),
                content_type='text/plain; charset=utf-8'
            )

        old_slice = get_object_or_404(PackageDirectionsSlice, id=first_id)
        new_slice = get_object_or_404(PackageDirectionsSlice, id=second_id)

        if old_slice.package != new_slice.package:
            return HttpResponseBadRequest(
                u'%s' % _(u'Для сравнения оба среза данных должны принадлежать одному пакету.'),
                content_type='text/plain; charset=utf-8'
            )

        if old_slice.creation_dt > new_slice.creation_dt:
            old_slice, new_slice = new_slice, old_slice

        page_title = _(u'Сравнение срезов пакета %s. %s - %s') % (
            old_slice.package,
            uni_strftime('%d %B %Y %H:%M', old_slice.creation_dt),
            uni_strftime('%d %B %Y %H:%M', new_slice.creation_dt)
        )

        all_titles = DirectionSlice.objects.filter(package_slice_id__in=(new_slice, old_slice))\
                                           .values_list('direction_title', flat=True)\
                                           .distinct()\
                                           .order_by('direction_title')

        old_titles = {}
        old_slices = DirectionSlice.objects.filter(package_slice=old_slice)\
                                           .values_list('direction_title', 'last_mask_date', 'starts_quantity')
        for title, last_mask_date, starts_quantity in old_slices:
            days = (last_mask_date - old_slice.creation_dt.date()).days + 1
            old_titles[title] = {
                'last_mask_date': last_mask_date,
                'thread_per_day': (float(starts_quantity) / days) if days != 0 else None,
                'days': days,
            }

        new_titles = {}
        new_slices = DirectionSlice.objects.filter(package_slice=new_slice)\
                                           .values_list('direction_title', 'last_mask_date', 'starts_quantity')
        for title, last_mask_date, starts_quantity in new_slices:
            days = (last_mask_date - new_slice.creation_dt.date()).days + 1
            new_titles[title] = {
                'last_mask_date': last_mask_date,
                'thread_per_day': (float(starts_quantity) / days) if days != 0 else None,
                'days': days,
            }

        outdated_directions = []
        brand_new_directions = []

        rows = []
        for title in all_titles:
            row = {
                'old_title': '',
                'old_last_mask_date': '',
                'old_thread_per_day': '',
                'old_days': '',

                'new_title': '',
                'new_last_mask_date': '',
                'new_thread_per_day': '',
                'new_days': '',

                'has_per_day_diff': False,
                'has_days_diff': False,
            }

            if title in old_titles:
                row['old_title'] = title
                data = old_titles[title]
                row['old_last_mask_date'] = uni_strftime('%d %B %Y', data['last_mask_date'])
                row['old_thread_per_day'] = data['thread_per_day']
                row['old_days'] = data['days']
            else:
                brand_new_directions.append(title)

            if title in new_titles:
                row['new_title'] = title
                data = new_titles[title]
                row['new_last_mask_date'] = uni_strftime('%d %B %Y', data['last_mask_date'])
                row['new_thread_per_day'] = data['thread_per_day']
                row['new_days'] = data['days']
            else:
                outdated_directions.append(title)

            if row['old_thread_per_day'] and row['new_thread_per_day']:
                old_value = row['old_thread_per_day']
                new_value = row['new_thread_per_day']
                if abs(old_value - new_value) / max(old_value, new_value) > 0.1:
                    row['has_per_day_diff'] = True

            if row['old_days'] and row['new_days']:
                old_value = row['old_days']
                new_value = row['new_days']
                if float(abs(old_value - new_value)) / max(old_value, new_value) > 0.1:
                    row['has_days_diff'] = True

            rows.append(row)

        return render(
            request,
            'admin/importinfo/packagedirectionsslice/compare.html',
            {
                'title': page_title,
                'outdated_directions': outdated_directions,
                'brand_new_directions': brand_new_directions,
                'full_compare': rows,
            }
        )
