# coding: utf-8
import logging

from django import forms
from django.contrib import admin
from django.contrib.admin import SimpleListFilter
from django.utils.translation import ugettext_lazy as _

from travel.rasp.admin.importinfo.models.trusted_station import TrustedStation
from travel.rasp.admin.importinfo.models.two_stage_import import TwoStageImportPackage
from travel.rasp.admin.lib.admin_options import AllLookupModelAdmin

log = logging.getLogger('/home/kateov/workspace/admin/log/trusted_station.log')


class CustomPackageFilter(SimpleListFilter):
    title = _(u'Пакеты')
    parameter_name = 'tsi_package__id__exact'

    def lookups(self, request, model_admin):
        package_ids = TrustedStation.objects.values_list('tsi_package', flat=True).distinct()
        packages = TwoStageImportPackage.objects.in_bulk(package_ids).items()

        return packages

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(tsi_package=self.value())

        else:
            return queryset


class TrustedStationAdminForm(forms.ModelForm):
    class Meta:
        model = TrustedStation
        exclude = []

    @staticmethod
    def get_initial_tsi_package(**kwargs):
        # если в форму зашли с пакета, прокидываем номер пакета в значение пакета по-умолчанию
        if 'initial' in kwargs:
            if u'_changelist_filters' in kwargs['initial']:
                filters = kwargs['initial'][u'_changelist_filters']
                var, value = filters.split('=')
                if var == u'tsi_package':
                    return value
        return None

    def __init__(self, *args, **kwargs):
        initial_tsi_package = self.get_initial_tsi_package(**kwargs)
        if initial_tsi_package:
            super(TrustedStationAdminForm, self).__init__(*args, **{'initial': {u'tsi_package': initial_tsi_package}})
        else:
            super(TrustedStationAdminForm, self).__init__(*args, **kwargs)

        self.fields['tsi_package_group'].queryset = self.fields['tsi_package_group'].queryset.order_by('title')

        if initial_tsi_package is None:
            if self.instance:
                initial_tsi_package = getattr(self.instance, 'tsi_package', None)
        if initial_tsi_package:
            self.fields['tsi_package_group'].queryset = \
                self.fields['tsi_package_group'].queryset.filter(package=initial_tsi_package)

    def clean(self):
        package = self.cleaned_data['tsi_package']
        group = self.cleaned_data['tsi_package_group']
        if group and group.package != package:
            raise forms.ValidationError(_(u'Группа {} не принадлежит пакету {}').format(group, package))
        return self.cleaned_data


@admin.register(TrustedStation)
class TrustedStationAdmin(AllLookupModelAdmin):
    form = TrustedStationAdminForm
    list_display = ('get_station_title', 'get_package_title', 'tsi_package_group')
    list_filter = (CustomPackageFilter, )
    raw_id_fields = ('station', 'tsi_package')
    search_fields = ('station__title', 'station__id')

    def get_station_title(self, obj):
        return obj.station.title
    get_station_title.short_description = _(u"Название станции")

    def get_package_title(self, obj):
        return obj.tsi_package.title
    get_package_title.short_description = _(u"Двухступенчатый пакет")
