# -*- coding: utf-8 -*-

import logging
from functools import wraps
from textwrap import dedent

import pytz
from django.db import transaction
from django.utils import html
from django.utils.safestring import mark_safe

from common.models.geo import Settlement, CityMajority, Country, Region, District
from travel.rasp.admin.importinfo.afdbchanges.actions.default_store import store


log = logging.getLogger(__name__)


SIMPLE_SETTLEMENT_FIELDS_AND_EXAMPLES = (
    ('title', u'Москва'),
    ('title_uk', u'Москва'),
    ('title_en', u'Moscow'),
    ('iata', u'MOW'),
    ('sirena_id', u'МОВ'),
)

RU_CASES = [_e[0] for _e in Settlement.L_title.extra['ru']]


def settlement_action(action_name, help_text):
    def register_settlement_action(func):
        @store.model_action(Settlement, action_name, help_text=help_text)
        @wraps(func)
        def wrapper(db_change_el):
            try:
                settlement = Settlement.objects.get(pk=db_change_el.get('object_id'))
            except Settlement.DoesNotExist:
                log.error(u'Не нашли города %s', db_change_el.get('object_id'))
                return

            return func(settlement, db_change_el)

        return func

    return register_settlement_action


@settlement_action(
    'title_ru',
    help_text=u'Пример <dbchange model="www.settlement" type="title_ru" object_id="213" value="Москва" />. '
              u'Если поле изменилось, поля {} затираются.'.format(u', '.join(
                  map(lambda c: u'title_ru_{}'.format(c), RU_CASES)
              ))
)
@transaction.atomic
def change_settlement_title_ru(settlement, db_change_el):
    new_title_ru = db_change_el.get('value').strip()
    log.info(u'Изменяем title_ru у города %s %s с "%s" на "%s"',
             settlement.id, settlement.title_ru, settlement.title_ru, new_title_ru)
    if settlement.title_ru != new_title_ru:
        log.info(u'Название сменилось, затираем падежные поля %s',
                 u', '.join(map(lambda c: u'title_ru_{}'.format(c), RU_CASES)))
        for ru_case in RU_CASES:
            setattr(settlement, 'title_ru_{}'.format(ru_case), u'')

    settlement.title_ru = new_title_ru
    settlement.save()
    log.info(u'Успешно изменили title_ru у города %s %s', settlement.id, settlement.title)


def register_simple_field_settlement_action(field_name, example_value):
    @settlement_action(
        field_name,
        help_text=u'Пример <dbchange model="www.settlement" type="{}" object_id="213" value="{}" />'.format(
            field_name, example_value
        )
    )
    @transaction.atomic
    def change_settlement_simple_field(settlement, db_change_el):
        new_field_value = db_change_el.get('value').strip()
        old_field_value = getattr(settlement, field_name)
        log.info(u'Изменяем %s у города %s %s с "%s" на "%s"',
                 field_name, settlement.id, settlement.title, old_field_value, new_field_value)
        setattr(settlement, field_name, new_field_value)
        settlement.save()
        log.info(u'Успешно изменили %s у города %s %s', field_name, settlement.id, settlement.title)


for f, e in SIMPLE_SETTLEMENT_FIELDS_AND_EXAMPLES:
    register_simple_field_settlement_action(f, e)


@settlement_action('majority', help_text=dedent(u'''\
    Пример <dbchange model="www.settlement" type="majority" object_id="213" value="2" />\
'''))
@transaction.atomic
def change_settlement_majority(settlement, db_change_el):
    try:
        new_majority = CityMajority.objects.get(pk=int(db_change_el.get('value')))
    except (CityMajority.DoesNotExist, ValueError, TypeError):
        log.error(u'Не смогли найти Важность города %s', db_change_el.get('value'))
        return

    old_majority = settlement.majority
    old_majority_id = settlement.majority_id
    log.info(u'Изменяем Важность города %s %s с "%s %s" на "%s %s"',
             settlement.id, settlement.title,
             old_majority_id, old_majority,
             new_majority.id, new_majority)
    settlement.majority = new_majority
    settlement.save()
    log.info(u'Успешно изменили Важность у города %s %s', settlement.id, settlement.title)


@settlement_action('country', help_text=dedent(u'''\
    Пример <dbchange model="www.settlement" type="country" object_id="213" value="225" />\
'''))
@transaction.atomic
def change_settlement_country(settlement, db_change_el):
    try:
        new_country = Country.objects.get(pk=int(db_change_el.get('value')))
    except (Country.DoesNotExist, ValueError, TypeError):
        log.error(u'Не смогли найти Страну города %s', db_change_el.get('value'))
        return

    old_country = settlement.country
    old_country_id = settlement.country_id
    log.info(u'Изменяем Страну города %s %s с "%s %s" на "%s %s"',
             settlement.id, settlement.title,
             old_country_id, old_country,
             new_country.id, new_country)
    settlement.country = new_country
    settlement.save()
    log.info(u'Успешно изменили Страну у города %s %s', settlement.id, settlement.title)


@settlement_action('region', help_text=dedent(u'''\
    Пример <dbchange model="www.settlement" type="region" object_id="213" value="1" />.
    Если value="", то регион у города сбрасывается.
    \
'''))
@transaction.atomic
def change_settlement_region(settlement, db_change_el):
    try:
        if db_change_el.get('value') == u'':
            new_region = None
        else:
            new_region = Region.objects.get(pk=int(db_change_el.get('value')))
    except (Region.DoesNotExist, ValueError, TypeError):
        log.error(u'Не смогли найти Регион города %s', db_change_el.get('value'))
        return

    old_region = settlement.region
    old_region_id = settlement.region_id
    log.info(u'Изменяем Регион города %s %s с "%s %s" на "%s %s"',
             settlement.id, settlement.title,
             old_region_id, old_region,
             new_region and new_region.id, new_region)
    settlement.region = new_region
    settlement.save()
    log.info(u'Успешно изменили Регион у города %s %s', settlement.id, settlement.title)


@settlement_action('district', help_text=dedent(u'''\
    Пример <dbchange model="www.settlement" type="district" object_id="213" value="1" />.
    Если value="", то район у города сбрасывается.
    \
'''))
@transaction.atomic
def change_settlement_district(settlement, db_change_el):
    try:
        if db_change_el.get('value') == u'':
            new_district = None
        else:
            new_district = District.objects.get(pk=int(db_change_el.get('value')))
    except (District.DoesNotExist, ValueError, TypeError):
        log.error(u'Не смогли найти Район города %s', db_change_el.get('value'))
        return

    old_district = settlement.district
    old_district_id = settlement.district_id
    log.info(u'Изменяем Район города %s %s с "%s %s" на "%s %s"',
             settlement.id, settlement.title,
             old_district_id, old_district,
             new_district and new_district.id, new_district)
    settlement.district = new_district
    settlement.save()
    log.info(u'Успешно изменили Район у города %s %s', settlement.id, settlement.title)


@settlement_action('timezone', help_text=dedent(u'''\
    Пример <dbchange model="www.settlement" type="timezone" object_id="213" value="Europe/Moscow" />.\
'''))
@transaction.atomic
def change_settlement_timezone(settlement, db_change_el):
    new_timezone = db_change_el.get('value')
    if new_timezone not in pytz.all_timezones_set:
        log.error(u'Временной зоны "%s" нет в базе временных зон', new_timezone)
        return

    log.info(u'Изменяем Временную зону города %s %s с "%s" на "%s"',
             settlement.id, settlement.title, settlement.time_zone, new_timezone)
    settlement.time_zone = new_timezone
    settlement.save()
    log.info(u'Успешно изменили Временную зону города %s %s', settlement.id, settlement.title)


@settlement_action('geolocation', help_text=mark_safe(
    dedent(u'''\
        Пример {}.<br>
        lng - долгота, диапазон от -180 до 180.<br>
        lat - широта, диапазон от -90 до 90.\
    ''').format(html.escape(u'<dbchange model="www.settlement" type="geolocation"'
                            u' object_id="213" lat="50.233" lng="23.50" />'))
))
@transaction.atomic
def change_settlement_geolocation(settlement, db_change_el):
    try:
        lat = float(db_change_el.get('lat'))
    except (ValueError, TypeError):
        log.error(u'Не удалось разобрать широту "%s"', db_change_el.get('lat'))
        return
    try:
        lng = float(db_change_el.get('lng'))
    except (ValueError, TypeError):
        log.error(u'Не удалось разобрать долготу "%s"', db_change_el.get('lng'))
        return

    if not (
        (-90 <= lat <= 90) and (-180 <= lng <= 180)
    ):
        log.error(u'Широта и долгота не попали в допустимый диапазон')
        return

    log.info(u'Изменяем Координаты города %s %s с lat="%s" lng="%s" на lat="%s" lng="%s"',
             settlement.id, settlement.title,
             settlement.latitude, settlement.longitude, lat, lng)
    settlement.latitude = lat
    settlement.longitude = lng
    settlement.save()
    log.info(u'Успешно изменили Координаты города %s %s', settlement.id, settlement.title)
