# -*- coding: utf-8 -*-

import logging
from functools import wraps
from textwrap import dedent

import pytz
from django.db import transaction
from django.utils import html
from django.utils.safestring import mark_safe

from common.models.geo import Station, CodeSystem, StationCode
from travel.rasp.admin.importinfo.afdbchanges.actions.default_store import store


log = logging.getLogger(__name__)

RU_CASES = [_e[0] for _e in Station.L_title.extra['ru']]


def station_action(action_name, help_text):
    def register_station_action(func):
        @store.model_action(Station, action_name, help_text=help_text)
        @wraps(func)
        def wrapper(db_change_el):
            try:
                station = Station.objects.get(pk=db_change_el.get('object_id'))
            except Station.DoesNotExist:
                log.error(u'Не нашли станции %s', db_change_el.get('object_id'))
                return

            return func(station, db_change_el)

        return func

    return register_station_action


@station_action(
    'title_ru',
    help_text=u'Пример <dbchange model="www.station" type="title_ru" object_id="9600216" value="Москва" />. '
              u'Если поле изменилось, поля {} затираются.'.format(u', '.join(
                  map(lambda c: u'title_ru_{}'.format(c), RU_CASES)
              ))
)
@transaction.atomic
def change_station_title_ru(station, db_change_el):
    new_title_ru = db_change_el.get('value').strip()
    log.info(u'Изменяем title_ru у станции %s %s с "%s" на "%s"',
             station.id, station.title_ru, station.title_ru, new_title_ru)
    if station.title_ru != new_title_ru:
        log.info(u'Название сменилось, затираем падежные поля %s',
                 u', '.join(map(lambda c: u'title_ru_{}'.format(c), RU_CASES)))
        for ru_case in RU_CASES:
            setattr(station, 'title_ru_{}'.format(ru_case), u'')

    station.title_ru = new_title_ru
    station.save()
    log.info(u'Успешно изменили title_ru у станции %s %s', station.id, station.title)


def register_simple_field_station_action(field_name, example_value):
    @station_action(
        field_name,
        help_text=u'Пример <dbchange model="www.station" type="{}" object_id="9600216" value="{}" />'.format(
            field_name, example_value
        )
    )
    @transaction.atomic
    def change_station_simple_field(station, db_change_el):
        new_field_value = db_change_el.get('value').strip()
        old_field_value = getattr(station, field_name)
        log.info(u'Изменяем %s у станции %s %s с "%s" на "%s"',
                 field_name, station.id, station.title, old_field_value, new_field_value)
        setattr(station, field_name, new_field_value)
        station.save()
        log.info(u'Успешно изменили %s у станции %s %s', field_name, station.id, station.title)


register_simple_field_station_action('title', u'Главный Вокзал'),
register_simple_field_station_action('title_uk', u'Главный Вокзал'),
register_simple_field_station_action('title_en', u'Main Station'),


def register_foreign_field_station_action(field_name, example_value):
    fk_field = getattr(Station, field_name).field
    is_required = not fk_field.blank
    model_class = fk_field.rel.model
    add_to_text = u''
    if is_required:
        add_to_text = u'<br>Значение обязательно к заполнению'

    @station_action(field_name, help_text=mark_safe(dedent(u'''\
        Пример {}\
        {}\
    '''.format(
        html.escape(u'<dbchange model="www.station" type="{}" object_id="{}" value="2" />'
                    .format(field_name, example_value)),
        add_to_text
    ))))
    @transaction.atomic
    def change_station_fk_field(station, db_change_el):
        if not db_change_el.get('value'):
            if is_required:
                log.error(u'%s у станции %s обязателен к заполеннию', model_class._meta.verbose_name.capitalize(),
                          db_change_el.get('value'))
                return
            else:
                new_value = None
        else:
            try:
                new_value = model_class.objects.get(pk=int(db_change_el.get('value')))
            except (model_class.DoesNotExist, ValueError, TypeError):
                log.error(u'Не смогли найти %s станции %s', model_class._meta.verbose_name, db_change_el.get('value'))
                return

        old_value = station.majority
        old_value_id = station.majority_id
        log.info(u'Изменяем %s станции %s %s с "%s %s" на "%s %s"',
                 model_class._meta.verbose_name.capitalize(),
                 station.id, station.title,
                 old_value_id, old_value,
                 new_value and new_value.id, new_value)
        setattr(station, field_name, new_value)
        station.save()
        log.info(u'Успешно изменили %s у станции %s %s',
                 model_class._meta.verbose_name.capitalize(), station.id, station.title)


register_foreign_field_station_action('majority', 1)
register_foreign_field_station_action('station_type', 20)
register_foreign_field_station_action('country', 225)
register_foreign_field_station_action('region', 2)
register_foreign_field_station_action('settlement', 213)
register_foreign_field_station_action('district', 2)


@station_action('timezone', help_text=dedent(u'''\
    Пример <dbchange model="www.station" type="timezone" object_id="9600213" value="Europe/Moscow" />.\
'''))
@transaction.atomic
def change_station_timezone(station, db_change_el):
    new_timezone = db_change_el.get('value')
    if new_timezone not in pytz.all_timezones_set:
        log.error(u'Временной зоны "%s" нет в базе временных зон', new_timezone)
        return

    log.info(u'Изменяем Временную зону станции %s %s с "%s" на "%s"',
             station.id, station.title, station.time_zone, new_timezone)
    station.time_zone = new_timezone
    station.save()
    log.info(u'Успешно изменили Временную зону станции %s %s', station.id, station.title)


@station_action('geolocation', help_text=mark_safe(
    dedent(u'''\
        Пример {}.<br>
        lng - долгота, диапазон от -180 до 180.<br>
        lat - широта, диапазон от -90 до 90.\
    ''').format(html.escape(u'<dbchange model="www.station" type="geolocation"'
                            u' object_id="9600213" lat="50.233" lng="23.50" />'))
))
@transaction.atomic
def change_station_geolocation(station, db_change_el):
    try:
        lat = float(db_change_el.get('lat'))
    except (ValueError, TypeError):
        log.error(u'Не удалось разобрать широту "%s"', db_change_el.get('lat'))
        return
    try:
        lng = float(db_change_el.get('lng'))
    except (ValueError, TypeError):
        log.error(u'Не удалось разобрать долготу "%s"', db_change_el.get('lng'))
        return

    if not (
        (-90 <= lat <= 90) and (-180 <= lng <= 180)
    ):
        log.error(u'Широта и долгота не попали в допустимый диапазон')
        return

    log.info(u'Изменяем Координаты станции %s %s с lat="%s" lng="%s" на lat="%s" lng="%s"',
             station.id, station.title,
             station.latitude, station.longitude, lat, lng)
    station.latitude = lat
    station.longitude = lng
    station.save()
    log.info(u'Успешно изменили Координаты станции %s %s', station.id, station.title)


@station_action('code', help_text=mark_safe(
    dedent(u'''\
        Пример {}.<br>
        Если оставить value пустым или не указать, код удалится.
    ''').format(html.escape(
        u'<dbchange model="www.station" type="code" object_id="9600213" system="esr" value="000230" />'))
))
@transaction.atomic
def change_station_code(station, db_change_el):
    new_code = db_change_el.get('value')
    try:
        system = CodeSystem.objects.get(code=db_change_el.get('system'))
    except CodeSystem.DoesNotExist:
        log.error(u'Не нашли систему кодирования %s', db_change_el.get('system'))
        return

    if not new_code:
        station.code_set.get(system=system).delete()
        log.info(u'Удалили код станции %s %s в системе %s', system.title, station.id, system.title)
        return

    try:
        code = station.code_set.get(system=system)
    except StationCode.DoesNotExist:
        code = StationCode(system=system, station=station)

    old_code = code.code
    code.code = new_code
    code.save()
    log.info(u'Успешно изменили Код станции %s %s в системе %s с "%s" на "%s"',
             station.id, station.title, system.title, old_code, code)
