# -*- coding: utf-8 -*-

import logging
import re

from common.models.geo import Station
from common.utils.caching import cache_until_switch
from travel.rasp.admin.importinfo.models.mappings import MappingContext, StationMapping
from travel.rasp.admin.lib.adminutils import get_rasp_admin_url
from travel.rasp.admin.lib.mail import mail_datachanges
from travel.rasp.admin.scripts.schedule.utils.supplier_station import SupplierStation


log = logging.getLogger(__name__)


class StationFinder(object):
    log = None
    supplier = None
    show_log = True
    old_multiple_stations_behaviour = True

    def __init__(self):
        self.messages = []

    space_re = re.compile(ur'\s+', re.U)

    @classmethod
    def clean_title(cls, title):
        if title is None:
            return u""

        if isinstance(title, basestring):
            title = cls.space_re.sub(u" ", title).strip()
            title = title.replace(u" -", u"-").replace(u"- ", u"-")
        else:
            title = unicode(title)

        return title

    @classmethod
    def clean_code(cls, code):
        if code is None:
            return u""
        else:
            code = unicode(code)
            code = cls.space_re.sub(u"", code).strip()
        return code

    def info(self, msg, **kwargs):
        msg = self.form_message(msg, **kwargs)

        if self.show_log:
            self.log.info(msg)

        self.messages.append(msg)

    def error(self, msg, **kwargs):
        msg = self.form_message(msg, **kwargs)

        if self.show_log:
            self.log.error(msg)

        self.messages.append(msg)

    def form_message(self, msg, **kwargs):
        msg += u" "
        args = []
        for k, v in kwargs.items():
            args.append(k + u"='" + unicode(v) + u"'")

        msg += u" ".join(args)
        return msg

    def find_station(self, title=u"", code=u"", number=None, route_title=None,
                     region=None, **kwargs):

        title = self.clean_title(title)
        code = self.clean_code(code)

        station = self.simple_search(title, code, number, route_title, region)

        if station is None:
            # Не нашли станцию, запускаем поиск для поставщика по умолчанию
            return self.default_search(title, code, number, route_title, region=region, **kwargs)
        else:
            return station

    find = find_station

    def create_mapping_if_not_exist(self, title, code, station):
        if not StationMapping.objects.filter(code=code, title=title, supplier=self.supplier):
            StationMapping.objects.create(code=code, station=station, title=title, supplier=self.supplier)
        self.info(u"Привязали станцию %s %s %s:title=%s,code=%s" %
                  (station.id, station.title, get_rasp_admin_url(station), title, code))

    def update_or_create_mapping(self, title, code, station):
        try:
            mapping = StationMapping.objects.get(code=code, title=title, supplier=self.supplier)
            old_station = mapping.station
            mapping.station = station

            mapping.save()
            self.info(u"Обновили привязку %s %s %s:title=%s,code=%s, old_station= %s %s %s" %
                      (station.id, station.title, get_rasp_admin_url(station), title, code,
                       old_station.id, old_station.title, get_rasp_admin_url(old_station)))
        except StationMapping.DoesNotExist:
            StationMapping.objects.create(code=code, station=station, title=title, supplier=self.supplier)
            self.info(u"Привязали станцию %s %s %s:title=%s,code=%s" %
                      (station.id, station.title, get_rasp_admin_url(station), title, code))
        except StationMapping.MultipleObjectsReturned:
            self.error(u"Несколько привязок по параметрам %s %s %s" %
                       (title, code, self.supplier.code))

    def simple_search(self, title, code, number, route_title, region):
        try:
            stations = StationMapping.get_station(
                self.supplier, title, code, number, route_title, region=region,
                show_info=self.show_log)

            if isinstance(stations, Station):
                return stations

            if self.old_multiple_stations_behaviour:
                self.error(u"Несколько объектов по парамерам, берем первый",
                           title=title, code=code, number=number,
                           route_title=route_title, region=region)
                return stations[0]
            else:
                self.error(u"Несколько объектов по парамерам",
                           title=title, code=code, number=number,
                           route_title=route_title, region=region)
                raise Station.MultipleObjectsReturned

        except Station.DoesNotExist:
            pass

    def default_search(self, title=None, code=None, number=None,
                       route_title=None, region=None, **kwargs):
        raise NotImplementedError

    def mail_errors(self):
        if self.messages:
            mail_datachanges(*self.get_report())

    def get_report(self):
        subject = u"Новые станции и ошибки от %s - %s" %\
                  (self.supplier.title, self.supplier.code)

        self.messages = list(set(self.messages))

        self.messages.sort()
        message = u"\n\n".join(self.messages)
        return subject, message

    def add_report_to_reporter(self, reporter):
        if self.messages:
            reporter.add_report(*self.get_report())


class Finder2SupplierStation(SupplierStation):
    def create_station(self):
        raise NotImplementedError()

    def get_existed_station(self):
        return None


class MappingSearcher(object):
    def __init__(self, supplier, finder):
        self.supplier = supplier
        self.finder = finder


class SF2Searcher(MappingSearcher):
    def search(self, station_data):
        mappings = self.get_mappings(station_data)
        if mappings.count() > 1:
            self.finder.error(u"Несколько объектов по парамерам %s" % station_data)
            raise Station.MultipleObjectsReturned
        elif mappings.count() == 1:
            return mappings[0].station
        else:
            return None

    def get_mappings(self, station_data):
        title = station_data.title or u""
        code = station_data.code or u""
        return StationMapping.objects.filter(supplier=self.supplier, title=title, code=code)


class SF2ContextSearcher(SF2Searcher):
    def search(self, supplier_station):
        if not supplier_station.context:
            return super(SF2ContextSearcher, self).search(supplier_station)

        mappings = self.get_all_mappings(supplier_station)
        if mappings.count() > 1:
            similar_mapping = self.get_mapping_with_similar_context(mappings, supplier_station.context)
            if similar_mapping:
                return similar_mapping.station

            mappings = mappings.filter(context__isnull=True)

        if mappings.count() > 1:
            self.finder.error(u"Несколько соответствий по парамерам %s" % supplier_station)
            raise Station.MultipleObjectsReturned
        elif mappings.count() == 1:
            return mappings[0].station
        else:
            return None

    def get_mappings(self, station_data):
        title = station_data.title or u""
        code = station_data.code or u""

        mappings = StationMapping.objects.filter(supplier=self.supplier, title=title, code=code,
                                                 context__isnull=True)

        if not mappings:
            mappings = StationMapping.objects.filter(supplier=self.supplier, code=code,
                                                     context__isnull=True)

        return mappings

    def get_all_mappings(self, station_data):
        title = station_data.title or u""
        code = station_data.code or u""
        mappings = StationMapping.objects.filter(supplier=self.supplier, title=title, code=code)

        if not mappings:
            mappings = StationMapping.objects.filter(supplier=self.supplier, code=code)

        return mappings

    def get_mapping_with_similar_context(self, mappings, context):
        mappings.select_related('context')
        mappings.order_by('-context__priority')
        for mapping in mappings:
            if self.is_similar_contexts(mapping.context, context):
                return mapping

    def is_similar_contexts(self, db_context, context):
        if db_context is None:
            return False

        non_empty_fields = []

        for f in MappingContext.CONTEXT_FIELDS:
            if getattr(db_context, f):
                non_empty_fields.append(f)

        return all(getattr(db_context, f) == getattr(context, f) for f in non_empty_fields)


class StationFinder2(object):
    def __init__(self, supplier, log, show_log=True, searcher_class=None):
        self.messages = set()
        self.supplier = supplier
        self.log = log
        self.show_log = show_log
        searcher_class = searcher_class or SF2Searcher
        self.searcher = searcher_class(self.supplier, self)

    def info(self, msg, **kwargs):
        msg = self.form_message(msg, **kwargs)

        if self.show_log:
            self.log.info(msg)

        self.messages.add(msg)

    def error(self, msg, **kwargs):
        msg = self.form_message(msg, **kwargs)

        if self.show_log:
            self.log.error(msg)

        self.messages.add(msg)

    def form_message(self, msg, **kwargs):
        msg += u" "
        args = []
        for k, v in kwargs.items():
            args.append(k + u"='" + unicode(v) + u"'")

        msg += u" ".join(args)
        return msg

    @cache_until_switch
    def find(self, station_data):

        station = self.search(station_data)

        if station is None:
            return self.get_station_from_station_data(station_data)
        else:
            return station

    def create_mapping_if_not_exist(self, station_data, station):
        if not StationMapping.objects.filter(code=station_data.code, title=station_data.title,
                                             supplier=self.supplier):
            StationMapping.objects.create(code=station_data.code, station=station,
                                          title=station_data.title, supplier=self.supplier)
        self.info(u"Привязали станцию %s %s %s: %s" %
                  (station.id, station.title, get_rasp_admin_url(station), station_data))

    def update_or_create_mapping(self, station_data, station):
        try:
            mapping = StationMapping.objects.get(code=station_data.code,
                                                 title=station_data.title, supplier=self.supplier)
            old_station = mapping.station
            mapping.station = station

            mapping.save()
            self.info(u"Обновили привязку %s %s %s: %s, old_station= %s %s %s" %
                      (station.id, station.title, get_rasp_admin_url(station), station_data,
                       old_station.id, old_station.title, get_rasp_admin_url(old_station)))
        except StationMapping.DoesNotExist:
            StationMapping.objects.create(code=station_data.code, station=station,
                                          title=station_data.title, supplier=self.supplier)
            self.info(u"Привязали станцию %s %s %s: %s" %
                      (station.id, station.title, get_rasp_admin_url(station), station_data))
        except StationMapping.MultipleObjectsReturned:
            self.error(u"Несколько привязок по параметрам %s %s" %
                       (station_data, self.supplier.code))

    def search(self, station_data):
        return self.searcher.search(station_data)

    def get_station_from_station_data(self, station_data):
        station = station_data.get_existed_station()
        if station:
            self.info(u"Берем станцию %s %s %s %s" %
                      (station.id, station.title, get_rasp_admin_url(station), station.L_full_geography()))
        else:
            station = station_data.create_station()
            self.info(u"Создали в %s станцию %s %s %s" %
                      (station.L_full_geography(), station.id, station.title, get_rasp_admin_url(station)))

        self.create_mapping_if_not_exist(station_data, station)

        return station

    def mail_errors(self):
        if self.messages:
            mail_datachanges(*self.get_report())

    def get_report(self):
        messages = list(self.messages)
        subject = u"Новые станции и ошибки от %s - %s" %\
                  (self.supplier.title, self.supplier.code)

        messages.sort()

        message = u"\n\n".join(messages)
        return subject, message

    def add_report_to_reporter(self, reporter):
        if self.messages:
            reporter.add_report(*self.get_report())
