# -*- coding: utf-8 -*-

import base64
import gzip
import logging
import re
import zlib
from functools import partial
from xml.etree import cElementTree as ElementTree

from django.conf import settings
from django.db import connection, transaction, models
from django.db.models.signals import pre_delete
from django.core.exceptions import ValidationError
from django.contrib.contenttypes.fields import GenericForeignKey
from django.contrib.contenttypes.models import ContentType
from django.utils.translation import ugettext_lazy as _

from common.models.transport import TransportType
from common.utils.caching import cache_method_result, cache_until_switch
from common.utils.fields import TrimmedCharField, TextFileField, DatabaseFileField, CodeCharField, RegExpField
from travel.rasp.admin.lib.fields import PickledObjectField
from common.models_utils import TABLO_MANUAL_STATUS_CHOISES, EVENT_CHOISES
from common.utils.regexp import RegExpSet
from travel.rasp.admin.importinfo.blacklist import match_supplier_route
from travel.rasp.admin.importinfo.models.model_extenders import ImportReportExtender, Route2TerminalExtender, TabloManualStatusExtender
from travel.rasp.admin.importinfo.models.validators import is_xml_validator
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException

# Регистрируем модели, но не импортируем сюда
import common.importinfo.models.bus  # noqa
import travel.rasp.admin.importinfo.models.af  # noqa
import travel.rasp.admin.importinfo.models.bus  # noqa
import travel.rasp.admin.importinfo.models.content_manager  # noqa
import travel.rasp.admin.importinfo.models.e_traffic  # noqa
import travel.rasp.admin.importinfo.models.ipektur  # noqa
import travel.rasp.admin.importinfo.models.mappings  # noqa
import travel.rasp.admin.importinfo.models.tablo  # noqa
import travel.rasp.admin.importinfo.models.trusted_station  # noqa
import travel.rasp.admin.importinfo.models.two_stage_import  # noqa


log = logging.getLogger(__name__)


class ImportReport(models.Model, ImportReportExtender):
    u"""Информация о импорте от поставщика"""
    imported = models.DateTimeField(verbose_name=_(u"Дата создаения отчета"),
                                    auto_now_add=True, blank=True, null=False,
                                    editable=False)
    route_set = PickledObjectField(verbose_name=_(u"Список рейсов"),
                                   null=True, editable=False)
    route_count = models.IntegerField(verbose_name=_(u"Количество рейсов"),
                                      null=False, default=0, editable=False)
    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u"Поставщик"), null=False,
                                 editable=False)

    class Meta:
        verbose_name = _(u"\u0460: Отчет импорта")
        verbose_name_plural = _(u"\u0460: Отчеты о импорте")
        ordering = ('-imported',)

    def __unicode__(self):
        return u"Отчет о импорте от %s, за %s" % (self.supplier.title, self.imported)


class Route2Terminal(models.Model, Route2TerminalExtender):
    u"""
    Исключения при привязке аэрорейсов к терминалам
    """
    number = TrimmedCharField(verbose_name=u"Номер рейса",
                              max_length=255, null=False)
    terminal = models.ForeignKey('www.StationTerminal', verbose_name=_(u"Терминал"),
                                 null=False)

    class Meta:
        unique_together = [('number', 'terminal')]
        verbose_name = _(u"\u0460: Привязка рейска к терминалу")
        verbose_name_plural = _(u"\u0460: Привязки рейсов к терминалам")


class TariffFile(models.Model):
    last_log = u""
    u"""Файлы для импорта тарифов"""
    tariff_file_name = models.CharField(blank=True, max_length=255,
                                        null=False,
                                        verbose_name=_(u"Название файла"))
    tariff_file = TextFileField(blank=True, default='', verbose_name=_(u"Файл с обновлениями"),
                                encoding='cp1251', null=True,
                                help_text=_(u"Принимаются только текстовые файлы в кодировке cp1251"))
    tariff_rich_file = DatabaseFileField(_(u'Файл с тарифами'), null=True, blank=True,
                                         help_text=_(u'Принимаем файлы xml, в том числе в сжатом виде file.xml.gz'))
    added = models.DateTimeField(auto_now_add=True,
                                 verbose_name=_(u"Добавлен"))
    load_log = models.TextField(null=True, blank=True)
    imported = models.DateTimeField(verbose_name=_(u"Дата импорта файла"), default=None,
                                    blank=True, null=True,
                                    help_text=_(u"Если не заполнена, занчит файл еще не импортировался"))

    def __unicode__(self):
        return self.tariff_file_name or self.tariff_rich_file.name

    class Meta:
        ordering = ('-added',)
        verbose_name = _(u'Электрички: Файл с тарифами')
        verbose_name_plural = _(u'Электрички: Файлы тарифов')

    def get_xml_root(self):
        if self.tariff_file_name:
            log.info(u'Берем старый файл для импорта')
            return ElementTree.fromstring(self.tariff_file.encode('cp1251'))

        log.info(u'Берем новый файл для импорта: %s', self.tariff_rich_file.name)
        if self.tariff_rich_file.name.endswith('.xml.gz'):
            log.info(u'Распаковываем архив')
            return ElementTree.parse(gzip.GzipFile(filename=self.tariff_rich_file.name, mode='rb',
                                                   fileobj=self.tariff_rich_file)).getroot()
        else:
            return ElementTree.parse(self.tariff_rich_file).getroot()


class CompanySynonym(models.Model):
    u"""
    Синонимы компаний для импорта.
    """
    title = CodeCharField(verbose_name=_(u"Синоним"), max_length=255, blank=False, null=False)
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u"Тип транспорта перевозчика"),
                               blank=False, null=False)
    company = models.ForeignKey('www.Company', verbose_name=_(u"Компания"), blank=False, null=False)

    def __unicode__(self):
        return u"%s ~ (%s %s)[%s]" % (self.title, self.company.id, self.company.title,
                                      self.t_type.code)

    class Meta:
        verbose_name = _(u"\u0460: Синоним перевозчика")
        verbose_name_plural = _(u"\u0460: Синонимы перевозчиков")
        unique_together = (('title', 't_type'),)


class TabloManualStatus(models.Model, TabloManualStatusExtender):
    """
    Ручная установка статусов в табло
    """
    station = models.ForeignKey('www.Station', blank=False, null=False,
                                verbose_name=_(u"Станция"))
    event = models.CharField(max_length=50, choices=EVENT_CHOISES, blank=False, null=False,
                             verbose_name=_(u"Событие"))
    number = RegExpField(max_length=100, blank=False, null=False,
                         verbose_name=_(u"Номер"), help_text=_(u"Регулярное выражение"))
    planned = models.DateTimeField(null=False, blank=False, verbose_name=_(u"Время по расписанию"))
    real = models.DateTimeField(null=False, blank=False, verbose_name=_(u"Время фактическое"))
    status = models.CharField(max_length=20, choices=TABLO_MANUAL_STATUS_CHOISES,
                              blank=False, null=False, verbose_name=_(u"Статус"))

    def __unicode__(self):
        return u" - ".join(map(unicode, [self.station.title, self.event, self.number, self.status,
                                         self.planned, self.real]))

    def save(self, **kwargs):
        self.add_manual_statuses()
        super(TabloManualStatus, self).save(**kwargs)

    class Meta:
        verbose_name = _(u"Табло: Ручной статус")
        verbose_name_plural = _(u"Табло: Ручные статусы")


def tablo_manual_status_predelete(sender, instance, **kwargs):
    instance.remove_manual_statuses()

pre_delete.connect(tablo_manual_status_predelete, TabloManualStatus)


class TabloImportSynonym(models.Model):
    title = models.CharField(_(u"Название в данных поставщика"), max_length=255, null=False)
    supplier = models.ForeignKey('www.Supplier', null=False, blank=False,
                                 verbose_name=_(u"Поставщик табло"))
    station = models.ForeignKey('www.Station', verbose_name=_(u"Станция"), null=True, blank=True)
    settlement = models.ForeignKey('www.Settlement', verbose_name=_(u"Город"), null=True, blank=True)

    class Meta:
        verbose_name = _(u"Табло: Синоним для импорта")
        verbose_name_plural = _(u"Табло: Синонимы для импорта")
        unique_together = (('title', 'supplier'),)

    @classmethod
    def get_title(cls, title, supplier):
        synonym = cls.objects.get(title=title, supplier=supplier)

        if synonym.station:
            return synonym.station.settlement.title

        elif synonym.settlement:
            return synonym.settlement.title

        else:
            raise cls.BadSynonym(_(u"Неверный синоним с id %s") % synonym.id)

    @classmethod
    def get_settlement(cls, some_title, supplier):
        try:
            synonym = cls.objects.get(title=some_title, supplier=supplier)
        except cls.DoesNotExist:
            return

        if synonym.settlement_id:
            return synonym.settlement
        elif synonym.station_id:
            return synonym.station.settlement
        else:
            raise cls.BadSynonym(_(u"Неверный синоним с id %s") % synonym.id)

    class BadSynonym(SimpleUnicodeException):
        pass


class BlackList(models.Model):
    """ Черный список рейсов """

    number = RegExpField(verbose_name=_(u'Номер рейса'), max_length=100,
                         blank=True, help_text=_(u'Номер рейса, который мы показываем'
                                                 u'(можно задавать регулярым выражением)'))
    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u'Поставщик'), null=False, blank=False)
    thread_type = models.ForeignKey('www.RThreadType', null=True, blank=True)
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'), blank=False)
    comment = models.TextField(verbose_name=_(u"Комментарий"), blank=False, help_text=_(
        u"Если ни одно поле не заполнено, и не заполнены станции ниток,"
        u" под черный список будут попадать все маршруты поставщика, с данным типом транспорта!!!"
    ))

    # extended group
    supplier_number = RegExpField(
        verbose_name=_(u'Номер рейса у поставщика'), max_length=300, blank=True,
        help_text=_(u'Номер поставщика, если не общий XML или треугольный, то возможный преобразования,'
                    u' например 160x2_olven Олвен, ast01x103x1030 ДЮК, 102 Караганда.'
                    u' (можно задавать регулярым выражением)')
    )
    supplier_title = RegExpField(
        verbose_name=_(u'Название рейса у поставщика'), max_length=300, blank=True,
        help_text=_(u'Преобразования минимальный, обрезаются только пробелы.'
                    u' Названия может строится из нескольких полей у поставщика, например у ДЮК СОКОЛ - 1ВОЛОГДА.'
                    u' (можно задавать регулярым выражением)')
    )
    start_time_start = models.TimeField(_(u"Время старта нитки, начало диапазона"), null=True, default=None, blank=True)
    start_time_end = models.TimeField(
        _(u"Время старта нитки, конец диапазона"), null=True, default=None, blank=True,
        help_text=_(u"Если конец диапазона не указан, ищется только совпадение с началом диапазона,"
                    u" если начало диапазона не указано, конец диапазона игнорируется.")
    )
    finish_time_start = models.TimeField(_(u"Время прибытия на конечную, начало диапазона"), null=True, default=None,
                                         blank=True)
    finish_time_end = models.TimeField(
        _(u"Время прибытия на конечную, конец диапазона"), null=True, default=None, blank=True,
        help_text=_(u"Если конец диапазона не указан, ищется только совпадение с началом диапазона, если начало"
                    u" диапазона не указано, конец диапазона игнорируется.")
    )
    is_moscow_time = models.BooleanField(_(u"Московское время"), default=False,
                                         help_text=_(u"По умолчанию время местное"))
    start_station = models.ForeignKey("www.Station", verbose_name=_(u"Начальная станция"), default=None, null=True,
                                      blank=True, related_name="black_list_start_stations")
    finish_station = models.ForeignKey("www.Station", verbose_name=_(u"Конечная станция"), default=None, null=True,
                                       blank=True, related_name="black_list_finish_stations")

    def save(self, **kwargs):
        self.number = self.number.strip()
        super(BlackList, self).save(**kwargs)

    def __unicode__(self):
        return self.t_type.title_ru + ': ' + self.comment

    class Meta:
        verbose_name = _(u'\u0460: Рейс черного списка')
        verbose_name_plural = _(u'\u0460: Черный список рейсов')
        ordering = ('t_type', 'number')

    @classmethod
    @cache_until_switch
    def get_regexp_set(cls, **filter_kwargs):
        str_list = [bl.number
                    for bl in
                    cls.objects.filter(**filter_kwargs) if bl.number]
        return RegExpSet(str_list)

    @classmethod
    def has_thread(cls, thread):
        return cls.has_record(thread.number, thread.supplier, thread.t_type, thread.company)

    @classmethod
    def has_record(cls, number, supplier, t_type, company):
        possible_numbers = []
        if t_type.id == TransportType.PLANE_ID and company:
            flight_number = number.split(settings.AVIAROUTE_NUMBER_DELIMITER)[1]
            if company.iata:
                possible_numbers.append(company.iata + flight_number)
                possible_numbers.append(company.iata + settings.AVIAROUTE_NUMBER_DELIMITER + flight_number)
            if company.sirena_id:
                possible_numbers.append(company.sirena_id + flight_number)
                possible_numbers.append(company.sirena_id + settings.AVIAROUTE_NUMBER_DELIMITER + flight_number)
            if company.icao:
                possible_numbers.append(company.icao + flight_number)
                possible_numbers.append(company.icao + settings.AVIAROUTE_NUMBER_DELIMITER + flight_number)
        else:
            possible_numbers.append(number)

        return any(number in cls.get_regexp_set(t_type=t_type, supplier=supplier)
                   for number in possible_numbers)

    @classmethod
    def has_train_number(cls, number, supplier_id, thread_type=1):
        if number in cls.get_regexp_set(t_type__code='train', supplier=supplier_id, thread_type=thread_type):
            return True

        if number in cls.get_regexp_set(t_type__code='train', supplier=supplier_id, thread_type__isnull=True):
            return True

        return False

    @property
    @cache_method_result
    def backlist_thread_stations(self):
        return list(self.blacklistthreadstation_set.all())

    def match_supplier_route(self, supplier_route):
        return match_supplier_route(self, supplier_route)


class BlackListThreadStation(models.Model):
    blacklist = models.ForeignKey(BlackList, null=False, blank=False)
    station = models.ForeignKey("www.Station", verbose_name=_(u"Станция"), null=False, blank=False)
    departure_time_start = models.TimeField(_(u"Время отправления, начало диапазона"),
                                            null=True, default=None, blank=True)
    departure_time_end = models.TimeField(
        _(u"Время отправления, конец диапазона"),
        null=True, default=None, blank=True,
        help_text=_(u"Если конец диапазона не указан, ищется только совпадение с началом диапазона, если начало"
                    u" диапазона не указано, конец диапазона игнорируется.")
    )
    arrival_time_start = models.TimeField(_(u"Время прибытия, начало диапазона"),
                                          null=True, default=None, blank=True)
    arrival_time_end = models.TimeField(
        _(u"Время прибытия, конец диапазона"),
        null=True, default=None, blank=True,
        help_text=_(u'Если конец диапазона не указан, ищется только совпадение с началом диапазона,'
                    u' если начало диапазона не указано, конец диапазона игнорируется.')
    )
    is_moscow_time = models.BooleanField(_(u"Московское время"), default=False,
                                         help_text=_(u"По умолчанию время местное"))

    class Meta:
        verbose_name = _(u"Станция нитки черного списка")
        verbose_name_plural = _(u"Станции нитки черного списка")


class CodeshareNumber(models.Model):
    number_re = RegExpField(_(u'Номер рейса, регулярное выражение'), max_length=255, blank=False, null=False)
    comment = models.TextField(_(u'Комментарий'), blank=False, null=False)
    exclude_from_tablo_import = models.BooleanField(_(u'Исключать из обновления табло'), default=True)

    @classmethod
    @cache_until_switch
    def get_excluded_from_tablo_regexp_set(cls):
        number_res = [
            csr.number_re
            for csr in cls.objects.filter(exclude_from_tablo_import=True)
        ]
        return RegExpSet(number_res)

    class Meta:
        db_table = 'importinfo_codeshareroute'
        verbose_name = _(u'Кодшеринговый номер рейса')
        verbose_name_plural = _(u'Кодшеринговые номера рейсов')


class RelatedLog(models.Model):
    object_id = models.PositiveIntegerField(null=True, blank=True, default=None)
    content_type = models.ForeignKey('contenttypes.ContentType', null=True, blank=True, default=None)
    emitter = GenericForeignKey()

    code = CodeCharField(_(u"Код"), null=True, blank=True, default=None, max_length=255,
                         help_text=_(u"Используется в том случае если объект имеет несколько логов"))

    tag = CodeCharField(_(u"Тег"), null=True, blank=True, default=None, max_length=255,
                        help_text=_(u"Используется в том случае если объект имеет несколько логов"))

    # на тот случай, если мы захотим сделать лог зипованым
    log_field_ru = models.TextField(_(u"Лог на русском"), db_column='log_field')
    log_field_en = models.TextField(_(u"Лог на английском"), default='')

    class Meta:
        verbose_name = _(u"\u0460: Лог объекта")
        verbose_name_plural = _(u"\u0460: Логи объектов")
        unique_together = (('code', 'tag'), ('object_id', 'content_type', 'tag'))

    @classmethod
    def create_from_object(cls, obj, tag=None):
        instance = cls()
        instance.emitter = obj
        instance.tag = tag
        instance.save()
        return instance

    @classmethod
    def get_by_object(cls, obj, tag=None):
        logs = cls.objects.filter(object_id=obj.id, content_type=ContentType.objects.get_for_model(obj),
                                  tag=tag)
        if logs:
            return logs[0]
        else:
            return cls.create_from_object(obj, tag)

    @classmethod
    def get_by_model(cls, model, tag=None):
        content_type = ContentType.objects.get_for_model(model)
        return cls.get_by_object(content_type, tag)

    # на тот случай, если мы захотим сделать лог зипованым и т.п.
    def _get_log(self, lang):
        log_field = getattr(self, 'log_field_%s' % lang)

        try:
            return zlib.decompress(
                base64.b64decode(log_field.encode('utf8'))
            ).decode('utf8')

        except Exception:
            # legasy support
            return log_field

    def _set_log(self, data, lang):
        data = base64.b64encode(zlib.compress(data.encode('utf8'), 9))
        setattr(self, 'log_field_%s' % lang, data.decode('utf8'))

    log_ru = property(partial(_get_log, lang='ru'), partial(_set_log, lang='ru'))
    log_en = property(partial(_get_log, lang='en'), partial(_set_log, lang='en'))

    # legacy
    log = log_ru


class IataCorrection(models.Model):
    code = CodeCharField(_(u"Код IATA"), max_length=4, blank=False)
    number = RegExpField(_(u"Номер рейса"), max_length=1000,
                         help_text=_(u"Номер без кода. Можно задавать регулярным выражением"))
    company = models.ForeignKey("www.Company", null=False, blank=False)

    def match_number(self, number):
        if not hasattr(self, '_number_re'):
            self._number_re = re.compile(self.number, re.I + re.U)

        return self._number_re.match(number)

    def __unicode__(self):
        return u"{} {} -> {}".format(self.code, self.number, self.company.title)

    class Meta:
        verbose_name = _(u"Коррекция IATA кода")
        verbose_name_plural = _(u"Коррекции IATA кодов")


class OriginalThreadData(models.Model):
    package = models.ForeignKey('importinfo.TwoStageImportPackage', null=True, default=None, blank=False,
                                editable=False)
    supplier = models.ForeignKey('www.Supplier', null=True, default=None, blank=False, editable=False)
    thread = models.ForeignKey('www.RThread', null=True, default=None, blank=True, editable=False)
    thread_uid = models.CharField(max_length=255, null=False, blank=False, editable=False, db_index=True)
    cysix = models.TextField(null=True, default=None, blank=True, editable=False)
    raw = models.TextField(null=True, default=None, blank=True, editable=False)

    @classmethod
    def fast_delete_by_package_id(cls, package_id, log=None):
        """
        Метод удаляет исходные данные всех ниток пакета
        """
        if log is not None:
            log.info(u'Удаляем исходные данные ниток для пакета %s', package_id)

        with transaction.atomic():
            cursor = connection.cursor()
            query = 'DELETE FROM importinfo_originalthreaddata WHERE package_id=%s'
            cursor.execute(query, [package_id])

        if log is not None:
            log.info(u'Исходные данные ниток для пакета %s удалены', package_id)

    @classmethod
    def fast_delete_by_supplier_id(cls, supplier_id, log=None):
        """
        Метод для заданного поставщика
        удаляет исходные данные ниток, не привязанных ни к одному пакету.
        """
        if log is not None:
            log.info(u'Удаляем исходные данные ниток для поставщика %s', supplier_id)

        with transaction.atomic():
            cursor = connection.cursor()
            query = 'DELETE FROM importinfo_originalthreaddata WHERE package_id IS NULL AND supplier_id=%s'
            cursor.execute(query, [supplier_id])

        if log is not None:
            log.info(u'Исходные данные ниток для поставщика %s удалены', supplier_id)

    class Meta:
        verbose_name = _(u"Исходные данные нитки")
        verbose_name_plural = _(u"Исходные данные ниток")


class TrainTransportOverride(models.Model):
    number = RegExpField(max_length=400, blank=False, null=False,
                         verbose_name=_(u'Номер'), help_text=_(u'Регулярное выражение'))
    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u'Тип транспорта'), null=False,
                               blank=False)
    t_subtype = models.ForeignKey('www.TransportSubtype', verbose_name=_(u'Подтип транспорта'),
                                  blank=True,
                                  null=True, default=None)

    def clean(self):
        if self.t_subtype_id and self.t_subtype.t_type_id != self.t_type_id:
            raise ValidationError(
                _(u'Подтипу транспорта {} соответствует тип транспорта {}')
                .format(self.t_subtype, self.t_subtype.t_type)
            )
