# -*- coding: utf-8 -*-

import gzip
import traceback
from datetime import datetime

from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.models_utils.i18n import L_field
from common.utils.fields import TextFileField, DatabaseFileField
from common.utils.safe_xml_parser import safe_parse_xml


class AFUpdateFile(models.Model):
    last_log = u""
    u"""Файлы обновления от Фетисова"""
    update_file_name = models.CharField(blank=True, max_length=255, null=False,
                                        verbose_name=_(u"Название файла"))
    update_file = TextFileField(verbose_name=_(u"Файл с обновлениями"),
                                encoding='cp1251', is_xml=True, blank=False,
                                help_text=_(u"Принимаются только текстовые файлы в кодировке cp1251"))
    load_log = models.TextField(null=True, blank=True)
    added = models.DateTimeField(auto_now_add=True, verbose_name=_(u"Добавлен"))
    # Если False то файл загрузиться при ежедневном обновлении
    loaded = models.BooleanField(verbose_name=_(u"Импортирован"), default=False, blank=True,
                                 help_text=_(u"Если False то файл загрузиться при ежедневном обновлении"))
    recount_schedule_on_the_fly = models.BooleanField(verbose_name=_(u"Пересчитывать station_schedule на лету"),
                                                      default=False)

    def __unicode__(self):
        return self.update_file_name

    class Meta:
        ordering = ('-added',)
        verbose_name = _(u'Электрички: Файл обновления')
        verbose_name_plural = _(u'Электрички: Файлы обновления')
        app_label = 'importinfo'

    def get_encoded_data(self):
        return self.update_file.encode("cp1251")

    def save(self, loaded=False, **kwargs):
        from travel.rasp.admin.lib.maintenance.flags import flags
        from travel.rasp.admin.lib.maintenance.scripts import job

        if loaded:
            self.loaded = True
            return super(AFUpdateFile, self).save(**kwargs)

        # Идет работа с базой, не загружать обновления
        if flags['maintenance']:
            raise Exception("Error")
        try:

            flags['maintenance'] = job.AF_UPDATE_IMPORT.flag_value
            self.load_log = u""

            from travel.rasp.admin.scripts.schedule.update_af_suburban import update_from_file
            last_log = update_from_file(self.get_encoded_data(), self.update_file_name,
                                        recount_schedule_on_the_fly=self.recount_schedule_on_the_fly)
            self.load_log = last_log
        except Exception:
            exc_info = traceback.format_exc()
            if isinstance(exc_info, str):
                exc_info = exc_info.decode('utf8', 'replace')

            self.load_log += u'\n' + exc_info

        finally:
            flags['maintenance'] = 0

        self.save(loaded=True, **kwargs)


class AFScheduleFile(models.Model):
    last_log = u""
    u"""Файлы расписаний от Фетисова"""
    schedule_file_name = models.CharField(blank=True, max_length=255,
                                          null=False,
                                          verbose_name=_(u"Название файла"))
    schedule_file = TextFileField(blank=False, verbose_name=_(u"Файл с расписаниями"),
                                  encoding='cp1251',
                                  help_text=_(u"Принимаются только текстовые файлы в кодировке cp1251"))
    added = models.DateTimeField(auto_now_add=True,
                                 verbose_name=_(u"Добавлен"))

    imported = models.DateTimeField(verbose_name=_(u"Дата импорта файла"), default=None,
                                    blank=True, null=True,
                                    help_text=_(u"Если не заполнена, значит файл еще не импортировался"))
    region = models.ForeignKey('www.Region', verbose_name=_(u"Область"),
                               help_text=_(u"Куда по умолчанию добавляются новые станции если есть"),
                               blank=False, null=False)

    class Meta:
        ordering = ('-added',)
        verbose_name = _(u'Электрички: Файл расписания')
        verbose_name_plural = _(u'Электрички: Файлы расписания')
        app_label = 'importinfo'

    def __unicode__(self):
        return self.schedule_file_name

    def get_encoded_data(self):
        return self.schedule_file.encode("cp1251")


class AFSubwayFile(models.Model):
    """Файлы метро от АФ"""

    subway_file_name = models.CharField(blank=True, max_length=255, null=False,
                                        verbose_name=_(u'Название архива'))
    stations_file_name = models.CharField(blank=True, max_length=255,
                                          null=False,
                                          verbose_name=_(u"Название файла со станциями"))
    stations_file = TextFileField(verbose_name=_(u'Файл со станциями'),
                                  blank=False, encoding='cp1251',
                                  help_text=_(u"Принимаются только текстовые CSV-файлы в кодировке cp1251"))
    passages_file_name = models.CharField(blank=True, max_length=255,
                                          null=False,
                                          verbose_name=_(u"Название файла с переходами"))
    passages_file = TextFileField(verbose_name=_(u'Файл с переходами'),
                                  blank=False, encoding='cp1251',
                                  help_text=_(u"Принимаются только текстовые CSV-файлы в кодировке cp1251"))
    threads_file_name = models.CharField(blank=True, max_length=255,
                                         null=False,
                                         verbose_name=_(u"Название файла с нитками"))
    threads_file = TextFileField(verbose_name=_(u'Файл с нитками'),
                                 blank=False, encoding='cp1251',
                                 help_text=_(u"Принимаются только текстовые XML-файлы в кодировке cp1251 в формате АФ"))
    settlement = models.ForeignKey('www.Settlement', verbose_name=u'Город',
                                   help_text=_(u'Для какого города грузить метро'),
                                   blank=False, null=False)
    add_mode = models.BooleanField(blank=False, null=False, default=False,
                                   verbose_name=_(u'Режим добавления'),
                                   help_text=_(u'Если режим добавления включен, то перед импортом не происходит'
                                               u' очистки старого метро'))
    load_log = models.TextField(null=True, blank=True)
    imported = models.DateTimeField(verbose_name=_(u"Дата импорта файла"), default=None,
                                    blank=True, null=True,
                                    help_text=_(u"Если не заполнена, занчит файл еще не импортировался"))

    @property
    def stations(self):
        return self.stations_file.encode('cp1251')

    @property
    def passages(self):
        return self.passages_file.encode('cp1251')

    @property
    def threads(self):
        return self.threads_file.encode('cp1251')

    def __unicode__(self):
        return self.subway_file_name

    class Meta:
        ordering = ('-imported',)
        verbose_name = _(u'А.Ф.: Файл обновления метро')
        verbose_name_plural = _(u'А.Ф.: Файлы обновления метро')
        app_label = 'importinfo'

    def save(self, *args, **kwargs):
        if self.imported:
            return super(AFSubwayFile, self).save(*args, **kwargs)

        from travel.rasp.admin.lib.maintenance.flags import flags, get_current_job
        from travel.rasp.admin.lib.maintenance.scripts import job

        self.load_log = u''

        # Идет работа с базой, не загружать обновления
        if flags['maintenance']:
            __, job_name = get_current_job()
            self.load_log = _(u'%s, нельзя запускать импорт метро') % job_name

            return super(AFSubwayFile, self).save(*args, **kwargs)

        try:
            flags['maintenance'] = job.SUBWAY_IMPORT.flag_value
        except Exception:
            exc_info = traceback.format_exc()
            if isinstance(exc_info, str):
                exc_info = exc_info.decode('utf8', 'replace')

            self.load_log += u'\n' + exc_info
        finally:
            flags['maintenance'] = 0

        self.imported = datetime.now()

        return super(AFSubwayFile, self).save(*args, **kwargs)


class AFMaskText(models.Model):
    TANKER_L_FIELDS = ['short_text', 'long_text']
    TANKER_HELP_FIELDS = ['description']
    TANKER_KEY_FIELD = 'code'

    @classmethod
    def tanker_add_short_text_filter(cls, initial_query_set):
        return initial_query_set.filter(short_text_ru__isnull=False) \
                                .exclude(short_text_ru__in=(u'', u'???'))

    code = models.CharField(_(u'Код'), max_length=10, null=False, blank=False)
    description = models.TextField(_(u'Описание для переводчиков'), default=u'', blank=True)
    L_short_text = L_field(_(u'Короткий текст, для комбинаций'), max_length=255, blank=True,
                           default='', base_lang='ru')
    L_long_text = L_field(_(u'Длинный текст, для единичных случаев'), max_length=255, blank=True,
                          default='', base_lang='ru')
    next_code = models.CharField(_(u'Код для +1 дня'), max_length=10, null=False, blank=True,
                                 default=u'')
    prev_code = models.CharField(_(u'Код для -1 дня'), max_length=10, null=False, blank=True,
                                 default=u'')

    def __unicode__(self):
        return self.code

    class Meta:
        verbose_name = _(u'А.Ф.: Преобразование маски в текст')
        verbose_name_plural = _(u'А.Ф.: Преобразования маски в текст')
        app_label = 'importinfo'
        ordering = ('id',)


class AFDBChangesFile(models.Model):
    changes_file = DatabaseFileField(_(u'Файл с изменениями данных'), null=True, blank=False,
                                     help_text=_(u'Принимаем файлы xml, в том числе в сжатом виде file.xml.gz'))
    imported_at = models.DateTimeField(null=True, blank=True)

    def __unicode__(self):
        return self.changes_file.name

    def get_xml_tree(self):
        self.changes_file.open()
        if self.changes_file.name.endswith('.xml.gz'):
            return safe_parse_xml(gzip.GzipFile(fileobj=self.changes_file))
        else:
            return safe_parse_xml(self.changes_file)

    class Meta:
        ordering = ('-imported_at',)
        verbose_name = _(u'А.Ф.: Файл изменения гео данных')
        verbose_name_plural = _(u'А.Ф.: Файл изменения гео данных')
        app_label = 'importinfo'


class AFTrainTurnoverFile(models.Model):
    u"""Файлы оборота поездов"""
    schedule_file_name = models.CharField(blank=True, max_length=255,
                                          null=False,
                                          verbose_name=_(u"Название файла"))
    schedule_file = TextFileField(blank=False, verbose_name=_(u"Файл с оборотами"),
                                  encoding='cp1251',
                                  help_text=_(u"Принимаются только текстовые файлы в кодировке cp1251"))
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_(u"Добавлен"))
    loaded = models.BooleanField(verbose_name=_(u"Импортирован"), default=False, blank=True)
    load_log = models.TextField(null=True, blank=True)

    class Meta:
        ordering = ('-created_at',)
        verbose_name = _(u'Оборот поездов: Файл нового расписания')
        verbose_name_plural = _(u'Оборот поездов: Файлы нового расписания')
        app_label = 'importinfo'

    def get_encoded_data(self):
        return self.schedule_file.encode("cp1251")


class AFTrainTurnoverUpdateFile(models.Model):
    u"""Файлы обновления оборота поездов"""
    update_file_name = models.CharField(blank=True, max_length=255,
                                        null=False,
                                        verbose_name=_(u"Название файла"))
    update_file = TextFileField(blank=False, verbose_name=_(u"Файл с обновлениями оборотов"),
                                encoding='cp1251',
                                help_text=_(u"Принимаются только текстовые файлы в кодировке cp1251"))
    created_at = models.DateTimeField(auto_now_add=True, verbose_name=_(u"Добавлен"))
    loaded = models.BooleanField(verbose_name=_(u"Импортирован"), default=False, blank=True)
    load_log = models.TextField(null=True, blank=True)

    class Meta:
        ordering = ('-created_at',)
        verbose_name = _(u'Оборот поездов: Файл обновления')
        verbose_name_plural = _(u'Оборот поездов: Файлы обновления')
        app_label = 'importinfo'

    def get_encoded_data(self):
        return self.update_file.encode("cp1251")
