# -*- coding: utf-8 -*-

from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.utils.caching import cache_until_switch
from common.utils.fields import CodeCharField, RegExpField, DatabaseFileField
from travel.rasp.admin.importinfo.models.validators import triangle_file_validator

from travel.rasp.admin.www.utils.common import RegExpSet


# RASPADMIN-333: Удалить таблицу треугольного в 2016 году.
class TriangleBusImportPackage(models.Model):
    POSSIBLE_ENCODINGS = ('utf8', 'cp1251')
    FILE_FORMATS = ('csv', 'xls')
    REQUIRED_FILE_NAMES = ('carriers', 'stations', 'timetable')
    MAPPING_FILE_NAME = 'station_mapping.csv'
    TIMETABLE_FIELDS = ('number', 'route_title', 'carrier_code', 'bus_type', 'station_title',
                        'station_code', 'distance')
    STATION_FIELDS = ('code', 'title', 'country', 'region', 'region_code', 'settlement', 'settlement_code',
                      'latitude', 'longitude')
    CARRIER_FIELDS = ('code', 'title', 'address', 'phone')

    currency = models.ForeignKey('currency.Currency', verbose_name=_(u"Валюта"), null=True, blank=True)

    supplier_email = models.EmailField(_(u'E-mail поставщика'), blank=True, null=True)

    notify_supplier = models.BooleanField(_(u'Уведомлять поставщика'), null=False, default=False)

    ENCODING_CHOICES = (
        ('utf-8', 'utf-8'),
        ('cp1251', 'cp1251'),
    )

    encoding = models.CharField(_(u"Кодировка"), max_length=20, choices=ENCODING_CHOICES,
                                default=None, blank=True, null=True)

    tsi_package = models.OneToOneField('importinfo.TwoStageImportPackage',
                                       null=True, blank=True, default=None,
                                       verbose_name=_(u'Двухступенчатый пакет'))

    # TODO: остальные поля удалить в 3.5
    country = models.ForeignKey('www.Country', verbose_name=_(u'страна'),
                                null=True, blank=True, default=None)

    t_type = models.ForeignKey('www.TransportType', verbose_name=_(u"тип транспорта"),
                               null=False, default=3)
    title = models.CharField(
        _(u"Название пакета"), max_length=255, default=None, null=True, blank=True,
        help_text=_(u"не обязательно, если не заполнено, присваивается название файла"))

    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u"поставщик данных"),
                                 null=True, blank=True, default=None,
                                 related_name=u"trianglebusimportpackage_set")

    station_code_supplier = models.ForeignKey(
        'www.Supplier', verbose_name=_(u"поставщик кодов станций"),
        default=None, null=True, blank=True,
        help_text=_(u"соответсвия и коды этого постащика будут "
                    u"использоваться для разрешения станций"),
        related_name=u"trianglebusimportpackage_stationcodesupplier_set")

    package_file = DatabaseFileField(
        _(u"Файл пакета"),
        content_type='application/zip',
        validators=[triangle_file_validator],
        null=True,
        blank=True,
        default=None
    )

    test_run = models.BooleanField(
        blank=False, default=True, verbose_name=_(u"тестовый запуск"),
        help_text=_(u"по умолчанию делается тестовый запуск, "
                    u"чтобы загрузить расписания и стации, снимите флаг"))

    load_log = models.TextField(null=True, blank=True)

    test_load_log = models.TextField(null=True, blank=True)

    last_import_date = models.DateField(_(u'Дата последнего импорта'),
                                        null=True, blank=True)
    last_mask_date = models.DateField(_(u'Дата, после которой все календари ниток будут пустыми'),
                                      null=True, blank=True)
    content_manager = models.ForeignKey(
        'importinfo.ContentManager',
        verbose_name=_(u'Ответственный контент-менеджер'),
        null=True,
        blank=True,
        default=None
    )

    recreate_stations = models.BooleanField(
        _(u"Пересоздавать станции, даже если есть соответствия"),
        default=False, blank=True)

    import_on_save = models.BooleanField(u"Импортировать при сохранении",
                                         default=False, blank=True)

    def __unicode__(self):
        return _(u"%(package_file)s от поставщика %(supplier_title)s") % {
            'package_file': unicode(self.tsi_package.package_file),
            'supplier_title': self.tsi_package.supplier.title}

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"треугольный импорт: Пакет")
        verbose_name_plural = _(u"треуголный импорт: Пакеты")


class MTASettlementMapping(models.Model):
    area_code = CodeCharField(max_length=255, null=False, blank=False)
    stop_city = CodeCharField(max_length=255, null=False, blank=False)
    settlement = models.ForeignKey('www.Settlement', null=True, blank=False)
    region = models.ForeignKey('www.Region', null=True, blank=True)

    class Meta:
        app_label = 'importinfo'
        unique_together = (('area_code', 'stop_city'),)
        verbose_name = _(u"мТА: Соответствие города")
        verbose_name_plural = _(u"мТА: Соответствия городов")


class MTADistrictMapping(models.Model):
    area_code = CodeCharField(max_length=255, null=False, blank=False, unique=True)
    area_name = models.CharField(max_length=255, null=False, blank=False)
    region = models.ForeignKey('www.Region', null=True, blank=True)
    district = models.ForeignKey('www.District', null=True, blank=True)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"мТА: Соответствие района")
        verbose_name_plural = _(u"мТА: Соответствия районов")


class MTAInternalNumberBlackList(models.Model):
    number = RegExpField(
        verbose_name=_(u'номер рейса'), max_length=100, blank=False,
        help_text=_(u'номер рейса можно задавать регулярым выражением'))

    comment = models.CharField(_(u"Примечание"), blank=True, null=True, default=None, max_length=255)

    def __unicode__(self):
        return self.number

    def save(self, **kwargs):
            self.number = self.number.strip()
            super(MTAInternalNumberBlackList, self).save(**kwargs)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"мТА: Номер черного списка")
        verbose_name_plural = _(u"мТА: Черный список внутренних номеров")

    @classmethod
    @cache_until_switch
    def get_regexp_set(cls, **filter_kwargs):
        str_list = [bl.number
                    for bl in
                    cls.objects.filter(**filter_kwargs)]
        return RegExpSet(str_list)
