# -*- coding: utf-8 -*-
from __future__ import unicode_literals
import logging
import re

from django.db import models
from django.utils.translation import ugettext_lazy as _

from common.utils.fields import TrimmedCharField, RegExpField


log = logging.getLogger(__name__)


class StationMapping(models.Model):
    """
    Соответствие между названиями станций у поставщиков и нашими станциями
    """

    supplier = models.ForeignKey('www.Supplier', null=False,
                                 verbose_name=_("поставщик"))
    title = TrimmedCharField(verbose_name=_("название станции у поставщика"),
                             max_length=255, null=False, default='', blank=True,
                             help_text=_("автоматически чистится от лишних пробелов"))

    station = models.ForeignKey('www.Station', null=False, verbose_name=_("станция"))
    code = TrimmedCharField(verbose_name=_("код станции у поставщика"),
                            max_length=255, null=False, default='', blank=True,
                            help_text=_("автоматически чистится от всех пробелов"))

    number = RegExpField(verbose_name=_("номер рейса"), blank=True,
                         default=None, max_length=100, null=True,
                         help_text=_("регулярное выражение"))

    route_title = RegExpField(verbose_name=_("название рейса"), blank=True,
                              default=None, max_length=255, null=True,
                              help_text=_("регулярное выражение"))

    region = models.ForeignKey('www.Region', verbose_name=_("регион рейса"),
                               help_text=_("регион рейса, если известен на момент импорта расписания"),
                               blank=True, default=None, null=True)

    context = models.ForeignKey('MappingContext', default=None, null=True, blank=True)

    def __unicode__(self):
        return self.supplier.code + " '" + unicode(self.title) + "' " + unicode(self.station)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _("Соответствие: Станции")
        verbose_name_plural = _("Соответствия: Станциям")
        db_table = 'importinfo_busstationtitlemapping'

    def map_description(self):
        result = "%s:" % self.supplier.code
        fields = ['title', 'code', 'number', 'route_title']
        for fname in fields:
            value = getattr(self, fname, None)
            if value:
                result += " %s=%s" % (fname, value)
        return result

    @classmethod
    def _get_map_list_by_title_and_code_from_query_set(cls, query_set, supplier, title, code):
        map_list = []
        if title and code:
            map_list = list(query_set.filter(supplier=supplier, title=title, code=code))
            if map_list:
                return map_list

        if code:
            map_list = list(query_set.filter(supplier=supplier, code=code))
            if map_list:
                return map_list

        if title:
            map_list = list(query_set.filter(supplier=supplier, title=title))
            if map_list:
                return map_list

        return map_list

    @classmethod
    def get_map_list_by_title_and_code(cls, supplier, title, code, region=None):
        map_list = None

        if region is not None:
            query_set = cls.objects.filter(region=region)
            map_list = cls._get_map_list_by_title_and_code_from_query_set(query_set, supplier, title, code)

        if not map_list:
            query_set = cls.objects.all()
            map_list = cls._get_map_list_by_title_and_code_from_query_set(query_set, supplier, title, code)

        return map_list

    @classmethod
    def re_filter(cls, map_list, field_name, value):
        result_list = []
        for title_mapping in map_list:
            re_string = getattr(title_mapping, field_name)
            if re_string:
                re_expression = re.compile(re_string, re.U | re.I)
                if re_expression.match(value):
                    result_list.append(title_mapping)
        return result_list

    @classmethod
    def simple_get_station(cls, supplier, title, code):
        from common.models.geo import Station

        map_list = cls.simple_get_map_list(supplier, title, code)

        if not map_list:
            raise Station.DoesNotExist

        if len(map_list) > 2:
            raise Station.MultipleObjectsReturned

        else:
            return map_list[0].station

    @classmethod
    def simple_get_map_list(cls, supplier, title, code):
        map_list = []
        if title and code:
            return list(cls.objects.filter(supplier=supplier, title=title, code=code))

        if code:
            return list(cls.objects.filter(supplier=supplier, code=code))

        if title:
            return list(cls.objects.filter(supplier=supplier, title=title))

        return map_list

    @classmethod
    def get_station(cls, supplier, title=None, code=None, number=None,
                    route_title=None, region=None, show_info=True):
        from common.models.geo import Station
        common_params = "%s title='%s' code='%s'" % (supplier.code, title, code)
        if show_info:
            log.info("%s: Ищем станцию по параметрам: title=%s, code=%s, number=%s, route_title=%s, region=%s",
                     supplier.code, title, code, number, route_title, region and region.title)

        map_list = cls.get_map_list_by_title_and_code(supplier, title, code, region)

        if not map_list:
            raise Station.DoesNotExist

        if len(map_list) == 1:
            station = map_list[0].station
            log.debug("Нашли станцию %s %s", station.id, station.title)
            return station

        # У нас получилось несколько объектов по коду и названию, пробуем
        # отфильтровать по номеру, затем по названию рейса
        if number:
            result_list = cls.re_filter(map_list, "number", number)
            if len(result_list) > 1 and route_title:
                result_list = cls.re_filter(map_list, 'route_title', route_title)

            if len(result_list) > 1:
                log.error("Несколько объектов с параметрами %s number='%s' route_title='%s'",
                          common_params, number, route_title)
                raise Station.MultipleObjectsReturned

            if result_list:
                station = result_list[0].station
                log.debug("Нашли станцию %s %s", station.id, station.title)
                return station

        # Ничего не нашли, пробуем отфильтровать только по названию
        if route_title:
            result_list = cls.re_filter(map_list, 'route_title', route_title)
            if len(result_list) > 1:
                log.error("Несколько объектов с параметрами %s route_title='%s'",
                          common_params, route_title)
                raise Station.MultipleObjectsReturned

            if result_list:
                station = result_list[0].station
                log.debug("Нашли станцию %s %s", station.id, station.title)
                return station

        # Не получилось успешно отфильтровать
        # Пробуем взять станцию без указания номера и названия рейса
        with_out_special_filters = []
        for title_mapping in map_list:
            if not title_mapping.number and not title_mapping.route_title:
                station = title_mapping.station
                log.debug("Нашли станцию %s %s", station.id, station.title)
                with_out_special_filters.append(station)

        if len(with_out_special_filters) == 1:
            return with_out_special_filters[0]
        elif len(with_out_special_filters) > 1:
            log.error("Несколько не фильтрующихся объектов с параметрами %s"
                      " без указания объекта по умолчанию",
                      common_params)
        else:
            log.error("Несколько привязок с параметрами %s",
                      common_params)

        return [m.station for m in map_list]

    def update_tsi_stations(self):
        from travel.rasp.admin.importinfo.models.two_stage_import import TwoStageImportPackage, TwoStageImportStation

        title = self.title or ''

        packages = TwoStageImportPackage.get_packages_for_supplier(self.supplier)

        TwoStageImportStation.objects.filter(code=self.code, title=title, package__in=packages) \
                                     .update(station_mapping=self)

        TwoStageImportStation.objects.filter(station_mapping=self.id) \
                                     .exclude(code=self.code, title=title) \
                                     .update(station_mapping=None)

    @classmethod
    def copy_mappings(cls, supplier, old_group_code, new_group_code):
        copied_mappings_count = 0

        mappings = cls.objects.filter(supplier=supplier, code__startswith=old_group_code + '_')
        already_existing_codes = set(
            cls.objects.filter(supplier=supplier, code__startswith=new_group_code + '_')
                       .values_list('code', flat=True)
        )
        for mapping in mappings:
            new_code = new_group_code + mapping.code[len(old_group_code):]
            if new_code not in already_existing_codes:
                mapping.id = None
                mapping.code = new_code
                mapping.save()
                copied_mappings_count += 1

        return copied_mappings_count


class MappingContext(models.Model):
    supplier = models.ForeignKey('www.Supplier', null=False, verbose_name=_("поставщик"))
    route_number = TrimmedCharField(_('Номер маршрута'), max_length=255,
                                    blank=True, null=False, default="")

    route_title = TrimmedCharField(_('Название маршрута'), max_length=255,
                                   blank=True, null=False, default="")

    first_station_code = TrimmedCharField(_('Код начальной станции'), max_length=255,
                                          blank=True, null=False, default="")

    first_station_title = TrimmedCharField(_('Название начальной станции'), max_length=255,
                                           blank=True, null=False, default="")

    last_station_code = TrimmedCharField(_('Код конечной станции'), max_length=255,
                                         blank=True, null=False, default="")

    last_station_title = TrimmedCharField(_('Название конечной станции'), max_length=255,
                                          blank=True, null=False, default="")

    priority = models.IntegerField(_("Приоритет"), null=False, blank=True, default=0)

    CONTEXT_FIELDS = ('route_number', 'route_title', 'first_station_code', 'first_station_title',
                      'last_station_code', 'last_station_title')

    def __unicode__(self):
        return _("Контекст %(supplier)s %(priority)s: %(fields)s") % {
            'supplier': self.supplier.code if self.supplier else "",
            'priority': self.priority,
            'fields': ", ".join(["%s=%s" % (f, getattr(self, f))
                                 for f in MappingContext.CONTEXT_FIELDS
                                 if getattr(self, f)])
        }

    class Meta:
        app_label = 'importinfo'
        verbose_name = _("контекст соответсвия")
        verbose_name_plural = _("контексты соответсвия")

    def get_help(self):
        from travel.rasp.admin.scripts.schedule.bus.import_mta import MtaMappingContext
        return {
            'mta': MtaMappingContext.__doc__.strip()
        }


class CompanyMapping(models.Model):
    """
    Соответствие между названиями компаний у поставщиков и нашими компаниями
    """

    supplier = models.ForeignKey('www.Supplier', null=False, verbose_name=_("поставщик"))
    title = TrimmedCharField(verbose_name=_("название компании у поставщика"),
                             max_length=255, null=False, default='', blank=True,
                             help_text=_("автоматически чистится от лишних пробелов"))

    company = models.ForeignKey('www.Company', null=False, verbose_name=_("компания"))
    code = TrimmedCharField(verbose_name=_("код компании у поставщика"),
                            max_length=255, null=False, default='', blank=True,
                            help_text=_("автоматически чистится от лишних пробелов"))

    def __unicode__(self):
        return "%s: %s %s -> %s %s" % (
            self.supplier.code, self.title, self.code, self.company.id, self.company.title
        )

    class Meta:
        app_label = 'importinfo'
        verbose_name = _("Соответствие: Компании перевозчику")
        verbose_name_plural = _("Соответствия: Компаниям перевозчикам")


class TransportModelMapping(models.Model):
    """
    Соответствие между названиями моделей транспорта у поставщиков и нашими моделями транспорта
    """

    supplier = models.ForeignKey('www.Supplier', null=False, verbose_name=_("поставщик"))
    title = TrimmedCharField(verbose_name=_("название модели транспорта у поставщика"),
                             max_length=255, null=False, default='', blank=True,
                             help_text=_("автоматически чистится от лишних пробелов"))

    t_model = models.ForeignKey('www.TransportModel', null=False, verbose_name=_("модель транспорта"))
    code = TrimmedCharField(verbose_name=_("код модели транспорта у поставщика"),
                            max_length=255, null=False, default='', blank=True,
                            help_text=_("автоматически чистится от лишних пробелов"))

    def __unicode__(self):
        return "%s: %s %s -> %s %s" % (
            self.supplier.code, self.title, self.code, self.t_model.id, self.t_model.title
        )

    class Meta:
        app_label = 'importinfo'
        verbose_name = _("Соответствие: Модели транспорта")
        verbose_name_plural = _("Соответствия: Моделям транспорта")
