# -*- coding: utf-8 -*-

import logging
import re
from datetime import timedelta

from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask

from common.utils.caching import cache_until_switch


log = logging.getLogger(__name__)


class ImportReportExtender(object):

    max_precentage_diff = 5

    @classmethod
    def get_percentage_diff(cls, previous_report, current_report):
        if previous_report.route_count == 0:
            if current_report.route_count == 0:
                return 0.
            else:
                diff = 100.
        else:
            diff = (100. * (current_report.route_count - previous_report.route_count) /
                    previous_report.route_count)

        return diff

    @classmethod
    def get_last_report(cls, supplier):
        try:
            return cls.objects.filter(supplier=supplier).order_by('-imported')[0]
        except IndexError:
            pass

    @classmethod
    def build_diff_dict(cls, previous_report, current_report):
        if previous_report.supplier != current_report.supplier:
            raise Exception('Suppliers must be equal')

        percentage_diff = cls.get_percentage_diff(previous_report,
                                                  current_report)

        new_set = current_report.route_set - previous_report.route_set
        removed_set = previous_report.route_set - current_report.route_set

        new_route_uids = sorted(list(new_set))
        removed_route_uids = sorted(list(removed_set))

        return {
            'supplier': current_report.supplier,
            'previous_report': previous_report,
            'current_report': current_report,
            'new_route_uids': new_route_uids,
            'removed_route_uids': removed_route_uids,
            'percentage_diff': percentage_diff,
            'previous_route_count': previous_report.route_count,
            'current_route_count': current_report.route_count,
            'previous_imported_dt': previous_report.imported,
            'current_imported_dt': current_report.imported,
            'is_ciritical_diff': abs(percentage_diff) >= cls.max_precentage_diff,
        }

    @classmethod
    def create_report(cls, supplier, save=False):
        from common.models.schedule import RThread

        today = environment.today()

        etalon_mask = RunMask.range(today + timedelta(7), today + timedelta(98), today=today)

        threads = RThread.objects.filter(supplier=supplier)

        def has_good_mask(thread):
            return thread.get_mask(today=today) & etalon_mask

        threads = list(filter(has_good_mask, threads))

        report = cls(supplier=supplier)
        report.route_count = len(threads)
        report.route_set = set([t.uid for t in threads])

        if save:
            report.save()

        return report


class Route2TerminalExtender(object):
    @classmethod
    @cache_until_switch
    def _get_numberstation_dict(cls):
        numberstation = {}
        for r2t in cls.objects.all().select_related('terminal__station'):
            numberstation[(r2t.number, r2t.terminal.station)] = r2t.terminal
        return numberstation

    @classmethod
    def get_terminal(cls, number, station):
        return cls._get_numberstation_dict().get((number, station))


class TabloManualStatusExtender(object):
    def add_manual_statuses(self):
        posible_routes = self.find_posible_routes()

        if not posible_routes:
            return

        entry = self.make_entry(posible_routes)

        entry.cancelled = self.status == 'cancelled'
        entry.manual_status = self.status
        entry.update_tablo()

    def remove_manual_statuses(self):
        posible_routes = self.find_posible_routes()

        if not posible_routes:
            return

        entry = self.make_entry(posible_routes)

        # Сбрасывает все установленые статусы
        entry.cancelled = False
        entry.manual_status = None
        entry.real = None
        entry.flush_lmt = True

        entry.update_tablo()

    def find_posible_routes(self):
        from stationschedule.models import ZTablo2

        filter_params = {}
        filter_params[str('%s__range' % self.event)] = (self.planned - timedelta(hours=12),
                                                        self.planned + timedelta(hours=12))
        filter_params['station'] = self.station
        posible_tablos = ZTablo2.objects.filter(**filter_params)

        number_re = re.compile(self.number, re.U | re.I)

        routes = []

        for tablo in posible_tablos:
            if tablo.route_id:
                route = tablo.route
            else:
                self.log.error(u"Запись в табло %s не содержит необходимой информации о маршруте",
                               tablo.id)
                continue

            if number_re.match(route.number):
                routes.append(route)

        return routes

    def make_entry(self, routes):
        from travel.rasp.admin.scripts.tablo.utils import TabloEntry

        entry = TabloEntry()
        entry.log = self.log
        entry.accept_manual_status = True
        entry.add_charters = False
        entry.update_next_airport_status = True
        entry.script = 'TabloManualStatusExtender'

        entry.station = self.station
        entry.routes = routes
        entry.planned = self.planned
        entry.real = self.real
        entry.event = self.event

        return entry
