# -*- coding: utf-8 -*-

import logging
from collections import defaultdict
from importlib import import_module

from django.db import models
from django.db.models import Q
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from common.models.geo import Country, Region, Settlement, Station
from travel.rasp.library.python.common23.date import environment
from common.utils.date import get_time_zone_choices
from common.utils.fields import DatabaseFileField, TrimmedCharField, CodeCharField
from travel.rasp.admin.importinfo.models.trusted_station import TrustedStation
from travel.rasp.admin.importinfo.utils.thread import get_threads_last_mask_date, get_threads_last_mask_date_and_starts_quantity

from travel.rasp.admin.www.models.geo import RoutePathOwner


log = logging.getLogger(__name__)


FACTORIES = {
    'dyc': 'travel.rasp.admin.scripts.schedule.bus.dyc.DycCysixFactory',
    'olven': 'travel.rasp.admin.scripts.schedule.bus.olven.OlvenCysixFactory',
    'kvc_tula': 'travel.rasp.admin.scripts.schedule.bus.tkvc.TkvcCysixFactory',
    'kazan-bus': 'travel.rasp.admin.scripts.schedule.bus.kazan_bus.KazanCysixFactory',
    'e-traffic': 'travel.rasp.admin.scripts.schedule.bus.e_traffic.ETrafficCysixFactory',
    'kmvavto': 'travel.rasp.admin.scripts.schedule.bus.kmvavto.KmvaCysixFactory',
    'buscomua': 'travel.rasp.admin.scripts.schedule.bus.buscomua_cysix.BuscomuaFactory',
    'beltranscom': 'travel.rasp.admin.scripts.schedule.bus.beltranscom.BeltranscomCysixFactory',
    'ukrmintrans': 'travel.rasp.admin.scripts.schedule.bus.ukrmintrans.UkrmintransCysixFactory',
    'avperm': 'travel.rasp.admin.scripts.schedule.bus.avperm.AvPermCysixFactory',
    'udmbus': 'travel.rasp.admin.scripts.schedule.bus.udmbuster.UdmCysixFactory',
    'mrtrans': 'travel.rasp.admin.scripts.schedule.bus.mrtrans.MrtransCysixFactory',
    'avokzal-kz': 'travel.rasp.admin.scripts.schedule.bus.avokzal_kz.AvokzalKzCysixFactory',
    'allticketsfor-me': 'travel.rasp.admin.scripts.schedule.bus.allticketsfor_me.AllticketsforMeCysixFactory',
}


class TwoStageImportPackage(models.Model):
    title = models.CharField(_(u"Название пакета"), max_length=255)
    supplier = models.ForeignKey('www.Supplier', verbose_name=_(u"Поставщик"), null=False)
    package_file = DatabaseFileField(_(u"Файл пакета"), null=True, blank=True)

    autoimport = models.BooleanField(_(u"Автоимпорт"), default=False)
    every_day_autoimport = models.BooleanField(
        _(u"Ежедневный Автоимпорт"),
        default=False, editable=False,
        help_text=_(u'Ежедневный автоимпорт выполняется только тогда, когда включен обычный автоимпорт')
    )
    comment = models.TextField(_(u"Примечание"), blank=True, null=True)
    country = models.ForeignKey(Country, null=True, blank=True, verbose_name=_(u'Страна'))
    region = models.ForeignKey(Region, null=True, blank=True, verbose_name=_(u'Регион'))
    settlement = models.ForeignKey(Settlement, null=True, blank=True, verbose_name=_(u'Город'))
    create_stations = models.BooleanField(_(u"Создавать не найденные станции"), default=False,
                                          help_text=_(u"Для тестовых нужд"))
    t_type = models.ForeignKey("www.TransportType", verbose_name=_(u"Тип транспорта"),
                               default=3, blank=False, null=False)
    t_subtype = models.ForeignKey('www.TransportSubtype', null=True, default=None, blank=True,
                                  verbose_name=_(u'Подтип транспорта'))
    url = CodeCharField(_(u"Ссылка для скачивания"),
                        max_length=255, blank=True, default=None, null=True)
    username = CodeCharField(_(u"Имя пользователя"),
                             max_length=255, blank=True, default=None, null=True)
    password = CodeCharField(_(u"Пароль"),
                             max_length=255, blank=True, default=None, null=True)

    last_import_date = models.DateField(_(u'Дата импорта'), null=True, blank=True)
    last_import_datetime = models.DateTimeField(_(u'Дата-время импорта'), null=True)
    last_mask_date = models.DateField(_(u'Дата, после которой все календари ниток будут пустыми'),
                                      null=True, blank=True)
    content_manager = models.ForeignKey(
        'importinfo.ContentManager',
        verbose_name=_(u'Ответственный контент-менеджер'),
        null=True,
        blank=False,
        default=None
    )
    notify_content_manager = models.BooleanField(_(u'Уведомлять контент-менеджера'), null=False, default=False)

    currency = models.ForeignKey('currency.Currency', verbose_name=_(u'Валюта'), null=True, blank=True)

    supplier_email = models.EmailField(_(u'E-mail поставщика'), blank=True, null=True)

    notify_supplier = models.BooleanField(_(u'Уведомлять поставщика'), null=False, default=False)

    ENCODING_CHOICES = (
        ('utf-8', 'utf-8'),
        ('cp1251', 'cp1251'),
    )
    encoding = models.CharField(_(u'Кодировка'), max_length=20, choices=ENCODING_CHOICES,
                                default=None, blank=True, null=True)

    PACKAGE_TYPE_CHOICES = (
        ('cysix', _(u'общий xml')),
        ('triangle', _(u'треугольный')),
        ('special', _(u'специальный'))
    )
    package_type = models.CharField(_(u'Тип пакета'), max_length=30, choices=PACKAGE_TYPE_CHOICES,
                                    default='cysix', blank=False, null=False)

    # RASPADMIN-332: Почистить двухступенчатый от полей связанный со старым треугольным пакетом
    triangle_package = models.ForeignKey('importinfo.TriangleBusImportPackage', null=True,
                                         blank=True, verbose_name=_(u"Треугольный пакет"))

    allow_back_pseudo_routes = models.BooleanField(_(u'Разрешить построение обратных псевдорейсов'),
                                                   null=False, default=False)

    new_trusted = models.BooleanField(_(u"Новые опорные автовокзалы"), default=True)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"Двухступенчаный импорт: Пакет")
        verbose_name_plural = _(u"Двухступенчаный импорт: Пакеты")
        db_table = 'importinfo_twostagebusimportpackage'

    def __unicode__(self):
        return u'%s (%s)' % (self.title, self.supplier,)

    def is_cysix(self):
        return self.package_type == 'cysix'

    def is_triangle(self):
        return self.package_type == 'triangle'

    def get_city_region_coutry(self):
        city = None
        region = None
        country = None

        if self.settlement:
            city = self.settlement
            if city.region:
                region = city.region
        elif self.region:
            region = self.region
            cid = region.settlement_set.values_list('id', flat=True).order_by('majority')[:1]
            if cid:
                city = Settlement.objects.get(pk=cid[0])
        elif self.country:
            country = self.country
            cid = country.settlement_set.values_list('id', flat=True).order_by('majority')[:1]
            if cid:
                city = Settlement.objects.get(pk=cid[0])
        else:
            city = Settlement.objects.get(pk=213)
            region = city.region
        if region:
            country = region.country
        elif city and city.country:
            country = city.country

        return city, region, country

    def create_finder(self, log=None):
        from travel.rasp.admin.importinfo.two_stage_import import TwoStageImportStationFinder

        factory = self.get_two_stage_factory()

        if factory:
            return factory.get_station_finder()

        mapping_supplier = self.get_mapping_supplier()

        return TwoStageImportStationFinder(mapping_supplier)

    def get_mapping_supplier(self):
        return self.supplier

    @classmethod
    def get_packages_for_supplier(cls, supplier):
        return list(cls.objects.filter(supplier=supplier))

    def get_two_stage_factory(self):
        if self.is_cysix():
            from cysix.two_stage_import.factory import CysixTSIFactory
            return CysixTSIFactory(self)

        if self.is_triangle():
            from cysix.triangle.triangle2cysix import TriangleCysixFactory
            return TriangleCysixFactory(self)

        if self.supplier.code in FACTORIES:
            factory_module, factory_name = FACTORIES[self.supplier.code].rsplit('.', 1)

            module = import_module(factory_module)

            return getattr(module, factory_name)(self)

    def use_cysix_factory(self):
        from cysix.two_stage_import.factory import CysixTSIFactory

        factory = self.get_two_stage_factory()

        return isinstance(factory, CysixTSIFactory)

    def can_use_filter(self, filter_code):
        from cysix.models import PackageFilter

        try:
            filter_ = PackageFilter.objects.get(package=self, filter__code=filter_code)
            return filter_.use
        except PackageFilter.DoesNotExist:
            pass

        return False

    def get_filter_params(self, filter_code):
        from cysix.models import PackageFilter

        params = None
        if self.can_use_filter(filter_code):
            pf = PackageFilter.objects.get(package=self, filter__code=filter_code)
            params = pf.parameters.get_parameters_as_dict()

        return params

    def get_filter(self, filter_code):
        from cysix.models import PackageFilter
        try:
            return PackageFilter.objects.get(package=self, filter__code=filter_code)
        except PackageFilter.DoesNotExist:
            pass

    def add_default_filters(self):
        from cysix.models import PackageFilter, Filter

        for f in Filter.objects.all():
            params = PackageFilter()
            params.filter = f
            params.package = self
            params.parameters = f.default_parameters
            params.use = f.default_use
            params.save()

        if self.is_triangle():
            self.set_filter_parameter('text_schedule_and_extrapolation', 'extrapolation_limit_length',
                                      TSISetting.DEFAULT_TRIANGLE_MAX_FORWARD_DAYS)

            self.set_filter_parameters('correct_stop_time', {
                'only_add_day': True,
                't_max': 100500.
            })

            self.set_filter_parameters('correct_departure_and_arrival', {
                'only_add_day': True,
                'check': 'by_our_distance',
                'correction': 'by_our_distance',
            })

            self.tsisetting.max_forward_days = TSISetting.DEFAULT_TRIANGLE_MAX_FORWARD_DAYS
            self.tsisetting.save()

    def set_filter_use(self, filter_code, value):
        pf = self.get_filter(filter_code)

        if pf is not None:
            pf.use = value

            pf.save()

    def set_filter_parameter(self, filter_code, param_code, value):
        pf = self.get_filter(filter_code)

        if pf is not None:
            pf.parameters.update_parameter_value(param_code, value)

            pf.save()

    def set_filter_parameters(self, filter_code, param_dict):
        pf = self.get_filter(filter_code)

        if pf is not None:
            for param_code, value in param_dict.iteritems():
                pf.parameters.update_parameter_value(param_code, value)

            pf.save()

    def set_filter_params(self, filter_code, params):
        from cysix.models import PackageFilter

        if self.can_use_filter(filter_code):
            pf = PackageFilter.objects.get(package=self, filter__code=filter_code)
            pf.parameters.update(params)

            pf.save()

    def update_last_mask_date(self):
        from common.models.schedule import RThread

        thread_queryset = RThread.objects.filter(route__two_stage_package=self)

        self.last_mask_date = get_threads_last_mask_date(thread_queryset)

    @property
    def triangle_package(self):
        from travel.rasp.admin.importinfo.models.bus import TriangleBusImportPackage
        try:
            return self.trianglebusimportpackage
        except TriangleBusImportPackage.DoesNotExist:
            pass

    def generate_directions_slices(self):
        from common.models.schedule import RThread

        now = environment.now()
        today = now.today()

        package_slice = PackageDirectionsSlice.objects.create(package=self, creation_dt=now)

        thread_ids_by_title = defaultdict(list)

        for thread in RThread.objects.filter(route__two_stage_package=self).only('id', 'title'):
            thread_ids_by_title[thread.title].append(thread.id)

        for thread_title, thread_ids in thread_ids_by_title.iteritems():
            thread_queryset = RThread.objects.filter(id__in=thread_ids)

            last_mask_date, starts_quantity = \
                get_threads_last_mask_date_and_starts_quantity(thread_queryset, today=today)

            if last_mask_date is not None:
                DirectionSlice.objects.create(
                    package_slice=package_slice,
                    direction_title=thread_title,
                    last_mask_date=last_mask_date,
                    starts_quantity=starts_quantity,
                )

    @cached_property
    def trusted_stations(self):
        return set(TrustedStation.objects.filter(tsi_package=self).values_list('station_id', 'tsi_package_group_id'))

    def new_trusted_stations_query(self, group):
        query = Q(tsi_package=self, tsi_package_group=None)
        if group is not None:
            query |= Q(tsi_package=self, tsi_package_group=group)

        return query

    def new_trusted_stations(self, group=None):
        return set(TrustedStation.objects.filter(
            self.new_trusted_stations_query(group)
        ).values_list('station_id', flat=True))

    def new_not_trusted_stations(self, group=None):
        return set(TrustedStation.objects.filter(
            ~self.new_trusted_stations_query(group)
        ).values_list('station_id', flat=True))


class TwoStageImportStation(models.Model):
    title = TrimmedCharField(_(u"Название для соответсвия"), max_length=255, null=False, default=u"", blank=True)
    real_title = TrimmedCharField(_(u"Название станции при автоматическом создании реальной станции"),
                                  max_length=255, null=True, default=u"", blank=True)
    geocode_title = TrimmedCharField(_(u'Название станции для геокодирования'),
                                     max_length=255, null=True, default=u"", blank=True)

    additional_info = models.TextField(verbose_name=_(u"Дополнительная информация"), null=True, default=None,
                                       blank=True)

    code = TrimmedCharField(_(u"Код станции"), max_length=255, null=False, default=u"", blank=True)
    station_mapping = models.ForeignKey(
        'importinfo.StationMapping', null=True, blank=False, on_delete=models.SET_NULL,
        verbose_name=_(u'соответствие станций'),
        help_text=_(u"точное соответствие, заполняется автоматически при импорте в промежуточную базу")
    )

    longitude = models.FloatField(verbose_name=_(u"долгота"), default=None,
                                  blank=True, null=True)
    latitude = models.FloatField(verbose_name=_(u"широта"), default=None,
                                 blank=True, null=True)

    package = models.ForeignKey(TwoStageImportPackage, null=False)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"двухступенчаный импорт: Станция")
        verbose_name_plural = _(u"двухступенчаный импорт: Станции")
        unique_together = (('title', 'code', 'package'),)
        db_table = 'importinfo_twostagebusimportstation'

    def unmap(self):
        if self.station_mapping:
            self.station_mapping.delete()

    def get_fuzzy_flag(self, package, path_key):

        fuzzy_flag, created = TSIThreadStationFlag.objects.get_or_create(
            package=package,
            path_key=path_key,
            station_key=self.key,
        )

        return fuzzy_flag

    def get_any_thread_title(self):
        rts = TwoStageImportThreadStation.objects.filter(station=self)
        if rts.count():
            return rts[0].thread.title

    def __unicode__(self):
        return _(u"<Станция двухступенчатого импорта: title='%(title)s', code='%(code)s'>") % {
            'title': self.title, 'code': self.code}

    def get_key(self):
        from travel.rasp.admin.scripts.schedule.utils.supplier_station import SupplierStation

        return SupplierStation.KEY_DELIMITER.join(map(unicode, [self.title, self.code]))

    key = property(get_key)

    def update_or_create_mapping(self, station):
        from travel.rasp.admin.importinfo.models.mappings import StationMapping
        try:
            mapping = StationMapping.objects.get(code=self.code, title=self.title,
                                                 supplier=self.package.get_mapping_supplier())
        except StationMapping.DoesNotExist:
            mapping = StationMapping(
                code=self.code, title=self.title,
                supplier=self.package.get_mapping_supplier())

        try:
            mapping.station = station
            mapping.save()

            mapping.update_tsi_stations()
        except StationMapping.MultipleObjectsReturned:
            log.error(u"Несколько привязок к %s", self)


class TwoStageImportThread(models.Model, RoutePathOwner):
    title = models.CharField(_(u"Название"), max_length=255)

    package = models.ForeignKey(TwoStageImportPackage, null=False, related_name='threads')

    # Связь с RThread
    path_key = models.TextField(_(u"Ключ пути"), default=None, null=True, blank=True)

    def get_setting(self):
        try:
            return TSIThreadSetting.objects.get(package=self.package, path_key=self.path_key)
        except TSIThreadSetting.DoesNotExist:
            return TSIThreadSetting.objects.create(package=self.package, path_key=self.path_key)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"двухступенчаный импорт: Нитка")
        verbose_name_plural = _(u"двухступенчаный импорт: Нитки")
        db_table = 'importinfo_twostagebusimportthread'

    def __unicode__(self):
        return u'%s %s' % (self.title, self.package,)

    def get_stations(self):
        return Station.objects\
            .filter(stationmapping__twostageimportstation__threadstations__thread=self)\
            .order_by("stationmapping__twostageimportstation__threadstations__id")


class TwoStageImportThreadStation(models.Model):
    thread = models.ForeignKey(TwoStageImportThread, null=False, related_name='threadstations')
    station = models.ForeignKey(TwoStageImportStation, null=False, related_name='threadstations')

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"двухступенчаный импорт: Станция нитки")
        verbose_name_plural = _(u"двухступенчаный импорт: Станции ниток")
        ordering = ('id',)
        db_table = 'importinfo_twostagebusimportthreadstation'

    def __unicode__(self):
        return u'%s [%s]' % (self.thread, self.station,)


class TSIThreadStationFlag(models.Model):
    FLAG_CHOISES = ((None, _(u'Наследовать')), (True, _(u'Да')), (False, _(u'Нет')))

    package = models.ForeignKey(TwoStageImportPackage, null=False, related_name='thread_station_flags')

    path_key = models.TextField(_(u"Ключ пути"), null=False, blank=False, db_index=True)

    station_key = models.CharField(_(u"Ключ станции"), null=False, blank=False, max_length=500)

    is_fuzzy = models.NullBooleanField(_(u"нечеткое время отправления/прибытия"),
                                       default=None, choices=FLAG_CHOISES)
    is_searchable_from = models.NullBooleanField(_(u"искать от станции"),
                                                 default=None, choices=FLAG_CHOISES)
    is_searchable_to = models.NullBooleanField(_(u"искать до станции"),
                                               default=None, choices=FLAG_CHOISES)
    in_station_schedule = models.NullBooleanField(_(u"в расписании по станции"),
                                                  default=None, choices=FLAG_CHOISES)
    in_thread = models.NullBooleanField(_(u"показывать на странице нитки"),
                                        default=None, choices=FLAG_CHOISES)

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"двухступенчаный импорт: Признаки станции нитки")
        verbose_name_plural = _(u"двухступенчаный импорт: Признаки станций ниток")


class TSIThreadSetting(models.Model):
    TIMEZONE_OVERRIDE_CHOICES = [
        (u'none', _(u'Не переопределять')),
        (u'local', _(u'Местное время станции')),
        (u'start_station', _(u'Местное время начальной станции')),
        (u'end_station', _(u'Местное время конечной станции')),
    ]

    TIMEZONE_OVERRIDE_CHOICES = tuple(TIMEZONE_OVERRIDE_CHOICES + get_time_zone_choices())

    package = models.ForeignKey(TwoStageImportPackage, null=False, related_name='tsi_thread_settings')

    path_key = models.TextField(_(u"Ключ пути"), null=False, blank=False, db_index=True)

    allow_to_import = models.BooleanField(_(u'Разрешено импортировать'), default=True)

    apply_base_stations = models.BooleanField(_(u"Учитывать «базовость» автовокзалов"),
                                              default=True)

    timezone_override = models.CharField(max_length=100, default=u'none')

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u'Двухступенчаный импорт: Настройки нитки')
        verbose_name_plural = _(u'Двухступенчаный импорт: Настройки нитки')


class TSISetting(models.Model):
    DEFAULT_MAX_FORWARD_DAYS = 300
    DEFAULT_TRIANGLE_MAX_FORWARD_DAYS = 90

    package = models.OneToOneField(TwoStageImportPackage)
    set_number = models.BooleanField(
        _(u"Импортировать с номером"),
        default=False,
        help_text=_(u'При изменении данной опции нужно стереть маршруты пакета, '
                    u'переимпортировать в промежуточную и переимпортировать пакет.')
    )
    filter_by_group = models.BooleanField(_(u"Фильтровать по группам"), default=False)
    import_order_data = models.BooleanField(_(u"Импортировать данные для покупки билетов"), default=False)
    leave_days_in_past = models.IntegerField(_(u"Оставлять дней в прошлом"), default=0)
    update_threads = models.BooleanField(
        _(u"Обновлять нитки"), default=True,
        help_text=_(u'При обновлении, у старых ниток, снимаются дни хождения, затем накладываются'
                    u' новые данные. Если снять галку то нитки будут удаляться, перед загрузкой'
                    u' новых данных.')
    )
    mark_new_company_as_strange = models.BooleanField(_(u'Помечать новые компании как странные'),
                                                      default=True)
    show_in_alldays_pages = models.BooleanField(_(u'Показывать в поиске и расписании на все дни'),
                                                default=True)
    max_forward_days = models.IntegerField(
        _(u'Максимальное количество дней вперед, на которые заполняются или экстраполируются маски ниток'),
        default=DEFAULT_MAX_FORWARD_DAYS,
        help_text=_(u'считая с дня импорта')
    )
    import_tariffs = models.BooleanField(_(u'Импортировать цены'), default=True)

    def __unicode__(self):
        return u"Настройки для пакета %s" % self.package.title

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"двухступенчаный импорт: Настройки")
        verbose_name_plural = _(u"двухступенчаный импорт: Настройки импорта пакета(только если есть общий xml формат)")


class CysixGroupFilter(models.Model):
    package = models.ForeignKey(TwoStageImportPackage)
    title = models.CharField(_(u"Название группы"), max_length=255, blank=True, default=u"")
    code = CodeCharField(_(u"Код Группы"), max_length=255, blank=False)

    tsi_middle_available = models.BooleanField(_(u"Импотировать в промежуточную базу"),
                                               default=False)
    tsi_import_available = models.BooleanField(_(u"Импотировать через двухступенчатый"),
                                               default=False)

    import_order_data = models.BooleanField(_(u"Импортировать данные для покупки билета"), default=False)
    use_thread_in_station_code = models.BooleanField(_(u'Использовать информацию о нитки в генерации кода станции'),
                                                     default=False)

    use_package_group_filters = models.BooleanField(_(u'Отдельные настройки фильтров'), default=False)

    def add_default_filters(self):
        from cysix.models import PackageGroupFilter, PackageFilter, Filter

        if PackageGroupFilter.objects.filter(cysix_group_filter=self).exists():
            return

        for filter_ in Filter.objects.exclude(code__in={'thread_comment', 'text_schedule_and_extrapolation'}):
            package_group_filter = PackageGroupFilter()
            package_group_filter.cysix_group_filter = self
            package_group_filter.filter = filter_

            package_filter = PackageFilter.objects.filter(package=self.package, filter__code=filter_.code).first()
            if package_filter:
                package_group_filter.parameters = package_filter.parameters
                package_group_filter.use = package_filter.use
            else:
                package_group_filter.parameters = filter_.default_parameters
                package_group_filter.use = filter_.default_use

            package_group_filter.save()

    def __unicode__(self):
        return _(u"%(title)s - %(code)s") % {
            'title': self.title,
            'code': self.code,
        }

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u"Двухступенчаный импорт: Группа общего формата")
        verbose_name_plural = _(u"Двухступенчаный импорт: Фильтр по группам общего формата")
        unique_together = (('code', 'package'),)


class PackageDirectionsSlice(models.Model):
    package = models.ForeignKey(TwoStageImportPackage)
    creation_dt = models.DateTimeField(_(u'Дата и время создания среза'),
                                       blank=False, null=False)

    def __unicode__(self):
        return _(u"Срез направлений пакета {package_title} на {dt}").format(
            package_title=self.package.title,
            dt=self.creation_dt.strftime('%Y-%m-%d %H:%M:%S'),
        )

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u'Двухступенчаный импорт: Срез направлений пакета')
        verbose_name_plural = _(u'Двухступенчаный импорт: Срезы направлений пакета')


class DirectionSlice(models.Model):
    package_slice = models.ForeignKey(PackageDirectionsSlice)
    direction_title = models.CharField(verbose_name=_(u'название направления'),
                                       blank=False, null=False, max_length=255)
    last_mask_date = models.DateField(
        _(u'Дата, после которой все календари ниток направления будут пустыми'),
        null=False, blank=False
    )
    starts_quantity = models.IntegerField(
        _(u'Суммарное количество стартов ниток направления, начиная с даты формирования среза'),
        null=False, blank=False
    )

    def __unicode__(self):
        return _(u'Срез направления {direction_title} пакета {package_title} на {dt}').format(
            direction_title=self.direction_title,
            package_title=self.package_slice.package.title,
            dt=self.package_slice.creation_dt.strftime('%Y-%m-%d %H:%M:%S'),
        )

    class Meta:
        app_label = 'importinfo'
        verbose_name = _(u'Двухступенчаный импорт: Срез направления')
        verbose_name_plural = _(u'Двухступенчаный импорт: Срезы направлений')

        # uncomment if django 1.5 or higher
        # index_together = [('package_slice', 'direction_title')]
