# -*- coding: utf-8 -*-

import logging
import re
from datetime import datetime

from django.utils.translation import ugettext_lazy as _

from common.utils.caching import cache_method_result
from travel.rasp.admin.lib.exceptions import TranslatableException
from travel.rasp.admin.scripts.schedule.utils.afmasktext import OldAfMaskBuilder
from travel.rasp.admin.scripts.schedule.utils.mask_builders import MaskBuilder
from travel.rasp.admin.lib.translation import lazy_format


EXCEPT_TEXT_STARTSWITH = (u'кроме', u'except', u'-')


log = logging.getLogger(__name__)


class TriangleMaskTemplateParser(object):
    DATE_FORMAT = '%Y-%m-%d'

    class ParseError(TranslatableException):
        pass

    def raise_error(self, text):
        raise self.ParseError(_(u"Ошибка разбора '%s': %s"), self.original_template_text, text)

    def __init__(self, start_date, end_date, today, country=None):
        self.today = today
        self.start_date = start_date
        self.end_date = end_date

        self.mask_builder = MaskBuilder(self.start_date, self.end_date, self.today,
                                        log=log, country=country)
        self.af_mask_builder = OldAfMaskBuilder(self.start_date, self.end_date, self.today,
                                                log=log, country=country)

    @cache_method_result
    def parse_template(self, template_text):
        result_mask = self.mask_builder.empty_mask()

        template_text_groups = self.split_template_text_to_groups(template_text)

        for group in template_text_groups:
            result_mask |= group.build_mask(self)

        return result_mask

    @cache_method_result
    def split_template_text_to_groups(self, template_text):
        self.original_template_text = template_text

        template_text = template_text.strip().lower()

        if not template_text or template_text == u'не ходит':
            return [TemplateTextGroup(self, [], [], None, None)]

        template_text_groups = []

        positive_templates = []
        negative_templates = []

        start_date = None
        end_date = None

        for template in map(unicode.strip, template_text.split(u';')):
            if not template:
                continue

            parts = map(unicode.strip, template.split(u','))
            template, modifiers = parts[-1], parts[:-1]

            if modifiers:
                # Новые модификаторы, сохраняем накопленные маски
                if positive_templates or negative_templates:
                    template_text_groups.append(TemplateTextGroup(
                        self,
                        positive_templates,
                        negative_templates,
                        start_date,
                        end_date
                    ))

                positive_templates = []
                negative_templates = []

                start_date, end_date = self.parse_modifiers(modifiers)

            if template.startswith(EXCEPT_TEXT_STARTSWITH):
                negative_templates.append(template)

            else:
                positive_templates.append(template)

        if positive_templates or negative_templates:
            template_text_groups.append(TemplateTextGroup(
                self,
                positive_templates,
                negative_templates,
                start_date,
                end_date
            ))

        return template_text_groups

    def parse_modifiers(self, modifiers):
        start_date = end_date = None

        for modifier in modifiers:
            date_, modifier_type = self.parse_modifier(modifier)

            if modifier_type == 'start':
                start_date = date_

            if modifier_type == 'end':
                end_date = date_

        if end_date:
            self.has_modifier_end_date = True

        return start_date, end_date

    modifier_start_re = re.compile(ur'^(?:[cс]|from)\s+\d{4}-\d\d-\d\d$', re.U + re.I)
    modifier_end_re = re.compile(ur'^(?:до|to)\s+\d{4}-\d\d-\d\d$', re.U + re.I)

    def parse_modifier(self, modifier):
        modifier = modifier.strip()

        if not modifier:
            self.raise_error(_(u'Пустой модификатор'))

        if self.modifier_start_re.match(modifier):
            start_date = self.parse_date(modifier.split()[1])
            return start_date, 'start'

        elif self.modifier_end_re.match(modifier):
            end_date = self.parse_date(modifier.split()[1])

            return end_date, 'end'

        else:
            self.raise_error(
                lazy_format(_(u"Не смогли разобрать модификатор '%s'"), modifier)
            )

    def split_template_text_to_bits(self, bit_set_text):
        bits = []

        for mask_bit_text in bit_set_text.split(u'|'):
            bits.append(self.build_mask_bit(mask_bit_text))

        return bits

    def parse_date(self, text):
        try:
            return datetime.strptime(text, self.DATE_FORMAT).date()
        except (ValueError, TypeError):
            self.raise_error(
                lazy_format(_(u'Ошибка разбора даты "%r"'), text)
            )

    def get_ordered_mask_bits(self):
        return [
            EverydayMaskBit,
            EvenMaskBit,
            OddMaskBit,

            ThroughTheDayMaskBit,
            ThroughTheGapMaskBit,
            ThroughTheGapWithStartDateMaskBit,

            DaysOfWeekMaskBit,
            DateMaskBit,

            AFMaskBit
        ]

    every_number_day_en_re = re.compile(ur'^from\s+(?P<date>\d{4}-\d\d-\d\d)\s*/\s*(?P<gap>\d+)$')

    def build_mask_bit(self, mask_text):
        mask_text = mask_text.strip().lower().replace(u"ё", u"е")

        for mask_bit_cls in self.get_ordered_mask_bits():
            if mask_bit_cls.match(mask_text, self):
                mask_bit = mask_bit_cls(mask_text)

                return mask_bit

        self.raise_error(
            lazy_format(_(u'Не поддеживаемый формат, маски дней хождений "%s"'), mask_text)
        )


class TemplateTextGroup(object):
    def __init__(self, parser, positive_templates, negative_templates, start_date, end_date):
        self.parser = parser
        self.positive_templates = positive_templates
        self.negative_templates = negative_templates

        self.start_date = start_date
        self.end_date = end_date

    def get_positive_bits(self):
        bits = []
        for bit_set_text in self.positive_templates:
            for bit in self.parser.split_template_text_to_bits(bit_set_text):
                bits.append(bit)

        if not bits and self.negative_templates:
            bits.append(EverydayMaskBit(u'ежедневно'))

        return bits

    def get_negative_bits(self):
        bits = []
        for negative_bit_set_text in self.negative_templates:
            bit_set_text = negative_bit_set_text.replace(u'кроме', u'')\
                                                .replace(u'except', u'')\
                                                .lstrip(u'-').strip()

            for bit in self.parser.split_template_text_to_bits(bit_set_text):
                bits.append(bit)

        return bits

    def build_mask(self, parser):
        positive_bits = self.get_positive_bits()
        negative_bits = self.get_negative_bits()

        if positive_bits:
            positive_mask = reduce(lambda x, y: x | y, (
                b.build_mask(parser, self.start_date, self.end_date)
                for b in positive_bits
            ), parser.mask_builder.empty_mask())
        else:
            positive_mask = parser.mask_builder.empty_mask()

        if negative_bits:
            negative_mask = reduce(lambda x, y: x | y, (
                b.build_mask(parser, self.start_date, self.end_date)
                for b in negative_bits
            ), parser.mask_builder.empty_mask())
        else:
            negative_mask = parser.mask_builder.empty_mask()

        return positive_mask.difference(negative_mask)

    def to_cysix(self, parser):
        positive_bits = self.get_positive_bits()
        negative_bits = self.get_negative_bits()

        for bit in positive_bits + negative_bits:
            bit.validate(parser, self.start_date, self.end_date)

        if positive_bits:
            days = u';'.join(
                b.to_cysix(parser, self.start_date, self.end_date)
                for b in positive_bits
            )
        else:
            days = u''

        if negative_bits:
            exclude_days = u';'.join(
                b.to_cysix(parser, self.start_date, self.end_date)
                for b in negative_bits
            )
        else:
            exclude_days = u''

        return {
            'days': days,
            'exclude_days': exclude_days
        }


class MaskBit(object):
    cysixlike = True

    def __init__(self, text):
        self.text = text

    @classmethod
    def match(cls, text, parser=None):
        raise NotImplementedError()

    def build_mask(self, parser, start_date, end_date):
        raise NotImplementedError()

    def to_cysix(self, parser, start_date, end_date):
        if self.cysixlike:
            return self.text

        return self.to_cysix_dates(parser, start_date, end_date)

    def to_cysix_dates(self, parser, start_date, end_date):
        return u';'.join(
            d.strftime(parser.DATE_FORMAT)
            for d in self.build_mask(parser, start_date, end_date).dates()
        )

    def validate(self, parser, start_date, end_date):
        pass


class TextMaskBit(MaskBit):
    base_text = None

    @classmethod
    def match(cls, text, parser=None):
        return text == cls.base_text


class RegexMaskBit(MaskBit):
    regex = None

    @classmethod
    def match(cls, text, parser=None):
        return bool(cls.regex.match(text))


class EverydayMaskBit(TextMaskBit):
    base_text = u'ежедневно'

    def build_mask(self, parser, start_date, end_date):
        return parser.mask_builder.daily_mask(start_date, end_date)


class EvenMaskBit(TextMaskBit):
    base_text = u'четные'

    def build_mask(self, parser, start_date, end_date):
        return parser.mask_builder.even_mask(start_date, end_date)


class OddMaskBit(TextMaskBit):
    base_text = u'нечетные'

    def build_mask(self, parser, start_date, end_date):
        return parser.mask_builder.odd_mask(start_date, end_date)


class ThroughTheDayMaskBit(TextMaskBit):
    base_text = u'через день'

    def build_mask(self, parser, start_date, end_date):
        self.validate(parser, start_date, end_date)

        return parser.mask_builder.through_the_day_mask(start_date, end_date)

    def validate(self, parser, start_date, end_date):
        if not start_date:
            parser.raise_error(_(u'Через день всегда должен'
                                 u' сопровождаться модификатором "с <дата>"'))


class ThroughTheGapMaskBit(RegexMaskBit):
    regex = re.compile(ur'^через\s+(\d+).*$', re.U + re.I)

    def build_mask(self, parser, start_date, end_date):
        day_in_gap = int(self.regex.match(self.text).group(1))

        self.validate(parser, start_date, end_date)

        return parser.mask_builder.gap_days_mask(day_in_gap, start_date, end_date)

    def validate(self, parser, start_date, end_date):
        if not start_date:
            parser.raise_error(
                lazy_format(_(u'Через n дней всегда должен'
                              u' сопровождаться модификатором "с <дата>": "%s"'),
                            self.text)
            )


class DaysOfWeekMaskBit(RegexMaskBit):
    regex = re.compile(ur'^[1-7]+$', re.U + re.I)

    def build_mask(self, parser, start_date, end_date):
        return parser.mask_builder.mask_from_days_of_week(self.text, start_date, end_date)


class DateMaskBit(RegexMaskBit):
    cysixlike = False

    regex = re.compile(ur'^\d{4}-\d\d-\d\d$', re.U + re.I)

    def build_mask(self, parser, start_date, end_date):
        return parser.mask_builder.one_day_mask(parser.parse_date(self.text), start_date, end_date)


class ThroughTheGapWithStartDateMaskBit(RegexMaskBit):
    cysixlike = False
    regex = re.compile(ur'^from\s+(?P<date>\d{4}-\d\d-\d\d)\s*/\s*(?P<gap>\d+)$')

    def build_mask(self, parser, start_date, end_date):
        days_in_gap, schedule_start_date = self.parse_and_validate(parser)

        return parser.mask_builder.gap_days_mask(
            days_in_gap, start_date, end_date,
            schedule_start_date=schedule_start_date
        )

    def parse_and_validate(self, parser):
        match = self.regex.match(self.text)
        schedule_start_date = parser.parse_date(match.groupdict()['date'])

        days_in_gap = int(match.groupdict()['gap'])
        if days_in_gap < 1:
            parser.raise_error(_(u'Не правильно указан шаг в маске'))

        return days_in_gap, schedule_start_date

    def validate(self, parser, start_date, end_date):
        self.parse_and_validate(parser)


class AFMaskBit(MaskBit):
    cysixlike = False

    @classmethod
    def match(cls, text, parser=None):
        return parser.af_mask_builder.is_af_mask_text(text.upper())

    def build_mask(self, parser, start_date, end_date):
        return self.build_and_validate(parser, start_date, end_date)

    def build_and_validate(self, parser, start_date, end_date):
        mask = parser.af_mask_builder.parse_af_text(self.text.upper(), start_date, end_date)
        if mask is None:
            parser.raise_error(
                lazy_format(_(u'Отмена в маске от AF не поддерживается "%s"'), self.text)
            )

        return mask

    def validate(self, parser, start_date, end_date):
        self.build_and_validate(parser, start_date, end_date)
