# coding: utf8

from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from io import BytesIO

from xml.etree import ElementTree

from common.models.geo import Station, Country
from common.models.schedule import TrainTurnover, TrainSchedulePlan
from travel.rasp.library.python.common23.date import environment

from travel.rasp.admin.lib.logs import get_collector_context
from travel.rasp.admin.lib.mask_builder.afmask_builders import AfMaskBuilder
from travel.rasp.admin.lib.mask_builder.bounds import MaskBounds
from travel.rasp.admin.scripts.schedule.af_processors.parse_utils import get_date_param


log = logging.getLogger(__name__)


def parse_and_save_turnover(turnover_el):
    station_esr = turnover_el.get('station_esr')
    start_date = get_date_param(turnover_el, 'start_date')
    end_date = get_date_param(turnover_el, 'end_date')
    number_before = turnover_el.get('number_before')
    number_after = turnover_el.get('number_after')
    template_code = turnover_el.get('template_code')
    graph_code = turnover_el.get('graph_code')

    station = Station.get_by_code(system='esr', code=station_esr)
    graph = TrainSchedulePlan.objects.get(code=graph_code)

    builder = AfMaskBuilder()
    bounds = MaskBounds(start_date, end_date)
    country = Country.objects.get(id=Country.RUSSIA_ID)
    mask = builder.build(bounds, template_code, country=country)

    key = {
        'station': station,
        'template_code': template_code,
        'number_before': number_before,
        'graph': graph
    }

    extra_params = {
        'start_date': start_date,
        'end_date': end_date,
        'number_after': number_after,
        'year_days': str(mask)
    }

    def create_turnover():
        TrainTurnover.objects.create(**dict(key, **extra_params))
        log.info('Оборот с параметрами %r создан.', turnover_el.attrib)

    def change_turnover():
        for k, v in extra_params.items():
            setattr(turnover, k, v)
        turnover.save()
        log.info('Оборот с параметрами %r изменен.', turnover_el.attrib)

    mode = turnover_el.get('mode')
    try:
        turnover = TrainTurnover.objects.get(**key)
    except TrainTurnover.DoesNotExist:
        turnover = None

    if turnover:
        if not mode:
            log.error('Оборот с параметрами %r уже существует.', turnover_el.attrib)
            change_turnover()
        if mode == 'delete':
            turnover.delete()
            log.info('Оборот с параметрами %r удален.', turnover_el.attrib)
        elif mode == 'change':
            change_turnover()
    else:
        if not mode:
            create_turnover()
        elif mode == 'change':
            log.error('Оборот с параметрами %r не существует.', turnover_el.attrib)
            create_turnover()


def load_turnovers_from_file(data, filename):
    with get_collector_context() as collector:
        log.info('Импортируем %s, %s', filename, environment.now())
        tree = ElementTree.parse(BytesIO(data))

        for turnover_el in tree.findall('train_turnover'):
            try:
                parse_and_save_turnover(turnover_el)
            except Exception:
                log.exception('Ошибка при импорте оборота %r', turnover_el.attrib)

        return collector.get_collected()
