# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.utils.translation import ugettext as _, gettext_noop as N_

from common.models.geo import Station
from common.models.transport import TransportType
from common.utils.caching import cache_method_result, cache_method_result_with_exception
from travel.rasp.admin.importinfo.models.mappings import StationMapping
from travel.rasp.admin.lib.adminutils import get_rasp_admin_url
from travel.rasp.admin.lib.logs import get_collector, remove_collector
from travel.rasp.admin.scripts.schedule.utils.errors import RaspImportError

log = logging.getLogger(__name__)


class TwoStageImportStationFinder(object):
    show_log = False

    def __init__(self, supplier):
        self.collector = get_collector(log)
        self.supplier = supplier
        super(TwoStageImportStationFinder, self).__init__()

    def find_or_create_by_tsi_station(self, tsi_station, region_id=None, settlement_id=None):
        try:
            return self.find_by_tsi_station(tsi_station)
        except Station.DoesNotExist:
            station = Station(t_type=TransportType.objects.get(code='bus'))
            station.title = tsi_station.title or "Название не указано"
            station.majority_id = 2

            station.settlement_id = settlement_id
            station.region_id = region_id

            if settlement_id:
                station.time_zone = station.settlement.time_zone

            elif region_id:
                station.time_zone = station.region.time_zone

            station.save()

            log.info(
                N_('Создали станцию %s %s с кодом %s в регионе %s %s'),
                station.id, station.title, tsi_station.code,
                station.region, get_rasp_admin_url(station)
            )

            if not StationMapping.objects.filter(code=tsi_station.code, title=tsi_station.title,
                                                 supplier=self.supplier):
                StationMapping.objects.create(code=tsi_station.code, station=station,
                                              title=tsi_station.title, supplier=self.supplier)

            log.info(N_('Привязали станцию %s %s %s: %s'),
                     station.id, station.title, get_rasp_admin_url(station), tsi_station)
            return station

    @cache_method_result_with_exception
    def find_by_supplier_station(self, supplier_station):
        mappings = self.find_exact_mappings(supplier_station)

        if len(mappings) == 1:
            return mappings[0].station

        elif not mappings:
            log.error(N_('Не нашли привязку для %s'), supplier_station)

            raise Station.DoesNotExist
        else:
            log.error(N_('Несколько соответствий для %s'), supplier_station)
            raise Station.MultipleObjectsReturned

    @cache_method_result_with_exception
    def find_by_tsi_station(self, tsi_station):
        mappings = self.find_exact_mappings(tsi_station)

        if len(mappings) == 1:
            return mappings[0].station

        elif not mappings:
            log.error(N_('Не нашли привязку для %s'), tsi_station)

            raise Station.DoesNotExist

        else:
            log.error(N_('Несколько соответствий для %s'), tsi_station)

            raise Station.MultipleObjectsReturned

    @cache_method_result
    def find_first_by_supplier_station(self, supplier_station):
        mappings = self.find_all_mappings(supplier_station)

        if len(mappings):
            return mappings[0].station

        else:
            log.error(N_('Не нашли привязку для %s'), supplier_station)
            raise Station.DoesNotExist

    def find_exact_mappings(self, supplier_station):
        mappings = StationMapping.objects.filter(
            title=supplier_station.title,
            code=supplier_station.code,
            supplier=self.supplier
        )
        return list(mappings)

    def find_exact_mappings_by_code_only(self, supplier_station):
        mappings = StationMapping.objects.filter(
            code=supplier_station.code,
            supplier=self.supplier
        )
        return list(mappings)

    def find_all_mappings(self, supplier_station):
        mappings = self.find_exact_mappings(supplier_station)
        return mappings

    def get_start_station_or_error(self, supplier_station):
        try:
            return self.find_by_supplier_station(supplier_station)

        except Station.DoesNotExist:
            raise RaspImportError(_('Не нашли начальную станцию'))

        except Station.MultipleObjectsReturned:
            raise RaspImportError(_('Несколько сответствий для начаной станции'))

    def get_report(self):
        subject = "Ошики при поиске станций от %s - %s" % (self.supplier.title, self.supplier.code)

        message = self.collector.get_collected()
        remove_collector(log, self.collector)

        return subject, message

    def __del__(self):
        try:
            remove_collector(log, self.collector)
        except Exception:
            pass

    def add_report_to_reporter(self, reporter):
        subject, message = self.get_report()
        if message:
            reporter.add_report(*self.get_report())
