# -*- coding: utf-8 -*-

import logging
from collections import OrderedDict
from datetime import datetime
from email.utils import encode_rfc2231
from functools import update_wrapper

from django import forms
from django.contrib import admin
from django.db import transaction
from django.conf import settings
from django.contrib import messages
from django.http import HttpResponseRedirect, HttpResponse, HttpResponseForbidden
from django.shortcuts import get_object_or_404
from django.utils.translation import ugettext_lazy as _, gettext_noop as N_, get_language
from django.utils.http import urlencode
from django.utils.encoding import smart_unicode

from travel.rasp.admin.importinfo.models import RelatedLog
from travel.rasp.admin.importinfo.models.mappings import StationMapping
from travel.rasp.admin.importinfo.models.two_stage_import import (TwoStageImportPackage, FACTORIES, PackageDirectionsSlice, TSISetting,
                                                CysixGroupFilter)
from cysix.forms import CysixFilterAdminForm
from cysix.models import PackageFilter, PackageGroupFilter
from travel.rasp.admin.importinfo.models.validators import triangle_file_validator
from travel.rasp.admin.importinfo.related_log import update_logs
from travel.rasp.admin.lib.admin_options import RaspExportModelAdmin
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.logs import LangCollectors
from travel.rasp.admin.lib.maintenance.flags import flags
from travel.rasp.admin.lib.maintenance.scripts import job
from travel.rasp.admin.lib.tmpfiles import clean_temp
from travel.rasp.admin.scripts.utils.file_wrapper.uploaders import get_temporary_url, delete_tmp_schedule
from travel.rasp.admin.scripts.utils.import_file_storage import remove_schedule_temporary_today_dir, remove_cysix_xml
from travel.rasp.admin.timecorrection.admin_functions import drop_and_calc_temp_thread_for_package

log = logging.getLogger(__name__)


class TSISettingInlineAdmin(admin.StackedInline):
    model = TSISetting
    verbose_name = u''
    verbose_name_plural = _(u'Настройки общего формата')


class TabularInlineWithCheckboxesInHead(admin.TabularInline):
    template = 'admin/edit_inline/tabular_with_checkboxes_in_head.html'


class CysixGroupFilterInlineAdmin(TabularInlineWithCheckboxesInHead):
    model = CysixGroupFilter
    extra = 0
    verbose_name_plural = _(u'Фильтр по группам общего формата')
    show_change_link = True


class CysixFilterAdmin(admin.TabularInline):
    model = PackageFilter
    extra = 0
    verbose_name = _(u'Фильтр общего xml-формата данных')
    verbose_name_plural = _(u'Фильтры общего xml-формата данных')
    readonly_fields = ('filter',)
    can_delete = False
    max_num = 0
    fields = ('use', 'filter', 'parameters')
    form = CysixFilterAdminForm
    ordering = ('filter__order', 'filter__code')


class CysixPackageGroupFilterAdmin(admin.TabularInline):
    model = PackageGroupFilter
    extra = 0
    verbose_name = _(u'Фильтр общего xml-формата данных для групп')
    verbose_name_plural = _(u'Фильтры общего xml-формата данных для групп')
    readonly_fields = ('filter',)
    can_delete = False
    max_num = 0
    fields = ('use', 'filter', 'parameters')
    form = CysixFilterAdminForm
    ordering = ('filter__order', 'filter__code')


class CysixGroupFilterAdmin(RaspExportModelAdmin):
    model = CysixGroupFilter
    verbose_name_plural = _(u'Фильтры по группам общего формата')
    verbose_name = _(u'Фильтр по группам общего формата')
    inlines = [CysixPackageGroupFilterAdmin]
    readonly_fields = ('package',)
    list_display = ('title', 'package', 'code')
    list_filter = ('package',)

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def save_model(self, request, obj, form, change):
        obj.save()
        if obj.use_package_group_filters:
            obj.add_default_filters()


class TwoStageImportPackageForm(forms.ModelForm):
    class Meta:
        model = TwoStageImportPackage
        exclude = []

    def clean(self):
        {
            'triangle': self._clean_triangle_package_type,
            'cysix': self._clean_cysix_package_type,
            'special': self._clean_special_package_type,
        }.get(
            self.cleaned_data['package_type'],
            self._clean_unknown_package_type
        )()

        cleaned_data = super(TwoStageImportPackageForm, self).clean()
        t_type = cleaned_data.get('t_type')
        t_subtype = cleaned_data.get('t_subtype')
        if t_type and t_subtype and t_subtype.t_type != t_type:
            raise forms.ValidationError(_(u'Подтипу транспорта {} соответствует тип транспорта {}')
                                        .format(t_subtype, t_subtype.t_type))
        return cleaned_data

    def _clean_triangle_package_type(self):
        package_file = self.cleaned_data['package_file']

        if not package_file:
            raise forms.ValidationError(_(u'Для треугольного пакета необходимо приложить файл.'))

        triangle_file_validator(package_file)

    def _clean_cysix_package_type(self):
        package_file = self.cleaned_data['package_file']
        url = self.cleaned_data['url']

        if not package_file and not url:
            raise forms.ValidationError(
                _(u'Для импорта из общего xml необходимо '
                  u'указать ссылку для скачивания или приложить файл.')
            )

    def _clean_special_package_type(self):
        supplier = self.cleaned_data['supplier']

        if supplier.code not in FACTORIES:
            raise forms.ValidationError(_(u'Нет реализации специального импорта для данного поставщика.'))

    def _clean_unknown_package_type(self):
        raise forms.ValidationError(_(u'Неизвестный тип пакета'))


class TwoStageImportPackageAdmin(RaspExportModelAdmin):
    form = TwoStageImportPackageForm
    inlines = [TSISettingInlineAdmin, CysixGroupFilterInlineAdmin, CysixFilterAdmin]

    list_display = (
        'title',
        'supplier',
        'package_type',
        'content_manager',
        'last_import_date',
        'last_mask_date',
        'supplier_email',
    )
    search_fields = ('title', 'supplier__title', 'supplier__code', 'supplier_email')
    raw_id_fields = ('country', 'region', 'settlement')
    list_filter = ('autoimport', 'every_day_autoimport',
                   't_type', 'content_manager',
                   'package_type', 'notify_supplier', 'supplier')
    fieldsets = (
        (
            None,
            {
                'fields': [
                    'content_manager',
                    'package_type',
                    'title',
                    'supplier',
                    'package_file',
                    'autoimport',
                    'every_day_autoimport',
                    'comment',
                    'country',
                    'region',
                    'settlement',
                    't_type',
                    't_subtype',
                    'allow_back_pseudo_routes',
                    'last_import_date',
                    'last_mask_date',
                ]
            }
        ),
        (
            _(u'Новый алгоритм опорных автовокзалов'),
            {
                'fields': [
                    'new_trusted'
                ]
            }
        ),
        (
            _(u'Опции отправки уведомления поставщику'),
            {
                'fields': [
                    'supplier_email',
                    'notify_supplier',
                    'notify_content_manager',
                ]
            }
        ),
        (
            _(u'Ссылка для скачивания (только для общего xml-формата)'),
            {
                'fields': [
                    'url',
                    'username',
                    'password'
                ]
            }
        ),
        (
            _(u'Опции треугольного формата'),
            {
                'fields': [
                    'currency',
                    'encoding',
                ]
            }
        ),
        (
            _(u'Для тестовых нужд'),
            {'fields': ['create_stations']}
        ),
    )
    readonly_fields = ('last_import_date', 'last_mask_date', 'every_day_autoimport')

    def set_every_day_autoimport(self, request, queryset):
        if not request.user.is_superuser:
            return HttpResponseForbidden()

        queryset.update(every_day_autoimport=True)
        messages.success(request, _(u'Признак ежедневного импортирования проставлен для {}').format(
            u', '.join(u'{}-{}'.format(p.id, p.title) for p in queryset)
        ))

    set_every_day_autoimport.short_description = _(u'Импортировать ежедневно')

    def unset_every_day_autoimport(self, request, queryset):
        if not request.user.is_superuser:
            return HttpResponseForbidden()

        queryset.update(every_day_autoimport=False)
        messages.success(request, _(u'Признак ежедневного импортирования снят для {}').format(
            u', '.join(u'{}-{}'.format(p.id, p.title) for p in queryset)
        ))

    unset_every_day_autoimport.short_description = _(u'Снять признак ежедевного импортирования')

    actions = [set_every_day_autoimport, unset_every_day_autoimport]

    def change_view(self, request, object_id, **kwargs):
        tsi_package = get_object_or_404(TwoStageImportPackage, pk=object_id)
        tsi_package.package_file  # Не убирать, т.к. в джанге сбрасывается коннект, при первом обращении к resolve

        extra_context = kwargs.pop('extra_context', {})

        actions = extra_context['actions'] = Action.get_all_actions()

        current_action = actions.get(request.POST.get('__action'))

        if request.method == 'POST' and current_action:
            request.method = 'GET'  # admin stub

            try:
                run_tsi(current_action.code, tsi_package, request=request)
                return HttpResponseRedirect(request.path + ('?_show_log=%s' % current_action.code))

            except Action.Error as e:
                current_action.error = smart_unicode(e)

        for action in actions.values():
            related_log = RelatedLog.get_by_object(tsi_package, action.code)

            action.log = getattr(related_log, 'log_%s' % get_language())

        last_slices = PackageDirectionsSlice.objects.filter(package=tsi_package).order_by('-creation_dt')[:2]
        if len(last_slices) == 2:
            extra_context.update({
                'compare_slices_url': '/admin/importinfo/packagedirectionsslice/compare/?' + urlencode({
                    'first_id': last_slices[0].id, 'second_id': last_slices[1].id
                })
            })

        extra_context['data_url'] = '/{}/{}/{}'.format(get_temporary_url(), tsi_package.supplier.code, tsi_package.id)

        return super(TwoStageImportPackageAdmin, self).change_view(request, object_id,
                                                                   extra_context=extra_context, **kwargs)

    def save_model(self, request, obj, form, change):
        obj.save()
        if not change:
            obj.add_default_filters()

    def save_related(self, request, form, formsets, change):
        super(TwoStageImportPackageAdmin, self).save_related(request, form, formsets, change)
        obj = form.instance
        if not change:
            TSISetting.objects.get_or_create(package=obj)

    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)

            return update_wrapper(wrapper, view)

        urlpatterns = super(RaspExportModelAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(\d+)/save_log/$', wrap(self.save_log_view), name='%s_%s_save_log' % info),
            url(r'^(\d+)/show_log/$', wrap(self.show_log_view), name='%s_%s_show_log' % info),
            url(r'^(\d+)/show-triangle-log/$', wrap(self.show_triangle_log), name='%s_%s_show_triangle_log' % info),
            url(r'^(\d+)/show-triangle-log/(.*)/$', wrap(self.show_triangle_log),
                name='%s_%s_show_triangle_log' % info),
        ] + urlpatterns

        return urlpatterns

    def save_log_view(self, request, package_id):
        log_code = request.GET.get('log_code', None)

        return self.save_log(int(package_id), log_code)

    @classmethod
    def save_log(cls, package_id, action_code):
        package = TwoStageImportPackage.objects.get(id=package_id)
        related_log = RelatedLog.get_by_object(package, action_code)

        my_log = getattr(related_log, 'log_%s' % get_language())

        response = HttpResponse(my_log, content_type='text/plain; charset=utf-8')

        filename = u'{}_log.txt'.format(package.title.strip())

        response['Content-Disposition'] = 'attachment; filename*={}'.format(
            encode_rfc2231(filename.encode('utf8'), 'utf-8' 'en')
        )

        return response

    def show_log_view(self, request, package_id):
        log_code = request.GET.get('log_code', None)

        return self.show_log(int(package_id), log_code)

    @classmethod
    def show_log(cls, package_id, action_code):
        package = TwoStageImportPackage.objects.get(id=package_id)
        related_log = RelatedLog.get_by_object(package, action_code)

        my_log = getattr(related_log, 'log_%s' % get_language())

        response = HttpResponse(my_log, content_type='text/plain; charset=utf-8')

        return response

    def show_triangle_log(self, request, package_id, log_tag=None):
        package = TwoStageImportPackage.objects.get(id=package_id)
        if package.triangle_package:
            related_log = RelatedLog.get_by_object(package.triangle_package, log_tag)

            my_log = getattr(related_log, 'log_%s' % get_language())
        else:
            my_log = ''

        response = HttpResponse(my_log, content_type='text/plain; charset=utf-8')

        return response


class Action(object):
    class Error(SimpleUnicodeException):
        pass

    actions = OrderedDict()

    do = None

    def __init__(self, code, button_title, log_name):
        self.code = code
        self.button_title = button_title
        self.log_name = log_name
        self.error = None
        self.log = None

    def __eq__(self, other):
        return self.code == other.code

    @classmethod
    def get_action(cls, code):
        if code:
            return cls.actions[code]()

    @classmethod
    def get_all_actions(cls):
        return OrderedDict((code, create_action()) for code, create_action in cls.actions.items())

    @classmethod
    def register(cls, code, button_title, verbose_log_name):
        def decorator(func):
            def create_action():
                action = cls(code, button_title, verbose_log_name)

                action.do = func

                return action

            cls.actions[code] = create_action

            return func

        return decorator

    def log_is_short(self):
        if self.log is not None and self.log.count(u'\n') > 100:
            return False

        return True


def run_tsi(action_code, package, set_up_flag=True, request=None):
    action = Action.get_action(action_code)
    log_tag = action.code

    with LangCollectors('', log_level=logging.INFO,
                        languages=settings.ADMIN_LANGUAGES) as collectors:
        if set_up_flag:
            old_flag = flags['maintenance']

            if flags['maintenance']:
                raise Action.Error(_(u'Нельзя работать с пакетом пока поднят флаг'))

            if request:
                parts = [request.user.username, request.user.last_name, request.user.first_name]
                user_title = u' '.join(p for p in parts if p)

            else:
                user_title = u''

            note_items = (
                (_(u'Начало запуска'), datetime.now().strftime('%d.%m.%y %H:%M')),
                (_(u'Пакет'), u'<a href="/admin/importinfo/twostageimportpackage/{}/">{}</a>'.format(
                    package.id, package.title
                )),
                (_(u'Запустил'), user_title if request else None)
            )

            note = u'<ul>'

            for k, v in note_items:
                note += u'<li>%s: %s</li>' % (k, v)

            note += u'</ul>'

            flags.set('maintenance', job.TWO_STAGE_IMPORT.flag_value, note)

        try:
            importer = get_package_importer(package)
            if not importer:
                raise Action.Error(_(u'Не найдено класса импортирования пакета'))

            action.do(importer)

        finally:
            if set_up_flag:
                flags['maintenance'] = old_flag

            logged_by_lang = dict()

            for lang in settings.ADMIN_LANGUAGES:
                logged_by_lang[lang] = collectors.get_collected(lang)

            update_logs(package, log_tag, logged_by_lang)


def get_package_importer(package):
    factory = package.get_two_stage_factory()

    if factory:
        return factory.get_two_stage_importer()

    return None


@Action.register('check_triangle_data', _(u'Проверить данные в треугольном формате'),
                 _(u'Лог проверки данных в треугольном формате'))
@clean_temp
@transaction.atomic
def check_triangle_data(importer):
    try:
        if importer.package.is_triangle():
            importer.check_xml()

        else:
            log.error(N_(u'Пакет не являтся треугольным'))

    except Exception:
        log.exception(N_(u'Ошибка при проверке треугольных данных'))

    finally:
        remove_schedule_temporary_today_dir(importer.package)
        transaction.set_rollback(True)


@Action.register('import_into_middle_base', _(u'Переимпортировать в промежуточную базу'),
                 _(u'Лог импорта в промежуточную базу'))
@clean_temp
@transaction.atomic
def import_package_into_middle_base(importer):
    try:
        importer.reimport_package_into_middle_base()

    except Exception:
        log.exception(N_(u'Ошибка промежуточного импорта'))
        transaction.set_rollback(True)


@Action.register('fake_import_package', _(u'Тестовый переимпорт'), _(u'Лог тестового переимпорта'))
@clean_temp
@transaction.atomic
def fake_import_package(importer):
    try:
        if importer.has_unbinded_stations():
            log.error(N_(u'Не хватает привязок импорт пакета будет с ошибками'))

        importer.reimport_package()

    except Exception:
        log.exception(N_(u'Ошибка при импорте пакета'))

    finally:
        transaction.set_rollback(True)

    log.info(N_(u"Откатываем изменения"))


@Action.register('import_package', _(u'Переимпорт'), _(u'Лог переимпорта'))
@clean_temp
@transaction.atomic
def import_package(importer):
    try:
        if importer.has_unbinded_stations():
            log.error(N_(u'Не хватает привязок импорт пакета будет с ошибками'))

        importer.reimport_package()

    except Exception:
        log.exception(N_(u'Ошибка при импорте пакета'))
        transaction.set_rollback(True)


@Action.register('clean_data', _(u'Удалить маршруты пакета в СГА и базе'), _(u'Лог очистки'))
@clean_temp
@transaction.atomic
def clean_package(importer):
    log.info(N_(u'Начинаем удаление маршрутов пакета в СГА и базе'))
    try:
        importer.clean_middle_base()
        importer.remove_package_routes()
    except Exception:
        log.exception(u'Ошибка удаления маршрутов в СГА и в базе')
        raise
    else:
        log.info(N_(u'Успешно удалили маршруты пакета в СГА и базе'))


@Action.register('clean_legacy_mappings', _(u'Удалить устаревшие привязки'), _(u'Лог очистки'))
@transaction.atomic
@clean_temp
def clean_legacy_mappings(importer):
    log.info(N_(u'Начинаем удаление устаревших привязок'))
    importer.clean_legacy_mappings()
    log.info(N_(u'Успешно удалили устаревшие привязки'))


@Action.register('search_duplicates', _(u'Перестроить отчет о дубликатах'), _(u'Лог поиска дубликатов'))
@clean_temp
@transaction.atomic
def search_duplicates(importer):
    log.info(N_(u'Начинаем поиск дубликатов'))
    importer.search_duplicates()
    log.info(N_(u'Поиск дубликатов закончен'))


@Action.register('clean_today_data', _(u'Удалить скачанные данные за сегодня'), _(u'Лог удаления сегодняшних данных'))
def clean_today_data(importer):
    log.info(N_(u'Начинаем удаление скачанных данных за сегодня'))

    not_deleted = remove_schedule_temporary_today_dir(importer.package)
    delete_tmp_schedule(importer.package)  # delete mds dir

    if not_deleted:
        log.warning(N_(u'Не смогли удалить:\n%s'), u'\n'.join(not_deleted))

    else:
        log.info(N_(u'Очистка прошла успешно'))


@Action.register('remove_cysix_xml', _(u'Удалить файл общего xml формата'), _(u'Лог файла общего xml формата'))
def remove_today_cysix_xml(importer):
    log.info(N_(u'Начинаем удаление файла общего xml формата'))

    try:
        remove_cysix_xml(importer.package)
        delete_tmp_schedule(importer.package, 'cysix.xml')  # delete mds cysix

    except Exception:
        log.exception(N_(u'Не смогли удалить файл общего xml формата'))
    else:
        log.info(N_(u'Удаление файла общего xml формата прошло успешно'))


@Action.register('copy_mappings_for_arm', _(u'Скопировать привязки для АРМ'),
                 _(u'Лог копирования привязок для АРМ'))
@transaction.atomic
def copy_mappings_for_arm(importer):
    log.info(N_(u'Начинаем копирование привязок'))
    copied_mappings_count = StationMapping.copy_mappings(
        supplier=importer.package.supplier,
        old_group_code='all',
        new_group_code='arm',
    )
    log.info(N_(u'Успешно скопировали %s привязок'), copied_mappings_count)


@Action.register('check_package_thread', _(u'Обновить данные подробного анализа корректировки'),
                 _(u'Лог обновления данных корректировки'))
@transaction.atomic
def check_package_thread(importer):
    log.info(N_(u'Начинаем обновление'))
    drop_and_calc_temp_thread_for_package(importer.package.id)
    log.info(N_(u'Обновление окончено'))
