# -*- coding: utf-8 -*-

from itertools import combinations

from django.utils.encoding import smart_unicode

from common.models.schedule import RTStation
from common.models.tariffs import ThreadTariff
from common.models_utils import fetch_related
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cache_method_result


_not_specified = object()


class TariffVariant(object):
    __slots__ = ['_data', '_cache___hash__']
    # _cache___hash__ для cache_method_result

    def __init__(self, prices):
        """
        Не изменяемый объект

        prices - это
            iterable
            (
                (station_from_id, station_to_id, fare)
            )
        или
            dict
            {
                (station_from_id, station_to_id): fare
            }

        """
        self._data = {}

        if isinstance(prices, dict):
            self._data.update(prices)
        else:
            for sf, st, fare in prices:
                self._data[sf, st] = fare

    def get_fare(self, station_from_id, station_to_id):
        try:
            return self._data[station_from_id, station_to_id]
        except KeyError:
            return None

    def __eq__(self, other):
        return hash(self) == hash(other) and self._data == other._data

    @cache_method_result
    def __hash__(self):
        prices = [
            (sf, st, fare)
            for (sf, st), fare in self._data.items()
        ]

        prices.sort()

        return hash(tuple(prices))

    def __nonzero__(self):
        return bool(self._data)

    def __repr__(self):
        return u'<{}: {}>'.format(
            self.__class__.__name__,
            smart_unicode(repr(self._data))
        ).encode('utf-8')


def gen_tariffs_from_variants(thread, variant, mask):
    first_run = thread.first_run(today=environment.today())
    if not hasattr(thread, 'rtstations'):
        thread.rtstations = list(thread.path)
        fetch_related(thread.rtstations, 'station', model=RTStation)

    for rts_from, rts_to in combinations(thread.rtstations, 2):
        fare = variant.get_fare(rts_from.station_id, rts_to.station_id)
        if fare and fare.tariff > 0:
            yield ThreadTariff.create_tariff_from_fare(thread=thread, first_run=first_run,
                                                       rts_from=rts_from, rts_to=rts_to, tariff_mask=mask, fare=fare)
