# coding: utf-8

from django.utils.encoding import force_text


class TranslatableException(Exception):
    """
    Хранит сообщение и аргументы отдельно,
    чтобы иметь возможность записывать в логи сообщение на нужном языке.

    То есть, чтобы иметь возможность при перехвате исключения
    взять перевод для msg_template, и только потом подставить аргументы.

    Пример использования с логами:

    from django.utils.translation import ugettext_lazy as _

    try:
        # (1) Без аргументов
        # raise TranslatableException(_(u'Пример сообщения для лога'))

        # (2) Один аргумент
        # raise TranslatableException(_(u'Пример сообщения для лога %s'), my_arg)

        # (3) Два аргумента
        raise TranslatableException(_(u'Пример сообщения %s для лога %s'), my_arg1, my_arg2)

    except TranslatableException as e:
        log.error(e.msg_template, *e.msg_args)
    """

    def __init__(self, msg, *msg_args, **msg_kwargs):
        self.msg_template = msg
        self.msg_args = msg_args
        self.msg_kwargs = msg_kwargs

    @property
    def msg(self):
        if self.msg_args:
            return force_text(self.msg_template % self.msg_args)

        elif self.msg_kwargs:
            return force_text(self.msg_template % self.msg_kwargs)
        else:
            return force_text(self.msg_template)

    def __unicode__(self):
        return self.msg

    def __str__(self):
        return self.msg.encode('utf8')


class FormattingException(TranslatableException):
    pass


class SimpleUnicodeException(TranslatableException):
    pass


class UnpackError(FormattingException):
    pass


class MainError(FormattingException):
    """
    Общая ошибка для использования в main функциях скриптов.
    Пример:
        def main():
            raise MainError('Нет тела функции')

        if __name__ == '__main__':
            try:
                main()
            except MainError as e:
                log.error(six.text_type(e))
                sys.exit(1)

    """
    pass
